#include "dolphot_common.h"
#include <assert.h>

typedef float float3[3];

// uncomment to use single aperture correction
//#define SINGLE_APCOR

// incorporate uncertainty of sky estimate into photometry (FitSky=1,2)
#define USESKYSIG
// incorporate PSF residual error into neighbor photometry snd sky noise
#define MODELPSFERR
// use sky noise in sky estimation routine
#define USESKYNOISE

/*
#include <stdint.h>
#include <mach/mach_time.h>
#define TIC_MAXID 100
uint64_t ticStart[TIC_MAXID];
uint64_t ticElapsed[TIC_MAXID];
int ticNCalls[TIC_MAXID];
int ticInit=1;

void tic(int id) {
   if (ticInit) {
      memset(ticStart,0,sizeof(ticStart));
      memset(ticElapsed,0,sizeof(ticElapsed));
      memset(ticNCalls,0,sizeof(ticNCalls));
      ticInit = 0;
   }
   if (ticStart[id]!=0) {
      fprintf(stderr,"Error: tic() called for ID=%d which was already running\n",id);
      exit(-1);
   }
   ticStart[id] = mach_absolute_time();
   ticNCalls[id] ++;
}

void toc(int id) {
   if (ticStart[id]==0) {
      fprintf(stderr,"Error: toc() called for ID=%d which was not running\n",id);
      exit(-1);
   }
   ticElapsed[id] += mach_absolute_time() - ticStart[id];
   ticStart[id] = 0;
}

void endtictoc(void) {
   int id;
   for (id=0;id<TIC_MAXID;id++) {
      if (ticStart[id]!=0) {
	 fprintf(stderr,"Error: endtictoc called while ID=%d was still running\n",id);
	 exit(-1);
      }
      if (ticNCalls[id]>0) {
	 printf("TicToc(%d): %d calls, %f total sec, %f sec/call\n",id,ticNCalls[id],ticElapsed[id]/1.e9,(ticElapsed[id]/1.e9)/ticNCalls[id]);
      }
   }
}
*/

//NR Utilities

//NR Routines for FFT;

#define NR_END 1
#define FREE_ARG char*
#define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))
#define FMAX(a,b) ((a) > (b) ? (a) : (b))

void nrerror(char error_text[]) {
   fprintf(stderr,"Numerical Recipes run-time error...\n");
   fprintf(stderr,"%s\n",error_text);
   fprintf(stderr,"...now exiting to system...\n");
   exit(1);
}

float *vector(int nl, int nh) {
   float *v;

   v=(float *)malloc((size_t) ((nh-nl+1+NR_END)*FLOATSIZE));
   if (!v) nrerror("allocation failure in vector()");
   return v-nl+NR_END;
}

void free_vector(float *v, int nl, int nh) {
   free((FREE_ARG) (v+nl-NR_END));
   return;
}


float **matrix(int nrl, int nrh, int ncl, int nch) {
   int i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
   float **m;

   m=(float **) malloc((size_t)((nrow+NR_END)*PTRSIZE));
   if (!m) nrerror("allocation failure 1 in matrix()");
   m += NR_END;
   m -= nrl;

   m[nrl]=(float *) malloc((size_t)((nrow*ncol+NR_END)*FLOATSIZE));
   if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
   m[nrl] += NR_END;
   m[nrl] -= ncl;

   for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

   return m;
}

void free_matrix(float **m, int nrl, int nrh, int ncl, int nch) {
   free((FREE_ARG) (m[nrl]+ncl-NR_END));
   free((FREE_ARG) (m+nrl-NR_END));
   return;
}

// GLOBAL (PARALLEL-SAFE) VARIABLES
// per-pixel data; same pixel not accessed in parallel
int **indx,*tindx;
float **snmap,**snmmap;
char **ran,**snmapflag;

// written outside parallel code
int Nstars;
reopenableFile *fdata,*fsky,*fpsf,*fres;
FILE *of,*fpsfs=NULL,*ffakeout;
ftype refpsf;
chiptype refpsfimg=NULL;

// written inside #pragma omp critical
FILE *falign=NULL,*fapcor=NULL,*fwarn,*fverb;

// *DiagData allocated per image in allocDiagData1 (outside parallel code)
#ifdef PGPLOT
#include PGHEAD
typedef struct {
   float x,y,dx,dy;
} alignDiagDataType;
typedef struct {
   int N,XMIN,XMAX,YMIN,YMAX;
   float sig;
   alignDiagDataType *data;
} alignDiagType;
alignDiagType ***alignDiagData=0;
typedef struct {
   int N;
   float dmmean,mmin,mmax,dmmin,dmmax;
   float *dm,*m;
   int inst,filt;
   char imgstr[161];
   char filtstr[41];
} apcorDiagType;
apcorDiagType ***apcorDiagData=0;
typedef struct {
   int N;
   float mid;
   float*data;
} psfDiagType;
psfDiagType ***psfDiagData=0;

// Defined in allocDiagData1
static int DIAG_NEXT,*DIAG_NZ;
static int DIAG_EXT,DIAG_Z;

// Called outside parallel code
void allocDiagData1(int Next) {
   int i;
   if (GetParamChar(DiagPlotType)==0) return;
   DIAG_NEXT = Next;
   alignDiagData = (alignDiagType***)calloc(Nimg,PTRSIZE);
   apcorDiagData = (apcorDiagType***)calloc(Nimg,PTRSIZE);
   psfDiagData = (psfDiagType***)calloc(Nimg,PTRSIZE);
   DIAG_NZ = (int*)calloc(Next+1,INTSIZE);
   if (!alignDiagData || !apcorDiagData || !psfDiagData || !DIAG_NZ) merr();
   for (i=0;i<Nimg;i++) {
      alignDiagData[i] = (alignDiagType**)calloc(Next+1,PTRSIZE);
      apcorDiagData[i] = (apcorDiagType**)calloc(Next+1,PTRSIZE);
      psfDiagData[i] = (psfDiagType**)calloc(Next+1,PTRSIZE);
      if (!alignDiagData[i] || !apcorDiagData[i] || !psfDiagData[i]) merr();
   }
}

// Called outside parallel code
void allocDiagData2(int ext) {
   int i;
   if (GetParamChar(DiagPlotType)==0) return;
   DIAG_NZ[ext] = dataim[0].Z;
   for (i=0;i<Nimg;i++) {
      alignDiagData[i][ext] = (alignDiagType*)calloc(dataim[0].Z,sizeof(alignDiagType));
      apcorDiagData[i][ext] = (apcorDiagType*)calloc(dataim[0].Z,sizeof(apcorDiagType));
      psfDiagData[i][ext] = (psfDiagType*)calloc(dataim[0].Z,sizeof(psfDiagType));
      if (!alignDiagData[i][ext] || !apcorDiagData[i][ext] || !psfDiagData[i][ext]) merr();
   }
}

// Called outside parallel code
void setDiagFrame(int ext,int z) {
   DIAG_EXT = ext;
   DIAG_Z = z;
}

// Called outside parallel code
void plotDiagData(void) {
   int img,ext,z,i,j,nplot;
   int insts[MAXNIMG],filts[MAXNIMG],Nif=0,Nifs[MAXNIMG];
   float dmmean[MAXNIMG][MAXNIMG],midpix[MAXNIMG][MAXNIMG];
   char str[161],filtstrs[MAXNIMG][41];

   if (GetParamChar(DiagPlotType)==0) return;
   if (!strcasecmp(GetParamString(DiagPlotType),"gif")) sprintf(str,"%s.diag.gif/gif",GetParamString(outfn));
   else if (!strcasecmp(GetParamString(DiagPlotType),"ps")) sprintf(str,"%s.diag.ps/vcps",GetParamString(outfn));
   else if (!strcasecmp(GetParamString(DiagPlotType),"png")) sprintf(str,"%s.diag.png/png",GetParamString(outfn));
   else {
      printf("Error: unknown DiagPlotType %s\n",GetParamString(DiagPlotType));
      return;
   }
   cpgopen(str);
   cpgpap(0,1);
   cpgsubp(2,2);
   nplot=4;
   for (img=0;img<Nimg;img++) {
      for (ext=0;ext<=DIAG_NEXT;ext++) {
	 for (z=0;z<DIAG_NZ[ext];z++) {
	    int first=1;
	    if (apcorDiagData[img][ext][z].N>0) {
	       float x[2],y[2];
	       while (nplot<4) {cpgpage(); nplot++;}
	       first=0; nplot=1;
	       x[0] = apcorDiagData[img][ext][z].mmin;
	       x[1] = apcorDiagData[img][ext][z].mmax;
	       y[0] = apcorDiagData[img][ext][z].dmmean;
	       y[1] = apcorDiagData[img][ext][z].dmmean;
	       cpgenv(apcorDiagData[img][ext][z].mmin-0.1,apcorDiagData[img][ext][z].mmax+0.1,apcorDiagData[img][ext][z].dmmax+0.05,apcorDiagData[img][ext][z].dmmin-0.05,0,0);
	       cpgpt(apcorDiagData[img][ext][z].N,apcorDiagData[img][ext][z].m,apcorDiagData[img][ext][z].dm,-4);
	       cpgsci(2);
	       cpgline(2,x,y);
	       cpgsci(1);
	       cpglab("Instrumental mag","Aperture Correction",apcorDiagData[img][ext][z].imgstr);
	       if (psfDiagData[img][ext][z].N>0) {
		  for (i=0;i<Nif && (apcorDiagData[img][ext][z].inst!=insts[i] || apcorDiagData[img][ext][z].filt!=filts[i]);i++);
		  if (i==Nif) {
		     strcpy(filtstrs[Nif],apcorDiagData[img][ext][z].filtstr);
		     insts[Nif] = apcorDiagData[img][ext][z].inst;
		     filts[Nif++] = apcorDiagData[img][ext][z].filt;
		     Nifs[i]=0;
		  }
		  dmmean[i][Nifs[i]] = apcorDiagData[img][ext][z].dmmean;
		  midpix[i][Nifs[i]] = psfDiagData[img][ext][z].mid;
		  Nifs[i]++;
	       }
	    }
	    if (psfDiagData[img][ext][z].N>0) {
	       float tr[6]={0,0,0,0,0,0};
	       float min=0,max=0;
	       if (first) {
		  while (nplot<4) {cpgpage(); nplot++;}
		  first=0; nplot=1;
	       }
	       else nplot++;
	       for (i=0;i<psfDiagData[img][ext][z].N*psfDiagData[img][ext][z].N;i++) {
		  if (psfDiagData[img][ext][z].data[i]<min) min=psfDiagData[img][ext][z].data[i];
		  else if (psfDiagData[img][ext][z].data[i]>max) max=psfDiagData[img][ext][z].data[i];
	       }
	       tr[0]=tr[3]=-(psfDiagData[img][ext][z].N+1)/2;
	       tr[1]=tr[5]=1;
	       cpgenv(tr[0]+0.5,tr[0]+psfDiagData[img][ext][z].N+0.5,tr[0]+0.5,tr[0]+psfDiagData[img][ext][z].N+0.5,1,0);
	       cpggray(psfDiagData[img][ext][z].data,psfDiagData[img][ext][z].N,psfDiagData[img][ext][z].N,1,psfDiagData[img][ext][z].N,1,psfDiagData[img][ext][z].N,min,max,tr);
	       sprintf(str,"PSF Correction Image (max=%5.3f, min=%5.3f)",max,min);
	       cpglab("X","Y",str);
	    }
	    if (alignDiagData[img][ext][z].N>0) {
	       float *x,*y,sig;
	       int N;
	       if (first) {
		  while (nplot<4) {cpgpage(); nplot++;}
		  first=0; nplot=2;
	       }
	       else nplot+=2;
	       x = (float*)calloc(alignDiagData[img][ext][z].N,FLOATSIZE);
	       y = (float*)calloc(alignDiagData[img][ext][z].N,FLOATSIZE);
	       if (!x || !y) merr();
	       sig = alignDiagData[img][ext][z].sig*0.707;
	       if (sig<0.1) sig=0.1;
	       cpgenv(alignDiagData[img][ext][z].XMIN,alignDiagData[img][ext][z].XMAX,alignDiagData[img][ext][z].YMIN,alignDiagData[img][ext][z].YMAX,1,0);
	       N=0;
	       for (i=0;i<alignDiagData[img][ext][z].N;i++) {
		  if (fabs(alignDiagData[img][ext][z].data[i].dx)<=sig) {
		     x[N]=alignDiagData[img][ext][z].data[i].x;
		     y[N++]=alignDiagData[img][ext][z].data[i].y;
		  }
	       }
	       cpgpt(N,x,y,-4);
	       N=0;
	       for (i=0;i<alignDiagData[img][ext][z].N;i++) {
		  if (alignDiagData[img][ext][z].data[i].dx<-sig) {
		     x[N]=alignDiagData[img][ext][z].data[i].x;
		     y[N++]=alignDiagData[img][ext][z].data[i].y;
		  }
	       }
	       cpgsci(4);
	       cpgpt(N,x,y,-4);
	       N=0;
	       for (i=0;i<alignDiagData[img][ext][z].N;i++) {
		  if (alignDiagData[img][ext][z].data[i].dx>sig) {
		     x[N]=alignDiagData[img][ext][z].data[i].x;
		     y[N++]=alignDiagData[img][ext][z].data[i].y;
		  }
	       }
	       cpgsci(2);
	       cpgpt(N,x,y,-4);
	       cpgsci(1);
	       sprintf(str,"X Alignment Residuals (threshold=%5.3f)",sig);
	       cpglab("Xref","Yref",str);
	       cpgenv(alignDiagData[img][ext][z].XMIN,alignDiagData[img][ext][z].XMAX,alignDiagData[img][ext][z].YMIN,alignDiagData[img][ext][z].YMAX,1,0);
	       N=0;
	       for (i=0;i<alignDiagData[img][ext][z].N;i++) {
		  if (fabs(alignDiagData[img][ext][z].data[i].dy)<=sig) {
		     x[N]=alignDiagData[img][ext][z].data[i].x;
		     y[N++]=alignDiagData[img][ext][z].data[i].y;
		  }
	       }
	       cpgpt(N,x,y,-4);
	       N=0;
	       for (i=0;i<alignDiagData[img][ext][z].N;i++) {
		  if (alignDiagData[img][ext][z].data[i].dy<-sig) {
		     x[N]=alignDiagData[img][ext][z].data[i].x;
		     y[N++]=alignDiagData[img][ext][z].data[i].y;
		  }
	       }
	       cpgsci(4);
	       cpgpt(N,x,y,-4);
	       N=0;
	       for (i=0;i<alignDiagData[img][ext][z].N;i++) {
		  if (alignDiagData[img][ext][z].data[i].dy>sig) {
		     x[N]=alignDiagData[img][ext][z].data[i].x;
		     y[N++]=alignDiagData[img][ext][z].data[i].y;
		  }
	       }
	       cpgsci(2);
	       cpgpt(N,x,y,-4);
	       cpgsci(1);
	       sprintf(str,"Y Alignment Residuals (threshold=%5.3f)",sig);
	       cpglab("Xref","Yref",str);
	       free(x);
	       free(y);
	    }
	 }
      }
   }
   for (i=0;i<Nif;i++) {
      float xmin,xmax,ymin,ymax;
      xmin=xmax=midpix[i][0];
      ymin=ymax=dmmean[i][0];
      for (j=1;j<Nifs[i];j++) {
	 if (midpix[i][j]<xmin) xmin=midpix[i][j];
	 else if (midpix[i][j]>xmax) xmax=midpix[i][j];
	 if (dmmean[i][j]<ymin) ymin=dmmean[i][j];
	 else if (dmmean[i][j]>ymax) ymax=dmmean[i][j];
      }
      if (i==0) {
	 while (nplot<4) {cpgpage(); nplot++;}
      }
      cpgenv(xmin-0.001,xmax+0.001,ymin-0.01,ymax+0.01,0,0);
      cpgpt(Nifs[i],midpix[i],dmmean[i],-4);
      cpglab("Central PSF Adjustment","Aperture Correction",filtstrs[i]);
   }
   cpgend();
}
#endif

static inline float skyval(int img,int x,int y) {
   if (fsky[img].lastoffset<0) return 0.;
   if (x<0) x=0; else if (x>=dataim[img].X) x=dataim[img].X-1;
   if (y<0) y=0; else if (y>=dataim[img].Y) y=dataim[img].Y-1;
   return sky[img][y][x];
}

#ifdef USESKYNOISE
static inline float bgnoise(int img,int x,int y) {
   if (fsky[img].lastoffset<0) return 1.; // AEDDEBUG - should this error out?
   float n = (sky[img][y][x]>0)?sky[img][y][x]:0;
#ifdef MODELPSFERR
   n += GetParamDouble(NoiseMult)*SQR(data[img][y][x]-res[img][y][x]);
#endif
   return n*invGAIN[img]+iRN[img];
}
#endif

static inline float noise(int img,int x,int y,float ssky) {
   float n=data[img][y][x];
#ifdef MODELPSFERR
   const float nadd = (n>res[img][y][x])?GetParamDouble(NoiseMult)*SQR(n-res[img][y][x]):0;
#endif
   if (n<0) n=0.;
   if (n<ssky) n=ssky;
#ifndef MODELPSFERR
   return n*invGAIN[img]+iRN[img];
#else
   return (n+nadd)*invGAIN[img]+iRN[img];
#endif
}

static inline float modelnoise(int img,int x,int y,int dx,int dy,float s,float ssky) {
   float n=data[img][y][x]-res[img][y][x]; // model of other subtracted stars
   if (n<0.) n=0.;
#ifdef MODELPSFERR
   const float nadd = GetParamDouble(NoiseMult)*n*n;
#endif
   if (psf[img][dy][dx]>0. && s>0.) n+=s*psf[img][dy][dx]; // add own star model
   if (ssky>0.) n+=ssky; // add sky
#if 1
   //if (n<data[img][y][x]) // AEDDEBUG - uncomment to replicate old code
   n+=fabs(n-data[img][y][x]); // inflate sigma^2 based on fit quality
#endif
#ifndef MODELPSFERR
   return n*invGAIN[img]+iRN[img];
#else
   return (n+nadd)*invGAIN[img]+iRN[img];
#endif
}

static inline int peak2(int img,int x,int y) {
   if (x>0 && data[img][y][x-1]>data[img][y][x] && data[img][y][x-1]<iDMAX[img]) return 0;
   if (x<dataim[img].X-1 && data[img][y][x+1]>data[img][y][x] && data[img][y][x+1]<iDMAX[img]) return 0;
   if (y>0 && data[img][y-1][x]>data[img][y][x] && data[img][y-1][x]<iDMAX[img]) return 0;
   if (y<dataim[img].Y-1 && data[img][y+1][x]>data[img][y][x] && data[img][y+1][x]<iDMAX[img]) return 0;
   return 1;
}

static inline void setindx(int i) {
   int j;
   if (i==0) {
      for (j=GetGlobalInt(YMIN)*GetParamInt(SubResRef);j<GetGlobalInt(YMAX)*GetParamInt(SubResRef);j++) memset(indx[j]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,INTSIZE*(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
   }
   else {
      for (j=GetGlobalInt(XMIN)*GetParamInt(SubResRef);j<GetGlobalInt(XMAX)*GetParamInt(SubResRef);j++) tindx[j]=i;
      for (j=GetGlobalInt(YMIN)*GetParamInt(SubResRef);j<GetGlobalInt(YMAX)*GetParamInt(SubResRef);j++) {
	 memcpy(indx[j]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),tindx+GetGlobalInt(XMIN)*GetParamInt(SubResRef),INTSIZE*(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
      }
   }
   return;
}

// scaled by SubResRef:
// indx[][], tindx[], ran[][], snmap[][], snmmap[][], snmapflag[][]
static int rMark=0,rMarkPSF=0;
static inline void markstars(int x,int y) {
   int xx,yy;
   x*=GetParamInt(SubResRef);
   y*=GetParamInt(SubResRef);
   int dx = rMark*GetParamInt(SubResRef);
   for (xx=-dx;xx<=dx;xx++) if (x+xx>=GetGlobalInt(XMIN)*GetParamInt(SubResRef) && x+xx<GetGlobalInt(XMAX)*GetParamInt(SubResRef)) for (yy=-dx;yy<=dx;yy++) if (y+yy>=GetGlobalInt(YMIN)*GetParamInt(SubResRef) && y+yy<GetGlobalInt(YMAX)*GetParamInt(SubResRef)) if (indx[y+yy][x+xx]>=0) stars[indx[y+yy][x+xx]].flag=1;
   return;
}

// returns unc = max of standard error and standard deviation/sqrt(N)
float calcskyval(int n0,float*list,float*vlist,int img,float*unc) {
   int i,n,CONT=1,firstloop=1;
   double R=0,RR=0,wt=0,smlt=2.25,tsig;

   if (smlt<GetParamDouble(SkySig)) smlt=GetParamDouble(SkySig);
   n=n0;
   while (CONT && n) {
      wt=R=RR=0.;
      for (i=0;i<n;i++) {
	 wt+=1/vlist[i];
	 R+=list[i]/vlist[i];
	 RR+=list[i]*list[i]/vlist[i];
      }
      R/=wt;
      if (n>1) RR=smlt*sqrt(1.+(RR-R*R*wt)/(n-1)/wt);
      else RR=smlt;
      if (firstloop && unc) {
#ifdef USESKYNOISE
	 if (n>1) *unc = RR/smlt/sqrt(n);
	 else *unc = 1./sqrt(wt);
#else
	 *unc = 1./sqrt(wt);
	 if (n>1 && *unc<RR/smlt/sqrt(n)) *unc = RR/smlt/sqrt(n);
#endif
	 firstloop=0;
      }
      if (R<0) tsig=sqrt(1.+iRN[img])*3.5;
      else tsig=sqrt(1.+iRN[img]+R*invGAIN[img])*3.5;
      CONT=0;
      for (i=0;i<n;i++) if (fabs(list[i]-R)>RR) {
	 list[i--]=list[--n];
	 vlist[i+1]=list[n];
	 CONT=1;
      }
      if (!CONT && RR>tsig && smlt>GetParamDouble(SkySig)) {
	 smlt*=0.9;
	 if (smlt<GetParamDouble(SkySig)) smlt=GetParamDouble(SkySig);
	 CONT=1;
      }
   }
   if (!n) {
      if (unc) *unc=0;
      return 2.e30;
   }
   return R;
}

static inline void clear_sky(void) {
   int img;

   for (img=0;img<Timg;img++) sky_set[img]=0;
   return;
}

float getsky_norm(int img,int x,int y,float*unc,int FORCE_SKY) {
   float *list=NULL,*vlist=NULL;
   double s;
   float sig;
   int n=0,x1,y1,dx,dy,rsky,r2;

   if (GetGlobalInt(NO_SKY) && fsky[img].lastoffset>=0) {
      *unc=1;
      return skyval(img,x,y);
   }
   if (sky_set[img] && FORCE_SKY==0) {
      if (unc) *unc=sky_unc[img];
      return sky_val[img];
   }
   rsky=(int)(RSky1[img]+0.999);
   list=(float*)calloc((2*rsky+1)*(2*rsky+1),FLOATSIZE);
   vlist=(float*)calloc((2*rsky+1)*(2*rsky+1),FLOATSIZE);
   if (list==NULL || vlist==NULL) merr();
   for (x1=x-rsky;x1<=x+rsky;x1+=GetParamInt(SkipSky)) for (y1=y-rsky;y1<=y+rsky;y1+=GetParamInt(SkipSky)) if (ppixOK(img,x1,y1)) {
      dx=abs(x-x1);
      dy=abs(y-y1);
      if (dx>=RSky0[img] || dy>=RSky0[img]) {
	 r2=dx*dx+dy*dy;
	 if (r2>=RSky0[img]*RSky0[img] && r2<=RSky1[img]*RSky1[img]) {
#ifdef USESKYNOISE
	    vlist[n]=bgnoise(img,x1,y1);
#else
	    vlist[n]=1.;
#endif
	    list[n++]=res[img][y1][x1];
	 }
      }
   }
   s=calcskyval(n,list,vlist,img,&sig);
   free(list);
   free(vlist);
   if (s>1.e30) {
      if (unc) *unc=1.;
      return 0.;
   }
   sky_set[img]=1;
   sky_val[img]=s;
   sky_unc[img]=sig;
   if (unc) *unc=sig;
   return s;
}

static inline int iSQR(const int i) {return i*i;}

// returns unc = max of standard error and standard deviation/sqrt(N)
float getsky_small(int img,int x,int y,float*unc) {
   int i,N=0,cont;
   double av,sd,wt;
   float *list,*vlist;
   int firstloop=1;

   const float RSky20_2 = RSky20[img]*RSky20[img];
   const float RSky21_2 = RSky21[img]*RSky21[img];
   list=(float*)calloc(4*GetGlobalInt(rpsfMax)*GetGlobalInt(rpsfMax),FLOATSIZE);
   vlist=(float*)calloc(4*GetGlobalInt(rpsfMax)*GetGlobalInt(rpsfMax),FLOATSIZE);
   if (!list || !vlist) merr();
   if (unc) *unc=0.;
   for (int y1=y-RSky21[img];y1<=y+RSky21[img];y1++) {
      const int dy2 = iSQR(y1-y);
      for (int x1=x-RSky21[img];x1<=x+RSky21[img];x1++) if (ppixOK(img,x1,y1)) {
	 const int r2 = iSQR(x1-x)+dy2;
	 if (r2>RSky20_2 && r2<=RSky21_2) {
#ifdef USESKYNOISE
	    vlist[N]=bgnoise(img,x1,y1);
#else
	    vlist[N]=1.;
#endif
	    if (unc) (*unc)+=1./vlist[N];
	    list[N++]=res[img][y1][x1];
	 }
      }
   }
   if (unc && N) *unc=1/sqrt(*unc);
   cont=1;
   av=sd=0;
   if (N==1) {
      av=list[0];
      sd=sqrt(vlist[1]);
      if (unc) *unc=sd;
   }
   while (cont && N>1) {
      av=sd=wt=0;
      for (i=0;i<N;i++) {
	 wt+=1/vlist[i];
	 av+=list[i]/vlist[i];
	 sd+=list[i]*list[i]/vlist[i];
      }
      sd = sqrt((sd-av*av/wt)*N/(N-1)/wt) * 2.5;
      av /= wt;
      if (firstloop && unc) {
	 *unc = sd/2.5/sqrt(N);
#ifdef USESKYNOISE
	 if (*unc<1./sqrt(wt)) *unc=1./sqrt(wt);
#endif
	 firstloop=0;
      }
      cont=0;
      for (i=0;i<N;) {
	 if (fabs(list[i]-av)>sd) {
	    list[i]=list[--N];
	    vlist[i]=vlist[N];
	    cont=1;
	 }
	 else i++;
      }
   }
   free(list);
   free(vlist);
   return av;
}

static inline float getsky(int img,int x,int y,float*unc,int argFitSky,int FORCE_SKY) {
   float tmp;
   if (argFitSky==0 && fsky[img].lastoffset>=0) {
      if (unc) *unc=0.;
      tmp=skyval(img,x,y);
   }
   else if (argFitSky==2) tmp=getsky_small(img,x,y,unc);
   else tmp=getsky_norm(img,x,y,unc,FORCE_SKY);
   if (tmp<0 && !GetParamInt(NegSky)) return 0.0;
   else return tmp;
}

static inline void centroid(int IMG,int x0,int y0,double *X,double *Y) {
   int img,x,y,xx,yy,csat,nsat,nok;
   float min,xc1,yc1,xc=0,yc=0,wt,twt=0;
   for (img=0;img<Timg;img++) if (img==IMG || (IMG<0 && img<Nimg)) {
      double tx,ty;
      wt=xc1=yc1=0;
      shift(img,x0,y0,&tx,&ty,1);
      x=(int)(tx+0.5);
      y=(int)(ty+0.5);
      min=0;
      csat=nsat=nok=0;
      for (yy=-GetParamInt(RCentroid);yy<=GetParamInt(RCentroid);yy++) for (xx=-GetParamInt(RCentroid);xx<=GetParamInt(RCentroid);xx++) if (abs(xx)+abs(yy)<GetParamInt(RCentroid)*2 && ppixOK(img,x+xx,y+yy) && res[img][y+yy][x+xx]<min) min=res[img][y+yy][x+xx];
      for (yy=-GetParamInt(RCentroid);yy<=GetParamInt(RCentroid);yy++) for (xx=-GetParamInt(RCentroid);xx<=GetParamInt(RCentroid);xx++) if (abs(xx)+abs(yy)<GetParamInt(RCentroid)*2) {
	 if (ppixOK(img,x+xx,y+yy)) {
	    if (res[img][y+yy][x+xx]>min) {
	       xc1+=xx*(res[img][y+yy][x+xx]-min);
	       yc1+=yy*(res[img][y+yy][x+xx]-min);
	       wt+=(res[img][y+yy][x+xx]-min);
	    }
	    nok++;
	 }
	 else if (posOK(img,x+xx,y+yy) && data[img][y+yy][x+xx]>=iDMAX[img]) {
	    xc1+=xx*(iDMAX[img]-min);
	    yc1+=yy*(iDMAX[img]-min);
	    wt+=(iDMAX[img]-min);
	    if (xx==0 && yy==0) csat=1;
	    else csat=2;
	    nsat++;
	 }
	 else if (ppixOK(img,x-xx,y-yy)) {
	    xc1+=xx*(res[img][y-yy][x-xx]-min);
	    yc1+=yy*(res[img][y-yy][x-xx]-min);
	    wt+=(res[img][y-yy][x-xx]-min);
	 }
      }
      if (nsat>=nok) {xc1=yc1=wt=0.;}
      else if (csat==1) {xc1*=0.05; yc1*=0.05; wt*=0.05;}
      else if (csat) {xc1*=0.005; yc1*=0.005; wt*=0.005;}
      twt+=wt;
      xc+=xc1+wt*(float)(tx-x);
      yc+=yc1+wt*(float)(ty-y);
   }
   if (twt>0) {
      *X=xc/twt+x0+0.5;
      *Y=yc/twt+y0+0.5;
   }
   else {
      *X=x0+0.5;
      *Y=y0+0.5;
   }
   return;
}

int star_flag(double x0,double y0,int IMG) {
   int img,nsat,nbad,ntot,n1,x1,y1,ix,iy,cx,cy,best=15,flag;
   double x,y;

   for (img=0;img<Nimg;img++) if (IMG<0 || img==IMG) {
      flag=0;
      shift(img,x0,y0,&x,&y,1);
      ix=(int)(x+1)-1; iy=(int)(y+1)-1;
      if (ix<rphot[img] || ix>=dataim[img].X-rphot[img] || iy<rphot[img] || iy>=dataim[img].Y-rphot[img]) flag=1;
      if (posOK(img,ix,iy)) {
	 nbad=nsat=ntot=0;
	 for (y1=iy-1;y1<=iy+1;y1++) for (x1=ix-1;x1<=ix+1;x1++) if (posOK(img,x1,y1)) {
	    if (y1==iy && x1==ix) n1=4;
	    else if (y1==iy || x1==ix) n1=2;
	    else n1=1;
	    ntot+=n1;
	    if (data[img][y1][x1]>=iDMAX[img]) nsat+=n1;
	    else if (data[img][y1][x1]<=iDMIN[img]) nbad+=n1;
	 }
	 if (nsat+nbad>=8) flag|=8;
	 if (nsat>=4 && nbad>=4) flag|=6;
	 else if (nsat>=4) flag|=4;
	 else if (nbad+nsat>=4) flag|=2;
	 else {
	    nbad=ntot=0;
	    cx=(int)((x+10-(int)(x+10))*50+0.5);
	    if (cx<0) cx=0; if (cx>50) cx=50;
	    cy=(int)((y+10-(int)(y+10))*50+0.5);
	    if (cy<0) cy=0; if (cy>50) cy=50;
	    for (y1=iy-rphot[img];y1<=iy+rphot[img];y1++) for (x1=ix+circ[cy][cx][img][y1-iy][0];x1<=ix+circ[cy][cx][img][y1-iy][1];x1++) {
	       ntot++;
	       if (!ppixOK(img,x1,y1)) nbad++;
	    }
	    if (nbad*2>=ntot) flag|=8;
	    if (nbad*3>=ntot) flag|=2;
	 }
      }
      else flag|=9;
      if (flag<best) best=flag;
   }
   return best;
}

#define FOR_Y1_PHOT y1=iy-rphot[img]; if (y1<0) y1=0; for (dy=y1-iy;dy<=rphot[img] && y1<dataim[img].Y;y1++,dy++)
#define FOR_X1_PHOT x1=ix+circ0; if (x1<0) x1=0; for (dx=x1-ix;dx<=circ1 && x1<dataim[img].X;x1++,dx++) if (dataOK(img,x1,y1))

static inline void eval1_apphot(int img,int ix,int iy,int cx,int cy,float*s,float*ss,float*ssky,float*cs,int NOSUBTRACT,int argFitSky,int FORCE_SKY) {
   float c=0,d=0,dsky;
   int y1,x1,dx,dy;

   *ssky=getsky(img,ix,iy,&dsky,argFitSky,FORCE_SKY);
   *cs=*s=*ss=0;
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      float *restrict resRow = (!NOSUBTRACT)?res[img][y1]:data[img][y1];
      FOR_X1_PHOT {
	 d+=cwtRow[dx];
	 c+=psfRow[dx]*cwtRow[dx];
	 (*s)+=(resRow[x1]-(*ssky))*cwtRow[dx];
	 (*ss)+=noise(img,x1,y1,*ssky)*cwtRow[dx]*cwtRow[dx];
      }
   }
   if (*ss>0 && c>0 && d>0) {
      *cs=c/d;
      (*s)/=c;
#ifdef USESKYSIG
      (*ss)=sqrt(*ss+d*d*dsky*dsky)/c;
#else
      (*ss)=sqrt(*ss)/c;
#endif
   }
   else *ss=*s=*cs=0.;
   return;
}

static inline void eval1_psfphot(int img,int ix,int iy,int cx,int cy,float*s,float*ss,float*ssky,float*cs,int NOSUBTRACT,int FORCE_SKY) {
   float c,d,n,s0=0.,dsky,dd;
   int y1,x1,dx,dy,it;

   *ssky=getsky(img,ix,iy,&dsky,GetParamInt(FitSky),FORCE_SKY);
   for (it=0;it<GetParamInt(PSFPhotIt);it++) {
      if (it) s0=*s;
      *cs=*s=*ss=c=d=dd=0;
      //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];dx<=circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1)) {
      FOR_Y1_PHOT {
	 const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
	 float *restrict cwtRow = cwt[cy][cx][img][dy];
	 float *restrict psfRow = psf[img][dy];
	 float *restrict resRow = (!NOSUBTRACT)?res[img][y1]:data[img][y1];
	 FOR_X1_PHOT {
	    if (!it) n=noise(img,x1,y1,*ssky);
	    else n=modelnoise(img,x1,y1,dx,dy,s0,*ssky);
	    const float w=cwtRow[dx]*psfRow[dx]/n;
	    (*cs)+=psfRow[dx]*w;
	    d+=w;
	    if (psfRow[dx]>0) {
	       (*s)+=(resRow[x1]-(*ssky))*w;
	       c+=psfRow[dx]*w;
	       dd+=w;
	       (*ss)+=w*w*n;
	    }
	 }
      }
      if (*ss>0 && c!=0 && d!=0) {
	 (*cs)/=d;
	 (*s)/=c;
#ifdef USESKYSIG
	 (*ss)=sqrt(*ss+dd*dd*dsky*dsky)/c;
#else
	 (*ss)=sqrt(*ss)/c;
#endif
	 }
      else *ss=*s=*cs=0.;
   }
   return;
}

// FitSky = 3 routines
#define INCREMENT_PHOT \
   I+=a;								\
   P+=psfRow[dx]*a;							\
   D+=d*a;								\
   DP+=d*psfRow[dx]*a;							\
   PP+=psfRow[dx]*psfRow[dx]*a;					\
   a*=a*n;								\
   W+=a;								\
   WP+=psfRow[dx]*a;							\
   WPP+=psfRow[dx]*psfRow[dx]*a;

#define INCREMENT_PHOT_NOD \
   I+=a;								\
   P+=psfRow[dx]*a;							\
   PP+=psfRow[dx]*psfRow[dx]*a;					\
   a*=a*n;								\
   W+=a;								\
   WP+=psfRow[dx]*a;							\
   WPP+=psfRow[dx]*psfRow[dx]*a;

#define INCREMENT_PHOT_SKY \
   I+=a;								\
   P+=psfRow[dx]*a;							\
   D+=d*a;								\
   DP+=d*psfRow[dx]*a;							\
   PP+=psfRow[dx]*psfRow[dx]*a;

static inline float eval1_sky3(int img,int ix,int iy,int cx,int cy) {
   float I=0,P=0,D=0,DP=0,PP=0,a,skywt=0.,b0=0.,X,d;
   int y1,x1,dx,dy,npix=0;

   if (fsky[img].lastoffset>=0) {
      b0=skyval(img,ix,iy);
      if (b0<0.) a=iRN[img];
      else a=iRN[img]+b0*invGAIN[img];
      skywt=25./a;
      //skywt=M_PI*(SQR(RSky1)-SQR(RSky0))/SQR(GetParamInt(SkipSky))/a;
   }
   I = skywt;
   D = b0*skywt;
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++) if (ppixOK(img,x1,y1)) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      float *restrict resRow = res[img][y1];
      FOR_X1_PHOT {
	 dx=x1-ix;
	 a=cwtRow[dx]/noise(img,x1,y1,skyval(img,x1,y1));
	 d=resRow[x1];
	 npix++;
	 INCREMENT_PHOT_SKY
      }
   }
   X=I*PP-P*P;
   if (X!=0 && P>0 && npix>1) return (D*PP-P*DP)/X;
   return 0.;
}

static inline float eval1_snr_sky3(int img,int ix,int iy,int cx,int cy,float ct0) {
   float I=0,P=0,PP=0,W=0.,WP=0.,WPP=0.,a,skywt=0.,b0=0,n;
   int y1,x1,dx,dy,npix=0;

   //set to 1/sigma(sky)^2 for 9 pixels;
   if (fsky[img].lastoffset>=0) {
      b0=skyval(img,ix,iy);
      if (b0<0.) a=iRN[img];
      else a=iRN[img]+b0*invGAIN[img];
      skywt=25./a;
      //skywt=M_PI*(SQR(RSky1)-SQR(RSky0))/SQR(GetParamInt(SkipSky))/a;
   }
   I = skywt;
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1) && psfRow[dx]>0) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      FOR_X1_PHOT {
	 n=noise(img,x1,y1,skyval(img,x1,y1));
	 if (ct0>0.) n+=ct0*psfRow[dx]*invGAIN[img];
	 a=cwtRow[dx]/n;
	 npix++;
	 INCREMENT_PHOT_NOD
      }
   }
   a=I*PP-P*P;
   n=I*I*WPP-2*I*P*WP+P*P*W;
   if (a>0 && n>0 && npix>1) return ct0*a/sqrt(n);
   return 0.;
}

static inline void eval1_apphot_sky3(int img,int ix,int iy,int cx,int cy,float*s,float*ss,float*ssky,float*cs,int NOSUBTRACT) {
   float c=0.,d=0.;
   int y1,x1,dx,dy;

   *ssky=eval1_sky3(img,ix,iy,cx,cy);
   *cs=*s=*ss=0;
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++) if (ppixOK(img,x1,y1)) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      float *restrict resRow = (!NOSUBTRACT)?res[img][y1]:data[img][y1];
      FOR_X1_PHOT {
	 dx=x1-ix;
	 d+=cwtRow[dx];
	 c+=psfRow[dx]*cwtRow[dx];
	 (*s)+=(resRow[x1]-(*ssky))*cwtRow[dx];
	 (*ss)+=noise(img,x1,y1,*ssky)*cwtRow[dx]*cwtRow[dx];
      }
   }
   if (*ss>0 && c!=0 && d!=0) {
      *cs=c/d;
      (*s)/=c;
      (*ss)=sqrt(*ss)/c;
   }
   else *ss=*s=*cs=0.;
   return;
}

static inline void eval1_psfphot_sky3(int img,int ix,int iy,int cx,int cy,float*s,float*ss,float*ssky,float*cs,int NOSUBTRACT) {
   float I,P,D,DP,PP,W,WP,WPP,a,n,skywt=0.,b0=0.,d,s0=0.,ssky0=0.,p0=0.,Ppos;
   int y1,x1,dx,dy,it,npix;

   if (fsky[img].lastoffset>=0) {
      b0=skyval(img,ix,iy);
      if (b0<0.) a=iRN[img];
      else a=iRN[img]+b0*invGAIN[img];
      skywt=25./a;
      //skywt=M_PI*(SQR(RSky1)-SQR(RSky0))/SQR(GetParamInt(SkipSky))/a;
   }
   for (it=0;it<GetParamInt(PSFPhotIt);it++) {
      if (it) {s0=*s; ssky0=*ssky;}
      I=P=D=DP=PP=W=WP=WPP=0.;
      I = skywt;
      D = b0*skywt;
      npix=0;
      //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];dx<=circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1)) {
      FOR_Y1_PHOT {
	 const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
	 float *restrict cwtRow = cwt[cy][cx][img][dy];
	 float *restrict psfRow = psf[img][dy];
	 float *restrict resRow = res[img][y1];
	 FOR_X1_PHOT {
	    d=resRow[x1];
	    if (!it) n=noise(img,x1,y1,skyval(img,x1,y1));
	    else n=modelnoise(img,x1,y1,dx,dy,s0,ssky0);
	    a=cwtRow[dx]/n;	
	    npix++;
	    INCREMENT_PHOT
         }
      }
      a=I*PP-P*P;
      if (a>0 && P>0 && npix>1) {
	 *ssky=(D*PP-P*DP)/a;
	 if (!NOSUBTRACT) {
	    *s=(I*DP-D*P)/a;
	    *ss=sqrt(I*I*WPP-2*I*P*WP+P*P*W)/a;
	    *cs=PP/P;
	    //if (isnan(*ss)) printf("Caught NaN: npix=%d, %e %e\n",npix,I*I*WPP-2*I*P*WP+P*P*W,a);
	    if (isnan(*ss)) *s=*ss=*cs=*ssky=0;
	 }
	 else {
	    p0=P/I;
	    I=P=D=DP=PP=W=WP=WPP=Ppos=0.;
	    I = skywt;
	    D = b0*skywt;
	    FOR_Y1_PHOT {
	       const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
	       float *restrict cwtRow = cwt[cy][cx][img][dy];
	       float *restrict psfRow = psf[img][dy];
	       float *restrict resRow = res[img][y1];
	       float *restrict dataRow = data[img][y1];
	       FOR_X1_PHOT {
		  if (psfRow[dx]>=p0) d=dataRow[x1]-(*ssky);
		  else d=resRow[x1];
		  if (!it) n=noise(img,x1,y1,skyval(img,x1,y1));
		  else n=modelnoise(img,x1,y1,dx,dy,s0,ssky0);
		  a=cwtRow[dx]/n;
		  if (psfRow[dx]>=p0) Ppos+=psfRow[dx]*a;
		  INCREMENT_PHOT
	       }
	    }
	    a=I*PP-P*P;
	    if (a>0 && P>0) {
	       D -= (*ssky)*skywt*Ppos/P;
	       *s=(I*DP-D*P)/a;
	       *ss=sqrt(I*I*WPP-2*I*P*WP+P*P*W)/a;
	       *cs=PP/P;
	    }
	    else *s=*ss=*cs=0.;
	 }
      }
      else *ss=*s=*cs=*ssky=0.;
   }
#ifdef NAN_PRINT
   if (isnan(*s)) {printf("Uncaught s=nan\n"); fflush(stdout);}
   if (isnan(*ss)) {printf("Uncaught ss=nan\n"); fflush(stdout);}
   if (isnan(*cs)) {printf("Uncaught cs=nan\n"); fflush(stdout);}
   if (isnan(*ssky)) {printf("Uncaught ssky=nan\n"); fflush(stdout);}
#endif
   if (isnan(*s) || isnan(*ss) || isnan(*cs) || isnan(*ssky)) *ss=*s=*cs=*ssky=0.;
   return;
}
#undef INCREMENT_PHOT
#undef INCREMENT_PHOT_NOD
#undef INCREMENT_PHOT_SKY

#define INCREMENT_PHOT \
   I+=a;								\
   P+=psfRow[dx]*a;							\
   D+=d*a;								\
   DP+=d*psfRow[dx]*a;							\
   DX+=d*dx*a;								\
   DY+=d*dy*a;								\
   PP+=psfRow[dx]*psfRow[dx]*a;					\
   X+=dx*a;								\
   Y+=dy*a;								\
   XX+=dx*dx*a;								\
   XY+=dx*dy*a;								\
   YY+=dy*dy*a;								\
   XP+=dx*psfRow[dx]*a;						\
   YP+=dy*psfRow[dx]*a;						\
   a*=a*n;								\
   W+=a;								\
   WP+=psfRow[dx]*a;							\
   WPP+=psfRow[dx]*psfRow[dx]*a;					\
   WXP+=dx*psfRow[dx]*a;						\
   WYP+=dy*psfRow[dx]*a;						\
   WX+=dx*a;								\
   WY+=dy*a;								\
   WXX+=dx*dx*a;							\
   WXY+=dx*dy*a;							\
   WYY+=dy*dy*a;

#define INCREMENT_PHOT_SKY \
   I+=a;								\
   P+=psfRow[dx]*a;							\
   D+=d*a;								\
   DP+=d*psfRow[dx]*a;							\
   DX+=d*dx*a;								\
   DY+=d*dy*a;								\
   PP+=psfRow[dx]*psfRow[dx]*a;					\
   X+=dx*a;								\
   Y+=dy*a;								\
   XX+=dx*dx*a;								\
   XY+=dx*dy*a;								\
   YY+=dy*dy*a;								\
   XP+=dx*psfRow[dx]*a;						\
   YP+=dy*psfRow[dx]*a;						\

#define INCREMENT_PHOT_NOD \
   I+=a;								\
   P+=psfRow[dx]*a;							\
   PP+=psfRow[dx]*psfRow[dx]*a;					\
   X+=dx*a;								\
   Y+=dy*a;								\
   XX+=dx*dx*a;								\
   XY+=dx*dy*a;								\
   YY+=dy*dy*a;								\
   XP+=dx*psfRow[dx]*a;						\
   YP+=dy*psfRow[dx]*a;						\
   a*=a*n;								\
   W+=a;								\
   WP+=psfRow[dx]*a;							\
   WPP+=psfRow[dx]*psfRow[dx]*a;					\
   WXP+=dx*psfRow[dx]*a;						\
   WYP+=dy*psfRow[dx]*a;						\
   WX+=dx*a;								\
   WY+=dy*a;								\
   WXX+=dx*dx*a;							\
   WXY+=dx*dy*a;							\
   WYY+=dy*dy*a;

static inline float eval1_sky4(int img,int ix,int iy,int cx,int cy,float*mx,float*my) {
   float I=0,P=0,D=0,DP=0,PP=0,X=0,Y=0,XX=0,XY=0,YY=0,XP=0,YP=0,DX=0,DY=0,a,skywt=0,b0=0,d;
   int y1,x1,dx,dy,npix=0;

   if (fsky[img].lastoffset>=0) {
      b0=skyval(img,ix,iy);
      if (b0<0.) a=iRN[img];
      else a=iRN[img]+b0*invGAIN[img];
      skywt=25./a;
      //skywt=M_PI*(SQR(RSky1)-SQR(RSky0))/SQR(GetParamInt(SkipSky))/a;
   }
   D = b0*skywt;
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      float *restrict resRow = res[img][y1];
      FOR_X1_PHOT {
	 dx=x1-ix;
	 a=cwtRow[dx]/noise(img,x1,y1,skyval(img,x1,y1));
	 d=resRow[x1];
	 npix++;
	 INCREMENT_PHOT_SKY
      }
   }
   d = (XX*YY*P*P - P*P*XY*XY - 2*YY*P*X*XP + 2*P*X*XY*YP + 2*P*XP*XY*Y - 2*XX*P*Y*YP - X*X*YP*YP + PP*YY*X*X + 2*X*XP*Y*YP - 2*PP*X*XY*Y - XP*XP*Y*Y + I*YY*XP*XP - 2*I*XP*XY*YP + I*PP*XY*XY + PP*XX*Y*Y + I*XX*YP*YP - I*PP*XX*YY);
   if (d!=0 && P>0 && npix>3) {
      if (mx) *mx = (DX*(YY*P*P - 2*P*Y*YP + PP*Y*Y + I*YP*YP - I*PP*YY)-DY*(P*P*XY - I*PP*XY + I*XP*YP - P*X*YP - P*XP*Y + PP*X*Y)-DP*(XP*Y*Y - I*XP*YY + I*XY*YP + P*X*YY - P*XY*Y - X*Y*YP)-D*(X*YP*YP + P*XP*YY - P*XY*YP - PP*X*YY + PP*XY*Y - XP*Y*YP))/d;
      if (my) *my = (DY*(XX*P*P - 2*P*X*XP + PP*X*X + I*XP*XP - I*PP*XX)-DX*(P*P*XY - I*PP*XY + I*XP*YP - P*X*YP - P*XP*Y + PP*X*Y)-DP*(X*X*YP + I*XP*XY - I*XX*YP - P*X*XY + P*XX*Y - X*XP*Y)-D*(XP*XP*Y - P*XP*XY + PP*X*XY + P*XX*YP - PP*XX*Y - X*XP*YP))/d;
      return (D*(YY*XP*XP - 2*XP*XY*YP + PP*XY*XY + XX*YP*YP - PP*XX*YY)-DY*(XP*XP*Y - P*XP*XY + PP*X*XY + P*XX*YP - PP*XX*Y - X*XP*YP)-DX*(X*YP*YP + P*XP*YY - P*XY*YP - PP*X*YY + PP*XY*Y - XP*Y*YP)-DP*(P*XY*XY - P*XX*YY + X*XP*YY - X*XY*YP - XP*XY*Y + XX*Y*YP))/d;
   }
   *mx=*my=0;
   return 0.;
}

static inline float eval1_snr_sky4(int img,int ix,int iy,int cx,int cy,float ct0) {
   float I=0,P=0,PP=0,X=0,Y=0,XX=0,XY=0,YY=0,XP=0,YP=0,W=0,WP=0,WPP=0,WXP=0,WYP=0,WX=0,WY=0,WXX=0,WXY=0,WYY=0;
   float a,skywt=0.,b0=0,n,n1,n2,n3,n4,d;
   int y1,x1,dx,dy,npix=0;

   //set to 1/sigma(sky)^2 for 9 pixels;
   if (fsky[img].lastoffset>=0) {
      b0=skyval(img,ix,iy);
      if (b0<0.) a=iRN[img];
      else a=iRN[img]+b0*invGAIN[img];
      skywt=25./a;
      //skywt=M_PI*(SQR(RSky1)-SQR(RSky0))/SQR(GetParamInt(SkipSky))/a;
   }
   I = skywt;
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1) && psfRow[dx]>0) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      FOR_X1_PHOT {
	 n=noise(img,x1,y1,skyval(img,x1,y1));
	 if (ct0>0.) n+=ct0*psfRow[dx]*invGAIN[img];
	 a=cwtRow[dx]/n;
	 npix++;
	 INCREMENT_PHOT_NOD
      }
   }
   d = (XX*YY*P*P - P*P*XY*XY - 2*YY*P*X*XP + 2*P*X*XY*YP + 2*P*XP*XY*Y - 2*XX*P*Y*YP - X*X*YP*YP + PP*YY*X*X + 2*X*XP*Y*YP - 2*PP*X*XY*Y - XP*XP*Y*Y + I*YY*XP*XP - 2*I*XP*XY*YP + I*PP*XY*XY + PP*XX*Y*Y + I*XX*YP*YP - I*PP*XX*YY);
   n1 = YY*X*X - 2*X*XY*Y + I*XY*XY + XX*Y*Y - I*XX*YY;
   n2 = X*X*YP + I*XP*XY - I*XX*YP - P*X*XY + P*XX*Y - X*XP*Y;
   n3 = XP*Y*Y - I*XP*YY + I*XY*YP + P*X*YY - P*XY*Y - X*Y*YP;
   n4 = P*XY*XY - P*XX*YY + X*XP*YY - X*XY*YP - XP*XY*Y + XX*Y*YP;
   n = WPP*n1*n1-WYP*2*n1*n2-WXP*2*n1*n3-WP*2*n1*n4+WYY*n2*n2+WXY*2*n2*n3+WY*2*n2*n4+WXX*n3*n3+WX*2*n3*n4+W*n4*n4;
   if (d!=0 && n>0 && npix>3) return ct0*fabs(d)/sqrt(n);
   return 0.;
}

static inline void eval1_apphot_sky4(int img,int ix,int iy,int cx,int cy,float*s,float*ss,float*ssky,float*mx,float*my,float*cs,int NOSUBTRACT) {
   float c=0.,d=0.;
   int y1,x1,dx,dy;

   *ssky=eval1_sky4(img,ix,iy,cx,cy,mx,my);
   *cs=*s=*ss=0;
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++) if (ppixOK(img,x1,y1)) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      float *restrict resRow = (!NOSUBTRACT)?res[img][y1]:data[img][y1];
      FOR_X1_PHOT {
	 dx=x1-ix;
	 d+=cwtRow[dx];
	 c+=psfRow[dx]*cwtRow[dx];
	 (*s)+=(resRow[x1]-(*ssky)-(*mx)*dx-(*my)*dy)*cwtRow[dx];
	 (*ss)+=noise(img,x1,y1,*ssky+(*mx)*dx+(*my)*dy)*cwtRow[dx]*cwtRow[dx];
      }
   }
   if (*ss>0 && c!=0 && d!=0) {
      *cs=c/d;
      (*s)/=c;
      (*ss)=sqrt(*ss)/c;
   }
   else *ss=*s=*cs=0.;
   return;
}

static inline void eval1_psfphot_sky4(int img,int ix,int iy,int cx,int cy,float*s,float*ss,float*ssky,float*mx,float*my,float*cs,int NOSUBTRACT) {
   float I,P,D,DP,DX,DY,PP,X,Y,XX,XY,YY,XP,YP,W,WP,WPP,WXP,WYP,WX,WY,WXX,WXY,WYY;
   float a,n,skywt=0.,b0=0.,d,s0=0.,ssky0=0.,mx0=0,my0=0,p0=0.,Ppos,n1,n2,n3,n4;
   int y1,x1,dx,dy,it,npix;

   *mx=*my=0.0;
   if (fsky[img].lastoffset>=0) {
      b0=skyval(img,ix,iy);
      if (b0<0.) a=iRN[img];
      else a=iRN[img]+b0*invGAIN[img];
      skywt=25./a;
      //skywt=M_PI*(SQR(RSky1)-SQR(RSky0))/SQR(GetParamInt(SkipSky))/a;
   }
   for (it=0;it<GetParamInt(PSFPhotIt);it++) {
      if (it) {s0=*s; ssky0=*ssky; mx0=*mx; my0=*my;}
      I=P=D=DP=DX=DY=PP=X=Y=XX=XY=YY=XP=YP=W=WP=WPP=WXP=WYP=WX=WY=WXX=WXY=WYY=0;
      npix=0;
      I = skywt;
      D = b0*skywt;
      //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];dx<=circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1)) {
      FOR_Y1_PHOT {
	 const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
	 float *restrict cwtRow = cwt[cy][cx][img][dy];
	 float *restrict psfRow = psf[img][dy];
	 float *restrict resRow = res[img][y1];
	 FOR_X1_PHOT {
	    d=resRow[x1];
	    if (!it) n=noise(img,x1,y1,skyval(img,x1,y1));
	    else n=modelnoise(img,x1,y1,dx,dy,s0,ssky0+mx0*dx+my0*dy);
	    a=cwtRow[dx]/n;
	    npix++;
	    INCREMENT_PHOT
	 }
      }
      d = (XX*YY*P*P - P*P*XY*XY - 2*YY*P*X*XP + 2*P*X*XY*YP + 2*P*XP*XY*Y - 2*XX*P*Y*YP - X*X*YP*YP + PP*YY*X*X + 2*X*XP*Y*YP - 2*PP*X*XY*Y - XP*XP*Y*Y + I*YY*XP*XP - 2*I*XP*XY*YP + I*PP*XY*XY + PP*XX*Y*Y + I*XX*YP*YP - I*PP*XX*YY);
      if (d!=0 && P>0 && npix>3) {
	 *ssky = (D*(YY*XP*XP - 2*XP*XY*YP + PP*XY*XY + XX*YP*YP - PP*XX*YY)-DY*(XP*XP*Y - P*XP*XY + PP*X*XY + P*XX*YP - PP*XX*Y - X*XP*YP)-DX*(X*YP*YP + P*XP*YY - P*XY*YP - PP*X*YY + PP*XY*Y - XP*Y*YP)-DP*(P*XY*XY - P*XX*YY + X*XP*YY - X*XY*YP - XP*XY*Y + XX*Y*YP))/d;
	 *mx = (DX*(YY*P*P - 2*P*Y*YP + PP*Y*Y + I*YP*YP - I*PP*YY)-DY*(P*P*XY - I*PP*XY + I*XP*YP - P*X*YP - P*XP*Y + PP*X*Y)-DP*(XP*Y*Y - I*XP*YY + I*XY*YP + P*X*YY - P*XY*Y - X*Y*YP)-D*(X*YP*YP + P*XP*YY - P*XY*YP - PP*X*YY + PP*XY*Y - XP*Y*YP))/d;
	 *my = (DY*(XX*P*P - 2*P*X*XP + PP*X*X + I*XP*XP - I*PP*XX)-DX*(P*P*XY - I*PP*XY + I*XP*YP - P*X*YP - P*XP*Y + PP*X*Y)-DP*(X*X*YP + I*XP*XY - I*XX*YP - P*X*XY + P*XX*Y - X*XP*Y)-D*(XP*XP*Y - P*XP*XY + PP*X*XY + P*XX*YP - PP*XX*Y - X*XP*YP))/d;
	 if (!NOSUBTRACT) {
	    /*
	      Quick note: math assuming that X=Y=XY=XP=YP=0
	      d = (P*P-I*PP);
	      *s = (D*P-DP*I)/d;
	      *ssky = (DP*P-D*PP)/d;
	     */
	    n1 = YY*X*X - 2*X*XY*Y + I*XY*XY + XX*Y*Y - I*XX*YY;
	    n2 = X*X*YP + I*XP*XY - I*XX*YP - P*X*XY + P*XX*Y - X*XP*Y;
	    n3 = XP*Y*Y - I*XP*YY + I*XY*YP + P*X*YY - P*XY*Y - X*Y*YP;
	    n4 = P*XY*XY - P*XX*YY + X*XP*YY - X*XY*YP - XP*XY*Y + XX*Y*YP;
	    *s = (DP*n1-DY*n2-DX*n3-D*n4)/d;
	    *ss = sqrt(WPP*n1*n1+WYY*n2*n2+WXX*n3*n3+W*n4*n4-2*WYP*n1*n2-2*WXP*n1*n3-2*WP*n1*n4+2*WXY*n2*n3+2*WY*n2*n4+2*WX*n3*n4)/fabs(d);
	    *cs=PP/P;
	    //if (isnan(*ss)) printf("Caught NaN: npix=%d, %e %e\n",npix,WPP*n1*n1+WYY*n2*n2+WXX*n3*n3+W*n4*n4-2*WYP*n1*n2-2*WXP*n1*n3-2*WP*n1*n4+2*WXY*n2*n3+2*WY*n2*n4+2*WX*n3*n4,d);
	    if (isnan(*ss)) *s=*ss=*cs=*ssky=0;
	 }
	 else {
	    p0=P/(I+skywt);
	    I=P=D=DP=DX=DY=PP=X=Y=XX=XY=YY=XP=YP=W=WP=WPP=WXP=WYP=WX=WY=WXX=WXY=WYY=Ppos=0;
	    I = skywt;
	    D = b0*skywt;
	    //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];dx<=circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1) && psf[img][dy][dx]>=p0) {
	    FOR_Y1_PHOT {
	       const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
	       float *restrict cwtRow = cwt[cy][cx][img][dy];
	       float *restrict psfRow = psf[img][dy];
	       float *restrict resRow = res[img][y1];
	       float *restrict dataRow = data[img][y1];
	       FOR_X1_PHOT {
		  if (psfRow[dx]>=p0) d=dataRow[x1]-(*ssky)-(*mx)*dx-(*my)*dy;
		  else d=resRow[x1];
		  if (!it) n=noise(img,x1,y1,skyval(img,x1,y1));
		  else n=modelnoise(img,x1,y1,dx,dy,s0,ssky0+mx0*dx+my0*dy);
		  a=cwtRow[dx]/n;
		  if (psfRow[dx]>=p0) Ppos+=psfRow[dx]*a;
		  INCREMENT_PHOT
	       }
	    }
	    d = (XX*YY*P*P - P*P*XY*XY - 2*YY*P*X*XP + 2*P*X*XY*YP + 2*P*XP*XY*Y - 2*XX*P*Y*YP - X*X*YP*YP + PP*YY*X*X + 2*X*XP*Y*YP - 2*PP*X*XY*Y - XP*XP*Y*Y + I*YY*XP*XP - 2*I*XP*XY*YP + I*PP*XY*XY + PP*XX*Y*Y + I*XX*YP*YP - I*PP*XX*YY);
	    if (d!=0 && P>0) {
	       D -= (*ssky)*skywt*Ppos/P;
	       *s = (DP*(YY*X*X - 2*X*Y*XY + XX*Y*Y + I*XY*XY - I*XX*YY) - DX*(XP*Y*Y - YP*X*Y + P*X*YY - P*Y*XY - XP*I*YY + YP*I*XY) - D*(P*XY*XY + XP*X*YY - XP*Y*XY - YP*X*XY + YP*Y*XX - P*XX*YY) - DY*(YP*X*X - XP*X*Y - P*X*XY + P*Y*XX + XP*I*XY - YP*I*XX))/d;
	       n1 = YY*X*X - 2*X*XY*Y + I*XY*XY + XX*Y*Y - I*XX*YY;
	       n2 = X*X*YP + I*XP*XY - I*XX*YP - P*X*XY + P*XX*Y - X*XP*Y;
	       n3 = XP*Y*Y - I*XP*YY + I*XY*YP + P*X*YY - P*XY*Y - X*Y*YP;
	       n4 = P*XY*XY - P*XX*YY + X*XP*YY - X*XY*YP - XP*XY*Y + XX*Y*YP;
	       *ss = sqrt(WPP*n1*n1-WYP*2*n1*n2-WXP*2*n1*n3-WP*2*n1*n4+WYY*n2*n2+WXY*2*n2*n3+WY*2*n2*n4+WXX*n3*n3+WX*2*n3*n4+W*n4*n4)/fabs(d);
	       *cs=PP/P;
	       if (isnan(*ss)) *s=*ss=*cs=*ssky=0;
	    }
	    else *s=*ss=*cs=0.;
	 }
      }
      else *ss=*s=*cs=*ssky=0.;
   }
   return;
}
#undef INCREMENT_PHOT
#undef INCREMENT_PHOT_NOD
#undef INCREMENT_PHOT_SKY

static inline float eval1_snr(int img,int ix,int iy,int cx,int cy,float ct0) {
   float c=0,w,n,ss=0;
   int y1,x1,dx,dy;

   if (GetParamInt(FitSky)==3 && GetParamInt(PSFPhot)) return eval1_snr_sky3(img,ix,iy,cx,cy,ct0);
   if (GetParamInt(FitSky)==4 && GetParamInt(PSFPhot)) return eval1_snr_sky4(img,ix,iy,cx,cy,ct0);
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0],dx=circ[cy][cx][img][dy][0];dx<=circ[cy][cx][img][dy][1];x1++,dx++) if (ppixOK(img,x1,y1) && psf[img][dy][dx]>0) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      FOR_X1_PHOT {
	 n=noise(img,x1,y1,skyval(img,x1,y1));
	 if (ct0>0.) n+=ct0*psfRow[dx]*invGAIN[img];
	 if (!GetParamInt(PSFPhot)) w=cwtRow[dx];
	 else w=cwtRow[dx]*psfRow[dx]/n;
	 c+=psfRow[dx]*w;
	 ss+=w*w*n;
      }
   }
   if (c>0. && ss>0.) return ct0*c/sqrt(ss);
   return 0.;
}

#if 1
void eval1_phot(int img,int ix,int iy,int cx,int cy,int big,float*cm,float*s,float*ss,float*ssky,float*mx,float*my,float*cs,int NOSUBTRACT,int FORCE_SKY)
{
   *cm=apcor[img]/iEXP[img];
   *mx=*my=0.0;
   if (GetParamInt(FitSky)==3 && big==0) {
      if (GetParamInt(PSFPhot)) eval1_psfphot_sky3(img,ix,iy,cx,cy,s,ss,ssky,cs,NOSUBTRACT);
      else eval1_apphot_sky3(img,ix,iy,cx,cy,s,ss,ssky,cs,NOSUBTRACT);
   }
   else if (GetParamInt(FitSky)==4 && big==0) {
      if (GetParamInt(PSFPhot)) eval1_psfphot_sky4(img,ix,iy,cx,cy,s,ss,ssky,mx,my,cs,NOSUBTRACT);
      else eval1_apphot_sky4(img,ix,iy,cx,cy,s,ss,ssky,mx,my,cs,NOSUBTRACT);
   }
   else if (GetParamInt(PSFPhot)) eval1_psfphot(img,ix,iy,cx,cy,s,ss,ssky,cs,NOSUBTRACT,FORCE_SKY);
   else eval1_apphot(img,ix,iy,cx,cy,s,ss,ssky,cs,NOSUBTRACT,GetParamInt(FitSky),FORCE_SKY);
   (*s)*=(*cm);
   (*ss)*=(*cm);
   (*ssky)*=(*cm);
}

float eval1_fom(int img,int ix,int iy,int cx,int cy,int star,float s,float ss,float ssky,float mx,float my,float cs,float*c,float*ndof,float*sh,int NOSUBTRACT)
{
   int x1,y1,dx,dy,ct=0;
   float a,a0,b,A=0,np=0,ns,m=0,w,wchi,wmax[4],wmaxtot=0,tdn;

   wmax[0]=wmax[1]=wmax[2]=wmax[3]=0;
   //for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++) if (ppixOK(img,x1,y1)) {
   FOR_Y1_PHOT {
      const int circ0 = circ[cy][cx][img][dy][0]; const int circ1 = circ[cy][cx][img][dy][1];
      float *restrict cwtRow = cwt[cy][cx][img][dy];
      float *restrict psfRow = psf[img][dy];
      float *restrict resRow = res[img][y1];
      float *restrict dataRow = data[img][y1];
      FOR_X1_PHOT {
	 dx=x1-ix;
	 ct++;
	 if (GetParamInt(PSFPhot) && GetParamInt(PSFPhotIt)>1) ns=modelnoise(img,x1,y1,dx,dy,s,ssky+mx*dx+my*dy);
	 else ns=noise(img,x1,y1,ssky+mx*dx+my*dy);
	 a0=psfRow[dx]*s;
	 if (GetGlobalInt(IMDIFF) && star>=0) tdn=a0+psfRow[dx]*refcts[star]/refmult[img];
	 else tdn=a0;
	 if (tdn>0.) b=1./(ns+GetParamDouble(NoiseMult)*tdn*tdn);
	 else b=1./ns;
	 if (!NOSUBTRACT) a=resRow[x1];
	 else a=dataRow[x1];
	 a-=ssky+mx*dx+my*dy+a0;
	 w=cwtRow[dx];
	 wchi=chiwt[cy][cx][img][dy][dx];
	 (*c)+=a*a*b*wchi;
	 if (wchi>wmax[0]) {wmax[3]=wmax[2]; wmax[2]=wmax[1]; wmax[1]=wmax[0]; wmax[0]=wchi;}
	 else if (wchi>wmax[1]) {wmax[3]=wmax[2]; wmax[2]=wmax[1]; wmax[1]=wchi;}
	 else if (wchi>wmax[2]) {wmax[3]=wmax[2]; wmax[2]=wchi;}
	 else if (wchi>wmax[3]) wmax[3]=wchi;
	 np+=wchi;
	 (*sh)+=a/sqrt(ns)*(psfRow[dx]-cs)*w;
	 if (GetGlobalInt(IMDIFF) && star>=0) A+=(psfRow[dx]-cs)*(psfRow[dx]-cs)*(s+refcts[star]/refmult[img])/sqrt(ns)*w;
	 else A+=(psfRow[dx]-cs)*(psfRow[dx]-cs)*s/sqrt(ns)*w;
      }
   }
   if (A>0 && ct>1) (*sh)/=A;
   else (*sh)=-9.999;
   if (*sh>9.999) *sh=9.999;
   else if (*sh<-9.999) *sh=-9.999;
   if (GetParamInt(FitSky)==4) wmaxtot=wmax[0]+wmax[1]+wmax[2]+wmax[3];
   else if (GetParamInt(FitSky)==3) wmaxtot=wmax[0]+wmax[1];
   else wmaxtot=wmax[0];
   if (ct>1 && np>wmaxtot) {
      *ndof=np-wmaxtot;
      (*c)=(*c)/(np-wmaxtot);
   }
   else {
      *c=1.;
      *ndof=0;
   }
   m=1/sqrt((*c)+0.25); // for chi minimization
   if (GetParamInt(SearchMode)==0) {
      m*=s/ss; // convert to SNR/chi maximization
   }
   if (m<0.0001) m=0.0001;
   *c = sqrt(*c);
#ifdef NAN_PRINT
   if (isnan(m)) {
      printf("Uncaught m=nan\n");
      fflush(stdout);
   }
#endif
#ifdef NAN_CRASH
   assert(!isnan(m));
#else
   if (isnan(m)) return 0;
#endif
   return m;
}

float eval1(int img,double x,double y,float pa,float pb,float pc,float*s,float*ss,float*c,float*ndof,float*sh,float*ssky,float*mx,float*my,float*cs,float*cm,int phot,int full,int star,int newpsf,int NOSUBTRACT,int CALC_SKY,int FORCE_SKY)
{
   int ix,iy,cx,cy,big=0;

   shift(img,x,y,&x,&y,1);
   ix=(int)(x+100)-100;
   iy=(int)(y+100)-100;
   if (ix<-rphot[img] || ix>=dataim[img].X+rphot[img] || iy<-rphot[img] || iy>=dataim[img].Y+rphot[img]) {
      *s=*ss=*c=*sh=*ssky=*ndof=0;
      return 0;
   }
   cx=(int)((x-ix)*50+0.5);
   if (cx<0) cx=0; if (cx>50) cx=50;
   cy=(int)((y-iy)*50+0.5);
   if (cy<0) cy=0; if (cy>50) cy=50;
   if (newpsf) calc1psf(img,x,y,rphot[img],pa,pb,pc,1,1);

   if (phot) {
      *mx=*my=0.0;
      if (GetGlobalDouble(RBig)>0 && pa==GetGlobalDouble(RBig) && pb==GetGlobalDouble(RBig) && pc==0.) big=1;
      eval1_phot(img,ix,iy,cx,cy,big,cm,s,ss,ssky,mx,my,cs,NOSUBTRACT,FORCE_SKY);
   }

   *c=*sh=*ndof=0;
   if (!(*ss>0)) return 0;
   if (full) return eval1_fom(img,ix,iy,cx,cy,star,(*s)/(*cm),(*ss)/(*cm),(*ssky)/(*cm),*mx,*my,*cs,c,ndof,sh,NOSUBTRACT);
   return ((*s)/(*cm))/((*ss)/(*cm));
}

#else
float eval1(int img,double x,double y,float pa,float pb,float pc,float*s,float*ss,float*c,float*ndof,float*sh,float*ssky,float*cm,int phot,int full,int star,int newpsf,int NOSUBTRACT,int CALC_SKY,int FORCE_SKY) {
   int ix,iy,cx,cy,x1,y1,dx,dy,ct=0,NOSKY34=0;
   float cs,a,a0,b,A,np,ns,m=0,w,wchi,wmax[4],wmaxtot=0,tdn,mx=0,my=0;

   *cm=apcor[img]/iEXP[img];
   shift(img,x,y,&x,&y,1);
   ix=(int)(x+100)-100;
   iy=(int)(y+100)-100;
   if (ix<-rphot[img] || ix>=dataim[img].X+rphot[img] || iy<-rphot[img] || iy>=dataim[img].Y+rphot[img]) {
      *s=*ss=*c=*sh=*ssky=*ndof=0;
      return 0;
   }
   cx=(int)((x-ix)*50+0.5);
   if (cx<0) cx=0; if (cx>50) cx=50;
   cy=(int)((y-iy)*50+0.5);
   if (cy<0) cy=0; if (cy>50) cy=50;
   if (newpsf) calc1psf(img,x,y,rphot[img],pa,pb,pc,1,1);
   if (GetGlobalDouble(RBig)>0 && pa==GetGlobalDouble(RBig) && pb==GetGlobalDouble(RBig) && pc==0.) NOSKY34=1;
   if (GetParamInt(FitSky)==3 && !NOSKY34) {
      if (GetParamInt(PSFPhot)) eval1_psfphot_sky3(img,ix,iy,cx,cy,s,ss,ssky,&cs,NOSUBTRACT);
      else eval1_apphot_sky3(img,ix,iy,cx,cy,s,ss,ssky,&cs,NOSUBTRACT);
   }
   else if (GetParamInt(FitSky)==4 && !NOSKY34) {
      if (GetParamInt(PSFPhot)) eval1_psfphot_sky4(img,ix,iy,cx,cy,s,ss,ssky,&mx,&my,&cs,NOSUBTRACT);
      else eval1_apphot_sky4(img,ix,iy,cx,cy,s,ss,ssky,&mx,&my,&cs,NOSUBTRACT);
   }
   else if (GetParamInt(PSFPhot)) eval1_psfphot(img,ix,iy,cx,cy,s,ss,ssky,&cs,NOSUBTRACT,FORCE_SKY);
   else eval1_apphot(img,ix,iy,cx,cy,s,ss,ssky,&cs,NOSUBTRACT,GetParamInt(FitSky),FORCE_SKY);
   *c=*sh=A=np=*ndof=0;
   if (full && *ss>0) {
      ct=0;
      wmax[0]=wmax[1]=wmax[2]=wmax[3]=0;
      for (y1=iy-rphot[img],dy=-rphot[img];y1<=iy+rphot[img];y1++,dy++) for (x1=ix+circ[cy][cx][img][dy][0];x1<=ix+circ[cy][cx][img][dy][1];x1++) if (ppixOK(img,x1,y1)) {
	 dx=x1-ix;
	 ct++;
	 if (GetParamInt(PSFPhot) && GetParamInt(PSFPhotIt)>1) ns=modelnoise(img,x1,y1,dx,dy,*s,*ssky+mx*dx+my*dy);
	 else ns=noise(img,x1,y1,(*ssky)+mx*dx+my*dy);
	 a0=psf[img][dy][dx]*(*s);
	 if (GetGlobalInt(IMDIFF) && star>=0) tdn=a0+psf[img][dy][dx]*refcts[star]/refmult[img];
	 else tdn=a0;
	 if (tdn>0.) b=1./(ns+GetParamDouble(NoiseMult)*tdn*tdn);
	 else b=1./ns;
	 if (!NOSUBTRACT) a=res[img][y1][x1];
	 else a=data[img][y1][x1];
	 a-=(*ssky)+mx*dx+my*dy+a0;
	 w=cwt[cy][cx][img][dy][dx];
	 wchi=chiwt[cy][cx][img][dy][dx];
	 (*c)+=a*a*b*wchi;
	 if (wchi>wmax[0]) {wmax[3]=wmax[2]; wmax[2]=wmax[1]; wmax[1]=wmax[0]; wmax[0]=wchi;}
	 else if (wchi>wmax[1]) {wmax[3]=wmax[2]; wmax[2]=wmax[1]; wmax[1]=wchi;}
	 else if (wchi>wmax[2]) {wmax[3]=wmax[2]; wmax[2]=wchi;}
	 else if (wchi>wmax[3]) wmax[3]=wchi;
	 np+=wchi;
	 (*sh)+=a/sqrt(ns)*(psf[img][dy][dx]-cs)*w;
	 if (GetGlobalInt(IMDIFF) && star>=0) A+=(psf[img][dy][dx]-cs)*(psf[img][dy][dx]-cs)*(*s+refcts[star]/refmult[img])/sqrt(ns)*w;
	 else A+=(psf[img][dy][dx]-cs)*(psf[img][dy][dx]-cs)*(*s)/sqrt(ns)*w;
      }
      if (A>0 && ct>1) (*sh)/=A;
      else (*sh)=-9.999;
      if (*sh>9.999) *sh=9.999;
      else if (*sh<-9.999) *sh=-9.999;
      if (GetParamInt(FitSky)==4) wmaxtot=wmax[0]+wmax[1]+wmax[2]+wmax[3];
      else if (GetParamInt(FitSky)==3) wmaxtot=wmax[0]+wmax[1];
      else wmaxtot=wmax[0];
      if (ct>1 && np>wmaxtot) {
	 *ndof=np-wmaxtot;
	 (*c)=(*c)/(np-wmaxtot);
      }
      else {
	 *c=1.;
	 *ndof=0;
      }
      m=1/sqrt((*c)+0.25); // for chi minimization
      if (GetParamInt(SearchMode)==0) {
	 m*=(*s)/(*ss); // convert to SNR/chi maximization
      }
      if (m<0.0001) m=0.0001;
      *c = sqrt(*c);
   }
   else if (*ss>0) m=(*s)/(*ss);
   (*s)*=(*cm);
   (*ss)*=(*cm);
   (*ssky)*=(*cm);
#ifdef NAN_PRINT
   if (isnan(m)) {
      printf("Uncaught mm=nan\n");
      fflush(stdout);
   }
#endif
#ifdef NAN_CRASH
   assert(!isnan(m));
#else
   if (isnan(m)) *s=*ss=*c=*sh=*ssky=*ndof=m=0;
#endif
   return m;
}
#endif

// GLOBAL (PARALLEL-SAFE) VARIABLES - each thread has different image
typedef struct {
   int calcS0[MAXNIMG];
   float s0[MAXNIMG],s[MAXNIMG],ss[MAXNIMG],c[MAXNIMG],ndof[MAXNIMG],sh[MAXNIMG],ssky[MAXNIMG],cs[MAXNIMG],cm[MAXNIMG];
} eval1_data_type;
static eval1_data_type eval1_data;

float runeval1(int img,double x,double y,float pa,float pb,float pc,int star,int calcS0)
{
   float m,mx,my;
   if (calcS0) {
      eval1(img,x,y,pa,pb,pc,eval1_data.s0+img,eval1_data.ss+img,eval1_data.c+img,eval1_data.ndof+img,eval1_data.sh+img,eval1_data.ssky+img,&mx,&my,eval1_data.cs+img,eval1_data.cm+img,1,0,star,1,1,0,0);
      m=eval1(img,x,y,pa,pb,pc,eval1_data.s+img,eval1_data.ss+img,eval1_data.c+img,eval1_data.ndof+img,eval1_data.sh+img,eval1_data.ssky+img,&mx,&my,eval1_data.cs+img,eval1_data.cm+img,1,1,star,0,0,0,0);
      if (eval1_data.s0[img]>0. && eval1_data.s[img]>0.) {
	 eval1_data.s0[img]=-2.5*log10(eval1_data.s[img]/eval1_data.s0[img]);
	 if (eval1_data.s0[img]>9.999) eval1_data.s0[img]=9.999;
	 else if (eval1_data.s0[img]<0.00001) eval1_data.s0[img]=0.00001;
      }
      else if (eval1_data.s0[img]>0.) eval1_data.s0[img]=9.999;
      else eval1_data.s0[img]=0.00001;
   }
   else {
      m=eval1(img,x,y,pa,pb,pc,eval1_data.s+img,eval1_data.ss+img,eval1_data.c+img,eval1_data.ndof+img,eval1_data.sh+img,eval1_data.ssky+img,&mx,&my,eval1_data.cs+img,eval1_data.cm+img,1,1,star,1,0,0,0);
      eval1_data.s0[img]=0.00001;
   }
   return m;
}

/* AEDDEBUG
  To-do:
  - bad photometry points (saturated, etc. that wouldn't be combined in out1star) shouldn't be included in eval outputs, especially combined forced photometry
 */
float eval(int IMG,double x,double y,float*PA,float*PB,float*PC,float*S,float*S0,float*SS,float*C,float*SH,float*SSKY,float*IS,float*IS0,float*ISS,float*IC,float*ISH,float*ISSKY,float*ICM,int checkstar,int star) {
   int img,comb[MAXNIMG],calcS0;
   float m[MAXNIMG];
   float M=0,wt,swt,twt=0,tswt=0,tndof=0;

   *S=*SS=*C=*SH=*SSKY=0;
   if (S0) *S0=0;
   calcS0 = ((IS0 || S0) && !GetGlobalInt(IMDIFF));
   if (IMG>=0) {
      eval1_data.s[IMG]=eval1_data.ss[IMG]=0.0;
      if (checkstar<0 || !star_flag(stars[checkstar].x,stars[checkstar].y,IMG)) m[IMG]=runeval1(IMG,x,y,PA[IMG],PB[IMG],PC[IMG],star,calcS0);
   }
   else {
#ifdef DOLPHOT_THREADED
      if (Nimg==1 || omp_get_max_threads()<2) {
	 for (img=0;img<Nimg;img++) {
	    eval1_data.s[img]=eval1_data.ss[img]=0.0;
	    if (checkstar<0 || !star_flag(stars[checkstar].x,stars[checkstar].y,img)) m[img]=runeval1(img,x,y,PA[img],PB[img],PC[img],star,calcS0);
	    if (GetParamInt(ForceSameMag)) {
	       comb[img]=1;
	       if (eval1_data.ss[img]>0.0 && hstmode[img].inst!=NONE) comb[img]=0;
	    }
	 }
      }
      else
#pragma omp parallel for schedule(guided)
#endif
	 for (img=0;img<Nimg;img++) {
	    eval1_data.s[img]=eval1_data.ss[img]=0.0;
	    if (checkstar<0 || !star_flag(stars[checkstar].x,stars[checkstar].y,img)) m[img]=runeval1(img,x,y,PA[img],PB[img],PC[img],star,calcS0);
	    if (GetParamInt(ForceSameMag)) {
	       comb[img]=1;
	       if (eval1_data.ss[img]>0.0 && hstmode[img].inst!=NONE) comb[img]=0;
	    }
	 }
   }
   if (GetParamInt(ForceSameMag) && IMG<0) {
      for (img=0;img<Nimg;img++) if (!comb[img]) {
	 int j;
	 float ts=0,twt=0,wt;
	 for (j=img;j<Timg;j++) if (!comb[j] && hstmode[j].inst==hstmode[img].inst && hstmode[j].filt==hstmode[img].filt) {
	    wt = 1.0/(eval1_data.ss[j]*eval1_data.ss[j]);
	    ts += eval1_data.s[j]*wt;
	    twt += wt;
	 }
	 if (twt>0.0) {
	    float mx,my;
	    ts /= twt;
	    for (j=img;j<Timg;j++) if (!comb[j] && hstmode[j].inst==hstmode[img].inst && hstmode[j].filt==hstmode[img].filt) {
	       eval1_data.s[j] = ts;
	       m[j] = eval1(j,x,y,PA[j],PB[j],PC[j],eval1_data.s+j,eval1_data.ss+j,eval1_data.c+j,eval1_data.ndof+j,eval1_data.sh+j,eval1_data.ssky+j,&mx,&my,eval1_data.cs+j,eval1_data.cm+j,0,1,star,1,0,0,0);
	       comb[j] = 1;
	    }
	 }
      }
   }
   for (img=0;img<Timg && (img<Nimg || IMG>=0);img++) if ((img==IMG || IMG==-1) && (checkstar<0 || !star_flag(stars[checkstar].x,stars[checkstar].y,img))) {
      if (m[img]<=0.) wt=swt=0.;
      else if (GetGlobalInt(IMDIFF) && IMG<0 && star>=0) {
	 swt=wt=(eval1_data.s[img]+refcts[star]/refmult[img]*eval1_data.cm[img])/eval1_data.ss[img]/eval1_data.ss[img];
	 if (swt<0.) {
	    swt=0.;
	    wt=-wt;
	 }
      }
      else if (GetGlobalInt(IMDIFF) && star>=0) {
	 wt=1./eval1_data.ss[img]/eval1_data.ss[img];
	 swt=wt*(eval1_data.s[img]+refcts[star]/refmult[img]*eval1_data.cm[img]);
	 if (swt<0.) swt=0.;
      }
      else if (IMG<0) {
	 if (eval1_data.s[img]>0) wt=swt=eval1_data.s[img]/eval1_data.ss[img]/eval1_data.ss[img];
	 else {
	    wt=-eval1_data.s[img]/eval1_data.ss[img]/eval1_data.ss[img];
	    swt=0.;
	 }
      }
      else {
	 wt=1./eval1_data.ss[img]/eval1_data.ss[img];
	 if (eval1_data.s[img]>0) swt=wt*eval1_data.s[img];
	 else swt=0.;
      }
      twt+=wt;
      tswt+=swt;
      tndof+=eval1_data.ndof[img];
      if (IS) IS[img]=eval1_data.s[img];
      if (IS0) IS0[img]=eval1_data.s0[img];
      if (ISS) ISS[img]=eval1_data.ss[img];
      if (ICM) ICM[img]=eval1_data.cm[img];
      if (IC) IC[img]=eval1_data.c[img];
      if (ISH) ISH[img]=eval1_data.sh[img];
      if (ISSKY) ISSKY[img]=eval1_data.ssky[img];
      (*S)+=eval1_data.s[img]*wt;
      if (S0) (*S0)+=eval1_data.s0[img]*wt;
      (*SS)+=eval1_data.ss[img]*eval1_data.ss[img]*wt*wt;
      (*C)+=eval1_data.c[img]*eval1_data.c[img]*eval1_data.ndof[img];
      (*SH)+=eval1_data.sh[img]*swt;
      (*SSKY)+=eval1_data.ssky[img]*wt;
   }
   if (tndof) (*C)=sqrt((*C)/tndof);
   if (tswt) (*SH)/=tswt;
   if (twt) {
      *SS=sqrt(*SS)/twt;
      if (IMG<0 && *S>0) {
	 float tss;
	 tss=sqrt(*S)/twt;
	 if (tss>*SS) *SS=tss;
      }
      (*S)/=twt;
      if (S0) (*S0)/=twt;
      (*SSKY)/=twt;
      if (*S) {
	 M=1./(sqrt((*C)*(*C)+0.25));
	 if (GetParamInt(SearchMode)==0) M*=(*S)/(*SS);
      }
   }
   assert(!isnan(M));
   return M;
}

static inline float getnc(float m[3]) {
   float x;
   if ((x=2*m[1]-m[0]-m[2])<=0) {
      if (m[1]>=m[0] && m[1]>=m[2]) return 0.;
      else if (m[0]>=m[2]) return -1.;
      return 1.;
   }
   x=(m[2]-m[0])*0.5/x;
   if (x<-1) return -1.;
   if (x>1) return 1.;
   return x;
}

static inline void getshape(float a,float b,float c,float*r,float*e) {
   float r1,r2,x;
   c/=a*b;
   a=1./(a*a);
   b=1./(b*b);
   *r=1./sqrt(sqrt(a*b-c*c*0.25));
   x=sqrt((a-b)*(a-b)+c*c);
   r1=1./sqrt((a+b-x)*0.5);
   r2=1./sqrt((a+b+x)*0.5);
   *e=1.-r2/r1;
   return;
}

static inline int getclass(float a,float b,float c,int force1OK) {
   float r,e;

   if (force1OK && GetParamInt(Force1)) return 1;
   getshape(a,b,c,&r,&e);
   if (r>GetParamDouble(MaxS)) return 5;
   if (r<GetParamDouble(MinS)) return 4;
   if (e>GetParamDouble(MaxE)) return 3;
   return 1;
}

static inline int getsclass(int i) {
   float pa=0,pb=0,pc=0;
   int img;

   for (img=0;img<Nimg;img++) {
      pa+=stars[i].pa[img];
      pb+=stars[i].pb[img];
      pc+=stars[i].pc[img];
   }
   return getclass(pa/Nimg,pb/Nimg,pc/Nimg,0);
}

/*
void checkstars(char*estr,int verb) {
   int i,img;
   for (i=0;i<Nstars;i++) for (img=0;img<Nimg;img++) if (stars[i].is[img]>0.) {
      if (stars[i].type==4) {if (fabs(stars[i].pa[img]-0.1)>0.001 || fabs(stars[i].pb[img]-0.1)>0.001 || stars[i].pc[img]!=0.) {printf("ACK %s (%d %d %f %f %f)\n",estr,i,stars[i].type,stars[i].pa[img],stars[i].pb[img],stars[i].pc[img]); exit(-1);}}
      else if (stars[i].type==5) {if (fabs(stars[i].pa[img]-GetGlobalDouble(RBig))>0.001 || fabs(stars[i].pb[img]-GetGlobalDouble(RBig))>0.001 || stars[i].pc[img]!=0.) {printf("ACK %s (%d %d %f %f %f)\n",estr,i,stars[i].type,stars[i].pa[img],stars[i].pb[img],stars[i].pc[img]); exit(-1);}}
      else {if (fabs(stars[i].pa[img]-1.0)>0.001 || fabs(stars[i].pb[img]-1.0)>0.001 || stars[i].pc[img]!=0.) {printf("ACK %s (%d %d %f %f %f)\n",estr,i,stars[i].type,stars[i].pa[img],stars[i].pb[img],stars[i].pc[img]); exit(-1);}}
   }
   if (verb) printf("passed %s\n",estr);
   return;
}
*/

static inline void setpsfpars(int i) {
   int img;
   if (stars[i].type<4) {
      getpsfpars(-1,stars[i].x,stars[i].y,stars[i].pa,stars[i].pb,stars[i].pc);
      return;
   }
   for (img=0;img<Timg;img++) {
      stars[i].pc[img]=0.;
      if (stars[i].type==4) stars[i].pa[img]=stars[i].pb[img]=0.1;
      else stars[i].pa[img]=stars[i].pb[img]=GetGlobalDouble(RBig);
   }
   return;
}

static inline void zerostar(int IMG,float*bis) {
   int img;
   for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) bis[img]=0.;
   return;
}

/*
   // NEW=0 CASE (normal)
   LOOP {
      it++;
      q is estimate of dx/PosStep (calculated using eval)
      if (it>5) q*=5./it;
      x+=q*PosStep;
      if |q|<=0.6 loop will exit
   }
   // plain English: keep evaluating with PosStep, until both axes adjust
   // less than 60% of PosStep - should be fast

   // NEW=1 CASE (first search)
   step initialized to PosStep*2.4
   LOOP {
      q is estimate of dx/step (calculated using eval)
      x+=q*step;
      scale = min(0.95, max(0.4,|q|))
      if (step<=PosStep) loop will exit
      step = max(step*scale, PosStep*0.9)
   }

   // NEW=-1 CASE (final fine photometry)
   step initialized to PosStep
   LOOP {
      q is estimate of dx/step (calculated using eval)
      x+=q*step;
      scale = min(0.9, max(0.2,|q|))
      if (step<=PosStep*0.5 AND step*scale<=PosStep*0.101) loop will exit
      scale = min(0.9, max(0.4,|q|))
      step = max(step*scale, PosStep*0.1)
   }
*/
float photsearch(int IMG,double*x,double*y,float*pa,float*pb,float*pc,float*s,float*ss,float*chi,float*sh,float*ssky,float*bis,float*biss,float*bicm,int *cl,int new,int id) {
   float step,stepc,m[3],q,orig,is,iss,ichi,indof,ish,isky,imx,imy,ics,icm,ma=0,mb=0,mc=0,sn[3],scale=0.75,aq;
   int i,img,it=0,CONT=1,N=0;
#define amin 0.05
#define cmax 1.90
#define PAD 1

   step=GetParamDouble(PosStep);
   if (new==1) step*=2.4; // with max scale=0.4 still allows 2-step converge
   if (((!GetGlobalInt(fitpsf) || *s<(*ss)*GetParamDouble(SigPSF)) && *cl<=2) || new==1 || GetParamDouble(PSFStep)<=0.) getpsfpars(IMG,*x,*y,pa,pb,pc);
   clear_sky();
   if (!GetGlobalInt(WARMSTART)) while (CONT) {
      if (new==1) scale=0.4;
      else if (new==-1) scale=0.2;
      CONT=0;
      it++;
      m[0]=eval(IMG,(*x)-step,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
      m[1]=eval(IMG,*x,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
      m[2]=eval(IMG,(*x)+step,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
      q=getnc(m);
      if (new==0 && it>5) q*=5./it;
      if ((new==1 || new==-1) && scale<(aq=fabs(q))) scale=aq;
      if (q<-0.6 || q>0.6) CONT=1;
      *x+=q*step;
      assert(!isnan(*x));
      if (*x<-PAD || *x>=GetGlobalInt(XMAX)+PAD) {*s=0; if (bis) zerostar(IMG,bis); return 0;}
      m[0]=eval(IMG,*x,(*y)-step,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
      m[1]=eval(IMG,*x,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
      m[2]=eval(IMG,*x,(*y)+step,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
      q=getnc(m);
      if (new==0 && it>5) q*=5./it;
      if ((new==1 || new==-1) && scale<(aq=fabs(q))) scale=aq;
      if (q<-0.6 || q>0.6) CONT=1;
      *y+=q*step;
      assert(!isnan(*y));
      if (*y<-PAD || *y>=GetGlobalInt(YMAX)+PAD) {*s=0; if (bis) zerostar(IMG,bis); return 0;}
      if (new==-1) {
	 if (scale>0.9) scale=0.9;
	 if (step>GetParamDouble(PosStep)*0.5 || step*scale>GetParamDouble(PosStep)*0.101) CONT=1;
	 else CONT=0;
	 if (scale<0.4) scale=0.4;
	 step*=scale;
	 if (step<GetParamDouble(PosStep)*0.1) step=GetParamDouble(PosStep)*0.1;
      }
      if (new==1) {
	 if (step>GetParamDouble(PosStep)) CONT=1;
	 else CONT=0;
	 if (scale>0.95) scale=0.95;
	 step*=scale;
	 if (step<GetParamDouble(PosStep)*0.9) step=GetParamDouble(PosStep)*0.9;
      }
   }
   m[1]=eval(IMG,*x,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
   if (*s<=0) {if (bis) zerostar(IMG,bis); return 0;}
   sn[1]=(*s)/(*ss);
   if (((GetGlobalInt(fitpsf) || new==1) && sn[1]>=GetParamDouble(SigPSF)) || *cl>2) {
      if (GetParamDouble(PSFStep)>0.) {
	 for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	    CONT=1;
	    it=0;
	    if (new==1) {
	       step=GetParamDouble(PSFStep)*2;
	       if (pa[img]>step*3) step=pa[img]*0.333;
	       if (pb[img]>step*3) step=pb[img]*0.333;
	       stepc=1.2;
	    }
	    else {
	       step=GetParamDouble(PSFStep);
	       if (pa[img]>step*5) step=pa[img]*0.2;
	       if (pb[img]>step*5) step=pb[img]*0.2;
	       stepc=0.6;
	    }
	    if (GetParamInt(EPSF)) {while (CONT) {
	       CONT=0;
	       it++;
	       if (pa[img]<step+amin) {
		  orig=(pa[img]-amin)/step-1;
		  pa[img]=step+amin;
	       }
	       else orig=0;
	       for (i=-1;i<=1;i++) m[i+1]=eval1(img,*x,*y,pa[img]+i*step,pb[img],pc[img],&is,&iss,&ichi,&indof,&ish,&isky,&imx,&imy,&ics,&icm,1,1,id,1,0,0,0);
	       q=getnc(m);
	       if (new==0 && it>3) q=3./it*(q-orig)+orig;
	       if (q<-0.6+orig || q>0.6+orig) CONT=1;
	       pa[img]+=q*step;
	       assert(!isnan(pa[img]));
	       if (pb[img]<step+amin) {
		  orig=(pb[img]-amin)/step-1;
		  pb[img]=step+amin;
	       }
	       else orig=0;
	       for (i=-1;i<=1;i++) m[i+1]=eval1(img,*x,*y,pa[img],pb[img]+i*step,pc[img],&is,&iss,&ichi,&indof,&ish,&isky,&imx,&imy,&ics,&icm,1,1,id,1,0,0,0);
	       q=getnc(m);
	       if (new==0 && it>3) q=3./it*(q-orig)+orig;
	       if (q<-0.6+orig || q>0.6+orig) CONT=1;
	       pb[img]+=q*step;
	       assert(!isnan(pb[img]));
	       if (pc[img]>cmax-stepc) {
		  orig=(pc[img]-cmax)/stepc+1;
		  pc[img]=cmax-stepc;
	       }
	       else if (pc[img]<stepc-cmax) {
		  orig=(pc[img]+cmax)/stepc-1;
		  pc[img]=stepc-cmax;
	       }
	       else orig=0;
	       for (i=-1;i<=1;i++) m[i+1]=eval1(img,*x,*y,pa[img],pb[img],pc[img]+i*stepc,&is,&iss,&ichi,&indof,&ish,&isky,&imx,&imy,&ics,&icm,1,1,id,1,0,0,0);
	       q=getnc(m);
	       if (new==0 && it>3) q=3./it*(q-orig)+orig;
	       if (q<-0.6+orig || q>0.6+orig) CONT=1;
	       pc[img]+=q*stepc;
	       assert(!isnan(pc[img]));
	       if (new==-1) {
		  if (CONT) {step*=0.95; stepc*=0.95;}
		  else {step*=0.6; stepc*=0.6;}
		  if (step>=0.1*GetParamDouble(PSFStep) || stepc>=0.1) CONT=1;
		  else CONT=0;
	       }
	       if (new==1) {
		  if (CONT) {step*=0.95; stepc*=0.95;}
		  else {step*=0.6; stepc*=0.6;}
		  if (step>=GetParamDouble(PSFStep) || stepc>=0.6) CONT=1;
		  else CONT=0;
	       }
	    }}
	    else {
	    if (pa[img]>step*5) step=(pa[img])*0.2;
	    while (CONT) {
	       CONT=0;
	       it++;
	       if (pa[img]<step+amin) {
		  orig=(pa[img]-amin)/step-1;
		  pa[img]=step+amin;
	       }
	       else orig=0;
	       for (i=-1;i<=1;i++) m[i+1]=eval1(img,*x,*y,pa[img]+i*step,pa[img]+i*step,pc[img],&is,&iss,&ichi,&indof,&ish,&isky,&imx,&imy,&ics,&icm,1,1,id,1,0,0,0);
	       q=getnc(m);
	       if (new==0 && it>3) q=3./it*(q-orig)+orig;
	       if (q<-0.6+orig || q>0.6+orig) CONT=1;
	       pa[img]+=q*step;
	       assert(!isnan(pa[img]));
	       pb[img]=pa[img];
	       pc[img]=0;
	       if (new==-1) {
		  if (CONT) step*=0.95;
		  else step*=0.6;
		  if (step>=0.1*GetParamDouble(PSFStep)) CONT=1;
		  else CONT=0;
	       }
	       if (new==1) {
		  if (CONT) step*=0.95;
		  else step*=0.6;
		  if (step>=GetParamDouble(PSFStep)) CONT=1;
		  else CONT=0;
	       }
	    }}
	    ma+=pa[img];
	    mb+=pb[img];
	    mc+=pc[img];
	    N++;
	 }
      }
      else if (!GetParamInt(Force1)) {
	 for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	    pa[img]=pb[img]=0.1;
	    pc[img]=0.;
	 }
	 m[0]=eval(IMG,*x,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
	 sn[0]=(*s)/(*ss);
	 if (GetParamInt(FitSky)!=2 && GetParamInt(FitSky)!=3 && GetParamInt(FitSky)!=4) {
	    for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) pa[img]=pb[img]=GetGlobalDouble(RBig);
	    m[2]=eval(IMG,*x,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,NULL,NULL,NULL,NULL,NULL,NULL,NULL,-1,id);
	    sn[2]=(*s)/(*ss);
	 }
	 else {
	    m[2] = m[1]-100.0;
	    sn[2] = 0.;
	 }
	 if (m[0]>m[1] && m[0]>m[2]) {
	    sn[1]=sn[0];
	    for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	       pa[img]=pb[img]=0.1;
	       pc[img]=0.;
	    }
	 }
	 else if (m[2]>m[1]) {
	    sn[1]=sn[2];
	    for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	       pa[img]=pb[img]=GetGlobalDouble(RBig);
	       pc[img]=0.;
	    }
	 }
	 else getpsfpars(IMG,*x,*y,pa,pb,pc);
	 for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	    ma+=pa[img];
	    mb+=pb[img];
	    mc+=pc[img];
	    N++;
	 }
      }
      else {
	 getpsfpars(IMG,*x,*y,pa,pb,pc);
	 for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	    ma+=pa[img];
	    mb+=pb[img];
	    mc+=pc[img];
	    N++;
	 }
      }
      if (!GetGlobalInt(WARMSTART)) {
	 *cl=getclass(ma/N,mb/N,mc/N,1);
	 if (sn[1]<GetParamDouble(SigPSF)) *cl=2;
	 if (GetParamDouble(PSFStep)<=0. && !GetParamInt(Force1)) {
	    for (img=0;img<Timg;img++) if ((IMG<0 && img<Nimg) || img==IMG) {
	       if (*cl==4) {
		  pa[img]=pb[img]=0.1;
		  pc[img]=0.;
	       }
	       else if (*cl==5) {
		  pa[img]=pb[img]=GetGlobalDouble(RBig);
		  pc[img]=0.;
	       }
	       else getpsfpars(img,*x,*y,pa,pb,pc);
	    }
	 }
	 else if (!GetGlobalInt(fitpsf) && (*cl==1 || *cl==2)) getpsfpars(IMG,*x,*y,pa,pb,pc);
      }
   }
   else {
      getpsfpars(IMG,*x,*y,pa,pb,pc);
      if (!GetGlobalInt(WARMSTART)) {
	 if (sn[1]<GetParamDouble(SigPSF)) *cl=2;
	 else *cl=1;
      }
   }
   m[0]=eval(IMG,*x,*y,pa,pb,pc,s,NULL,ss,chi,sh,ssky,bis,NULL,biss,NULL,NULL,NULL,bicm,-1,id);
   sn[0]=(*s)/(*ss);
   if (!GetGlobalInt(WARMSTART) && sn[0]<GetParamDouble(SigPSF)) {
      if (!GetGlobalInt(fitpsf) && *cl>2) getpsfpars(IMG,*x,*y,pa,pb,pc);
      *cl=2;
   }
   if (GetGlobalInt(WARMSTART) && *cl<3) {
      if (sn[0]<GetParamDouble(SigPSF)) *cl=2;
      *cl=1;
   }
   return m[0];
}

void avgstar(int i,stype *s) {
   double m;
   int img;

   if (s->type==1) stars[i].type=1;
   if (s->type==2 && stars[i].type>2) stars[i].type=2;
   if (stars[i].s>0 && s->s>0) m=stars[i].s/(stars[i].s+(s->s));
   else if (stars[i].s>0) m=1.0;
   else if (s->s>0) m=0.0;
   else m=0.5;
   stars[i].x=stars[i].x*m+(s->x)*(1-m);
   stars[i].y=stars[i].y*m+(s->y)*(1-m);
   stars[i].s=stars[i].s*m+(s->s)*(1-m);
   stars[i].ss=sqrt(stars[i].ss*stars[i].ss*m+(s->ss)*(s->ss)*(1-m));
   stars[i].chi=stars[i].chi*m+(s->chi)*(1-m);
   stars[i].sh=stars[i].sh*m+(s->sh)*(1-m);
   stars[i].sky=stars[i].sky*m+(s->sky)*(1-m);
   if (stars[i].passID>(s->passID)) stars[i].passID=(s->passID);
   for (img=0;img<Nimg;img++) {
      stars[i].is[img]=stars[i].is[img]*m+(s->is[img])*(1-m);
      stars[i].iss[img]=stars[i].iss[img]*m+(s->iss[img])*(1-m);
      stars[i].icm[img]=stars[i].icm[img]*m+(s->icm[img])*(1-m);
      stars[i].pa[img]=stars[i].pa[img]*m+(s->pa[img])*(1-m);
      stars[i].pb[img]=stars[i].pb[img]*m+(s->pb[img])*(1-m);
      stars[i].pc[img]=stars[i].pc[img]*m+(s->pc[img])*(1-m);
   }
   if (GetParamDouble(PSFStep)<=0) setpsfpars(i);
   return;
}

// Not parallel safe, but not called in threaded code
// mark: star ID (<0 means finding stars)
// nit: photometry averaged every third iteration starting halfway
void phot(int x,int y,int mark,int nit) {
   int st,i,j,img,bcl;
   double sx,sy;
   float bs,bss,bsky,bc,bsh,m;
   stype old;
   static float oldis[MAXNIMG],oldiss[MAXNIMG],oldcm[MAXNIMG],oldpa[MAXNIMG],oldpb[MAXNIMG],oldpc[MAXNIMG],bis[MAXNIMG],biss[MAXNIMG],bcm[MAXNIMG],bpa[MAXNIMG],bpb[MAXNIMG],bpc[MAXNIMG];

   if (mark<0) {
      centroid(-1,x,y,&sx,&sy);
      if (sx<GetGlobalInt(XMIN)) sx=GetGlobalInt(XMIN);
      if (sx>=GetGlobalInt(XMAX)) sx=GetGlobalInt(XMAX)-0.01;
      if (sy<GetGlobalInt(YMIN)) sy=GetGlobalInt(YMIN);
      if (sy>=GetGlobalInt(YMAX)) sy=GetGlobalInt(YMAX)-0.01;
      if (ran[(int)(sy*GetParamInt(SubResRef))][(int)(sx*GetParamInt(SubResRef))]) return;
      bcl=1;
      bs=bss=0;
      m=photsearch(-1,&sx,&sy,bpa,bpb,bpc,&bs,&bss,&bc,&bsh,&bsky,bis,biss,bcm,&bcl,1,-1);
   }
   else {
      memcpy(&old,stars+mark,sizeof(stype));
      memcpy(oldis,stars[mark].is,Timg*FLOATSIZE);
      memcpy(oldiss,stars[mark].iss,Timg*FLOATSIZE);
      memcpy(oldcm,stars[mark].icm,Timg*FLOATSIZE);
      memcpy(oldpa,stars[mark].pa,Timg*FLOATSIZE);
      memcpy(oldpb,stars[mark].pb,Timg*FLOATSIZE);
      memcpy(oldpc,stars[mark].pc,Timg*FLOATSIZE);
      old.is=oldis;
      old.iss=oldiss;
      old.icm=oldcm;
      old.pa=oldpa;
      old.pb=oldpb;
      old.pc=oldpc;
      sx=old.x;
      sy=old.y;
      bcl=old.type;
      bs=old.s;
      bss=old.ss;
      memcpy(bpa,old.pa,Timg*FLOATSIZE);
      memcpy(bpb,old.pb,Timg*FLOATSIZE);
      memcpy(bpc,old.pc,Timg*FLOATSIZE);
      m=photsearch(-1,&sx,&sy,bpa,bpb,bpc,&bs,&bss,&bc,&bsh,&bsky,bis,biss,bcm,&bcl,0,mark);
   }
   if (sx<GetGlobalInt(XMIN) || sx>=GetGlobalInt(XMAX) || sy<GetGlobalInt(YMIN) || sy>=GetGlobalInt(YMAX) || indx[(int)(sy*GetParamInt(SubResRef))][(int)(sx*GetParamInt(SubResRef))]>=0) m=0;
   else if (mark<0 && ran[(int)(sy*GetParamInt(SubResRef))][(int)(sx*GetParamInt(SubResRef))]) m=0;
   if (m==0) bs=bss=0;
   if (bs<=0) bs=0;
   if ((bs>=bss*GetParamDouble(SigFind) && bs>0 && m>0) || mark>=0) {
      if (mark<0) {
	 if (Nstars>=MAXNSTARS) {
	    printf("****Too many stars\n");
	    exit(-1);
	 }
	 st=Nstars;
	 Nstars++;
	 stars[st].passID=-mark;
	 stars[st].type=bcl;
      }
      else {
	 st=mark;
	 //if (mark>=0 && (fabs(bx-stars[mark].x0)>dPosMax || fabs(by-stars[mark].y0)>dPosMax)) bs=0;
	 if (m==0 || bs<=0) {
	    if (old.s!=0) markstars((int)old.x,(int)old.y);
	 }
	 else if (fabs(sx-old.x)>1.5*GetParamDouble(PosStep) || fabs(sy-old.y)>1.5*GetParamDouble(PosStep)) {
	    markstars((int)sx,(int)sy);
	 }
	 else for (img=0,j=1;img<Nimg && j;img++) if (fabs(bis[img]-oldis[img])>1 && fabs(bis[img]-oldis[img])>(bis[img]+oldis[img])*0.0005) {
	    markstars((int)sx,(int)sy);
	    j=0;
	 }
      }
      stars[st].flag=0;
      if (m==0 || bs<=0 || indx[(int)(sy*GetParamInt(SubResRef))][(int)(sx*GetParamInt(SubResRef))]>=0) {
	 stars[st].x=sx;
	 stars[st].y=sy;
	 stars[st].s=0;
	 stars[st].ss=1000.;
	 stars[st].sky=0;
	 stars[st].chi=999.99;
	 stars[st].sh=0;
	 for (i=0;i<Nimg;i++) stars[st].is[i]=0;
	 for (i=0;i<Nimg;i++) {
	    stars[st].pa[i]=bpa[i];
	    stars[st].pb[i]=bpb[i];
	    stars[st].pc[i]=bpc[i];
	 }
      }
      else {
	 stars[st].x=sx;
	 stars[st].y=sy;
	 stars[st].s=bs;
	 stars[st].ss=bss;
	 stars[st].sky=bsky;
	 stars[st].chi=bc;
	 stars[st].sh=bsh;
	 stars[st].type=bcl;
	 for (i=0;i<Nimg;i++) {
	    stars[st].is[i]=bis[i];
	    stars[st].iss[i]=biss[i];
	    stars[st].icm[i]=bcm[i];
	    stars[st].pa[i]=bpa[i];
	    stars[st].pb[i]=bpb[i];
	    stars[st].pc[i]=bpc[i];
	 }
	 if (nit>GetParamInt(MaxIT)/2 && nit%3==GetParamInt(MaxIT)%3) {
	    avgstar(st,&old);
	    if (stars[st].type<3 && !GetGlobalInt(fitpsf)) getpsfpars(-1,stars[st].x,stars[st].y,stars[st].pa,stars[st].pb,stars[st].pc);
	 }
	 indx[(int)(stars[st].y*GetParamInt(SubResRef))][(int)(stars[st].x*GetParamInt(SubResRef))]=st;
	 if (mark<0) for (j=-(int)GetParamDouble(RCombine);j<=(int)GetParamDouble(RCombine);j++) for (i=-(int)GetParamDouble(RCombine);i<=(int)GetParamDouble(RCombine);i++) if ((int)(sy*GetParamInt(SubResRef))+j>=GetGlobalInt(YMIN)*GetParamInt(SubResRef) && (int)(sy*GetParamInt(SubResRef))+j<GetGlobalInt(YMAX)*GetParamInt(SubResRef) && (int)(sx*GetParamInt(SubResRef))+i>=GetGlobalInt(XMIN)*GetParamInt(SubResRef) && (int)(sx*GetParamInt(SubResRef))+i<GetGlobalInt(XMAX)*GetParamInt(SubResRef) && i*i+j*j<(GetParamDouble(RCombine)-0.5)*(GetParamDouble(RCombine)-0.5)) ran[(int)(sy*GetParamInt(SubResRef))+j][(int)(sx*GetParamInt(SubResRef))+i]=1;
      }
   }
   return;
}

void imgadd1(int img,int i) {
   int x1,y1,ix,iy;
   double x,y;

   if (stars[i].is[img]<=0) return;
   shift(img,stars[i].x,stars[i].y,&x,&y,1);
   ix=(int)x; iy=(int)y;
   calc1psf(img,x,y,RPSF[img],stars[i].pa[img],stars[i].pb[img],stars[i].pc[img],stars[i].type,0);

   for (y1=iy-RPSF[img];y1<=iy+RPSF[img];y1++) for (x1=ix-RPSF[img];x1<=ix+RPSF[img];x1++) if (ppixOK(img,x1,y1)) res[img][y1][x1]+=stars[i].is[img]/stars[i].icm[img]*psf[img][y1-iy][x1-ix];
   return;
}

void imgadd(int i) {
   int img;
   for (img=0;img<Nimg;img++) imgadd1(img,i);
   return;
}

void imgaddall(int img) {
   int i;
   for (i=0;i<Nstars;i++) imgadd1(img,i);
}

void imgsub1(int img,int i) {
   int x1,y1,ix,iy;
   double x,y;

   if (stars[i].is[img]<=0) return;
   shift(img,stars[i].x,stars[i].y,&x,&y,1);
   ix=(int)x; iy=(int)y;
   calc1psf(img,x,y,RPSF[img],stars[i].pa[img],stars[i].pb[img],stars[i].pc[img],stars[i].type,0);
   for (y1=iy-RPSF[img];y1<=iy+RPSF[img];y1++) for (x1=ix-RPSF[img];x1<=ix+RPSF[img];x1++) if (ppixOK(img,x1,y1)) res[img][y1][x1]-=stars[i].is[img]/stars[i].icm[img]*psf[img][y1-iy][x1-ix];
   return;
}

void imgsub(int i) {
   int img;
   for (img=0;img<Nimg;img++) imgsub1(img,i);
   return;
}

void imgsuball(int img) {
   int i;
   for (i=0;i<Nstars;i++) imgsub1(img,i);
}

//Quicksort from NR; 0-index by AED
#define SWAP(a,b) {memcpy(&temp,a,ssize);memcpy(a,b,ssize);memcpy(b,&temp,ssize);}
#define M 7
#define NSTACK 50

void sortstars() {
   int i,ir,j,k,l=0;
   int jstack=0,*istack,ssize;
   stype temp,a;

   if (GetGlobalInt(WARMSTART)) return;
   ir=Nstars-1;
   ssize=sizeof(stype);
   if ((istack=(int*)calloc(NSTACK,INTSIZE))==NULL) merr();
   for (;;) {
      if (ir-l<M) {
	 for (j=l+1;j<=ir;j++) {
	    memcpy(&a,stars+j,ssize);
	    for (i=j-1;i>=l;i--) {
	       if (stars[i].s>=a.s) break;
	       memcpy(stars+i+1,stars+i,ssize);
	    }
	    memcpy(stars+i+1,&a,ssize);
	 }
	 if (jstack==0) break;
	 ir=istack[jstack--];
	 l=istack[jstack--];
      } else {
	 k=(l+ir)>>1;
	 SWAP(stars+k,stars+l+1)
	 if (stars[l+1].s<stars[ir].s) SWAP(stars+l+1,stars+ir)
	 if (stars[l].s<stars[ir].s) SWAP(stars+l,stars+ir)
	 if (stars[l+1].s<stars[l].s) SWAP(stars+l+1,stars+l)
	 i=l+1;
	 j=ir;
	 memcpy(&a,stars+l,ssize);
	 for (;;) {
	    do i++; while (stars[i].s>a.s);
	    do j--; while (stars[j].s<a.s);
	    if (j<i) break;
	    SWAP(stars+i,stars+j);
	 }
	 memcpy(stars+l,stars+j,ssize);
	 memcpy(stars+j,&a,ssize);
	 jstack+=2;
	 if (jstack > NSTACK) {
	    printf("****NSTACK too small in sortstars.");
	    exit(-1);
	 }
	 if (ir-i+1>=j-l) {
	    istack[jstack]=ir;
	    istack[jstack-1]=i;
	    ir=j-1;
	 } else {
	    istack[jstack]=j-1;
	    istack[jstack-1]=l;
	    l=i;
	 }
      }
   }
   free(istack);
   return;
}
#undef M
#undef NSTACK
#undef SWAP

int igoodstar(int img,int i,int redge,int badmult) {
   int x,y,ix,iy,ct=0,tpix=0;
   double tx,ty;

   shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
   if (!posOK(img,tx,ty)) return 0;
   if (star_flag(stars[i].x,stars[i].y,img)) return 0;
   if (stars[i].is[img]<=0 || stars[i].is[img]<=5*stars[i].iss[img]) return 0;
   ix=(int)tx; iy=(int)ty;
   for (x=-redge;x<=redge;x++) for (y=-redge;y<=redge;y++) if (posOK(img,ix+x,iy+y)) {
      tpix++;
      if (!datafOK(img,ix+x,iy+y)) {
	 if (abs(x)<2 && abs(y)<2) return 0;
	 ct++;
      }
   }
   if (ct*badmult>=tpix) return 0;
   return 1;
}

int goodstar(int IMG,int i,int rsep,int redge0,int strict) {
   int j,ix,iy,redge;

   if (getsclass(i)>1) return 0;
   if (strict==2) {
      if (stars[i].s<stars[i].ss*10 || stars[i].s<=0 || stars[i].sh>0.15 || stars[i].sh<-0.15 || stars[i].chi>2.5) return 0;
   }
   else if (strict==1) {
      if (stars[i].s<stars[i].ss*10 || stars[i].s<=0 || stars[i].sh>0.25 || stars[i].sh<-0.25 || stars[i].chi>15.) return 0;
   }
   else {
      if (stars[i].s<stars[i].ss*5 || stars[i].s<=0) return 0;
   }
   ix=(int)stars[i].x; iy=(int)stars[i].y;
   for (j=0;j<i;j++) if (abs(ix-(int)stars[j].x)<=rsep && abs(iy-(int)stars[j].y)<=rsep) return 0;
   redge=redge0;
   if (redge<0) redge=GetGlobalInt(rpsfMax);
   for (j=i+1;j<Nstars && stars[j].s*2>=stars[i].s;j++) if (abs(ix-(int)stars[j].x)<=redge && abs(iy-(int)stars[j].y)<=redge) return 0;
   if (ix<GetParamInt(XMIN2) || ix-redge<GetGlobalInt(XMIN) || ix>=GetParamInt(XMAX2) || ix+redge>=GetGlobalInt(XMAX) || iy<GetParamInt(YMIN2) || iy-redge<GetGlobalInt(YMIN) || iy>=GetParamInt(YMAX2) || iy+redge>=GetGlobalInt(YMAX)) return 0;
   if (IMG>=0) {
      if (redge0<0) return igoodstar(IMG,i,RPSF[IMG],3);
      else return igoodstar(IMG,i,redge,3);
   }
   for (j=0;j<Nimg;j++) {
      if (redge0<0) {if (igoodstar(j,i,RPSF[j],3)) return 1;}
      else {if (igoodstar(j,i,redge,3)) return 1;}
   }
   return 0;
}

//adapted from NR by AED;
#define SWAP(a,b) {temp=(a);(a)=(b);(b)=temp;}

void gaussj(double**a,int n,double*b) {
   int *indxc,*indxr,*ipiv;
   int i,icol=0,irow=0,j,k,l,ll;
   double big,dum,pivinv,temp;

   indxc=(int*)calloc(n,INTSIZE);
   indxr=(int*)calloc(n,INTSIZE);
   ipiv=(int*)calloc(n,INTSIZE);
   if (!indxc || !indxr || !ipiv) merr();
   for (i=0;i<n;i++) {
      big=0.0;
      for (j=0;j<n;j++) if (ipiv[j]!=1) for (k=0;k<n;k++) {
	 if (!ipiv[k]) {
	    if (fabs(a[j][k])>=big) {
	       big=fabs(a[j][k]);
	       irow=j;
	       icol=k;
	    }
	 }
	 else if (ipiv[k]>1) {
	    printf("gaussj: Singular Matrix!\n");
	    exit(-1);
	 }
      }
      ipiv[icol]++;
      if (irow!=icol) {
	 for (l=0;l<n;l++) SWAP(a[irow][l],a[icol][l])
	 SWAP(b[irow],b[icol])
      }
      indxr[i]=irow;
      indxc[i]=icol;
      if (a[icol][icol]==0.) {
	 printf("gaussj: Singular Matrix!\n");
	 exit(-1);
      }
      pivinv=1./a[icol][icol];
      a[icol][icol]=1.;
      for (l=0;l<n;l++) a[icol][l]*=pivinv;
      b[icol]*=pivinv;
      for (ll=0;ll<n;ll++) if (ll!=icol) {
	 dum=a[ll][icol];
	 a[ll][icol]=0.0;
	 for (l=0;l<n;l++) a[ll][l]-=a[icol][l]*dum;
	 b[ll]-=b[icol]*dum;
      }
   }
   for (l=n-1;l>=0;l--) if (indxr[l]!=indxc[l]) for (k=0;k<n;k++) SWAP(a[k][indxr[l]],a[k][indxc[l]])
   free(ipiv);
   free(indxr);
   free(indxc);
   return;
}
#undef SWAP

typedef double ltype[5];
//adapted from NR by AED;
void lfit(ltype*data,int N,double*vec,int NV) {
   int i,j,k;
   double wt,*afunc,**covar;

   afunc=(double*)calloc(NV,DOUBLESIZE);
   covar=(double**)calloc(NV,PTRSIZE);
   if (!afunc || !covar) merr();
   for (i=0;i<NV;i++) {
      covar[i]=(double*)calloc(NV,DOUBLESIZE);
      if (!covar[i]) merr();
      for (j=0;j<NV;j++) covar[i][j]=0.;
      vec[i]=0.;
   }
   for (i=0;i<N;i++) {
      afunc[0]=1.;
      afunc[1]=data[i][2];
      afunc[2]=data[i][3];
      if (NV>3) {
	 afunc[3]=data[i][2]*data[i][2];
	 afunc[4]=data[i][3]*data[i][3];
	 afunc[5]=data[i][2]*data[i][3];
      }
      for (j=0;j<NV;j++) {
	 wt=afunc[j]*data[i][1];
	 for (k=0;k<=j;k++) covar[j][k]+=wt*afunc[k];
	 vec[j]+=data[i][0]*wt;
      }
   }
   for (i=1;i<NV;i++) for (j=0;j<i;j++) covar[j][i]=covar[i][j];
   gaussj(covar,NV,vec);
   free(afunc);
   for (i=0;i<NV;i++) free(covar[i]);
   free(covar);
   return;
}

void solvepsf(int Np,int**plist) {
   int img,i,N,xx,v;
   ltype *list;
   double sd;

   list=(ltype*)calloc(Np,sizeof(ltype));
   if (!list) merr();
   fprintf(finfo,"PSF solution\n");
   fflush(finfo);
   for (img=0;img<Nimg;img++) {
      char verbstr[321],*verbptr=verbstr;
      printf("Image %d Solved PSF:\n",img+1);
      if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0) {
	 verbptr += sprintf(verbptr,"PSF image %d:",img+1);
      }
      for (v=0;v<3;v++) {
	 N=0;
	 for (i=0;i<Np;i++) if (plist[i][img+1]) if (stars[plist[i][0]].s>0 && stars[plist[i][0]].s>stars[plist[i][0]].ss*GetParamDouble(SigPSF)) {
	    list[N][1]=stars[plist[i][0]].s/stars[plist[i][0]].ss;
	    list[N][2]=stars[plist[i][0]].x-GetGlobalInt(X)*0.5;
	    list[N][3]=stars[plist[i][0]].y-GetGlobalInt(Y)*0.5;
	    if (v==0) list[N++][0]=stars[plist[i][0]].pa[img];
	    else if (v==1) list[N++][0]=stars[plist[i][0]].pb[img];
	    else list[N++][0]=stars[plist[i][0]].pc[img];
	 }
	 if (N) {
	    xx=1;
	    while (xx) {
	       double z=0;
	       xx=0;
	       for (i=0;i<6;i++) apsf[img][v][i]=0;
	       if (GetParamInt(PSFsol)==0) {
		  for (i=0;i<N;i++) {
		     apsf[img][v][0]+=list[i][0]*list[i][1];
		     z+=list[i][1];
		  }
		  if (z<=0) return;
		  apsf[img][v][0]/=z;
	       }
	       else if (GetParamInt(PSFsol)==1) lfit(list,N,apsf[img][v],3);
	       else lfit(list,N,apsf[img][v],6);
	       sd=z=0;
	       for (i=0;i<N;i++) {
		  list[i][4]=apsf[img][v][0]+list[i][2]*apsf[img][v][1]+list[i][3]*apsf[img][v][2]+list[i][2]*list[i][2]*apsf[img][v][3]+list[i][3]*list[i][3]*apsf[img][v][4]+list[i][2]*list[i][3]*apsf[img][v][5];
		  sd+=(list[i][0]-list[i][4])*(list[i][0]-list[i][4])*list[i][1]*list[i][1];
		  z+=list[i][1]*list[i][1];
	       }
	       sd=sqrt(sd/z)*2;
	       for (i=0;i<N;)
		  if (fabs(list[i][0]-list[i][4])>sd) {
		     memcpy(list[i],list[--N],sizeof(ltype));
		     xx=1;
		  }
		  else i++;
	    }
	 }
	 printf("  %c= %5.3f + %5.3fX + %5.3fY + %5.3fX^2 + %5.3fY^2 + %5.3fXY\n",97+v,apsf[img][v][0],apsf[img][v][1]*GetGlobalInt(X),apsf[img][v][2]*GetGlobalInt(Y),apsf[img][v][3]*GetGlobalInt(X)*GetGlobalInt(X),apsf[img][v][4]*GetGlobalInt(Y)*GetGlobalInt(Y),apsf[img][v][5]*GetGlobalInt(X)*GetGlobalInt(Y));
	 fprintf(finfo," %f %f %f %f %f %f\n",apsf[img][v][0],apsf[img][v][1],apsf[img][v][2],apsf[img][v][3],apsf[img][v][4],apsf[img][v][5]);
	 if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0) verbptr += sprintf(verbptr," %f %f %f %f %f %f",apsf[img][v][0],apsf[img][v][1]*GetGlobalInt(X),apsf[img][v][2]*GetGlobalInt(Y),apsf[img][v][3]*GetGlobalInt(X)*GetGlobalInt(X),apsf[img][v][4]*GetGlobalInt(Y)*GetGlobalInt(Y),apsf[img][v][5]*GetGlobalInt(X)*GetGlobalInt(Y));
      }
      printf("X = x/%d-0.5; Y = y/%d-0.5\n",GetGlobalInt(X),GetGlobalInt(Y));
      fflush(stdout);
      if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0) {
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	 {
	    fprintf(fverb,"%s\n",verbptr);
	    fflush(fverb);
	 }
      }
   }
   for (i=0;i<Np;i++) {
      imgadd(plist[i][0]);
      if (stars[plist[i][0]].s>0) indx[(int)(stars[plist[i][0]].y*GetParamInt(SubResRef))][(int)(stars[plist[i][0]].x*GetParamInt(SubResRef))]=-1;
      phot(stars[plist[i][0]].x,stars[plist[i][0]].y,plist[i][0],0);
      imgsub(plist[i][0]);
   }
   free(list);
   return;
}

void getimgstr(int img,char*imgstr) {
#ifdef USEWFPC2
   if (hstmode[img].inst==WFPC2) {
      sprintf(imgstr,"%s (WFPC2_%s, %3.1f sec)",base[img],WFPC2imagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USEACS
   if (hstmode[img].inst==ACS) {
      sprintf(imgstr,"%s (ACS_%s, %3.1f sec)",base[img],ACSimagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USEWFC3
   if (hstmode[img].inst==WFC3) {
      sprintf(imgstr,"%s (WFC3_%s, %3.1f sec)",base[img],WFC3imagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USEROMAN
   if (hstmode[img].inst==ROMAN) {
      sprintf(imgstr,"%s (ROMAN_%s, %3.1f sec)",base[img],ROMANimagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USENIRCAM
   if (hstmode[img].inst==NIRCAM) {
      sprintf(imgstr,"%s (NIRCAM_%s, %3.1f sec)",base[img],NIRCAMimagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USENIRISS
   if (hstmode[img].inst==NIRISS) {
      sprintf(imgstr,"%s (NIRISS_%s, %3.1f sec)",base[img],NIRISSimagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USEMIRI
   if (hstmode[img].inst==MIRI) {
      sprintf(imgstr,"%s (MIRI_%s, %3.1f sec)",base[img],MIRIimagestring(img),iEXP[img]);
   }
   else
#endif
#ifdef USEEUCLID
   if (hstmode[img].inst==NISP) {
      sprintf(imgstr,"%s (NISP_%s, %3.1f sec)",base[img],NISPimagestring(img),iEXP[img]);
   }
   else
   if (hstmode[img].inst==VIS) {
      sprintf(imgstr,"%s (VIS_%s, %3.1f sec)",base[img],VISimagestring(img),iEXP[img]);
   }
   else
#endif
      sprintf(imgstr,"%s (%3.1f sec)",base[img],iEXP[img]);
}

/*
  Theory behind PSF residual calculation

  1. ssky[i] = background value for each star
  - FitSky=1 or FitSky=2, calculated from getsky_norm on subtracted image
  - FitSky=3 or FitSky=4, calculated from eval on unsubtracted image
  - Either way, best estimate of true background

  2. avpsf[y][x] = mean PSF of PSF stars with adequate SNR in this chip

  3. Loop over pixels in y and x (out to RPSF)
  - sig[i] populated with star brightness / pixel noise
  * LOOKS LIKE ERROR; sig[] should be brightness / pixel noise^2
  - list[i] populated with (pixel residual-sky) / star brightness
  - solve for "deltaPSF" assuming 1/variance on "list" goes as sig*(avpsf+deltaPSF) and putting prior on current "poff". Quadratic solution reduces to
    deltaPSF = 0.5*(av-z*avpsf-prior+sqrt(z*z*avpsf*avpsf+prior*prior+av*av+2*prior*z*avpsf+2*z*avpsf*av-2*prior*av-4*prior*z*poff))/z;
  - iterate on "deltaPSF" with 3.5 sigma clipping
  - off[y][x] = deltaPSF
  - toff += avpsf[y][x]  // total of old PSF
  - toffd += avpsf[y][x] + off[y][x]  // total of new PSF

  4. toff/=toffd  // sum(old PSF) / sum(new PSF)

  5. Fix PSFs
  - add all stars back (residual should match data)
  - Set new PSF = (off+avpsf), normalized to keep same sum
    poff[y][x] += toff*off[y][x] + (toff-1)*avpsf[y][x]
  - subtract all stars (forcing poffreset=1)
 */
//#define FIXPSF_DEBUG 0
//#define FIXPSF_DEBUG_VERB
int fix1psf(int img,int Np,int**plist,int it) {
   int x,y,xx,yy,i,j,N;
   float *list,*sig,av,sd,z,**off,**avpsf,*ssky,toff=0,toffd=0,d;
   float is,mx,my;
   double tx,ty;

   ssky=(float*)calloc(Np,FLOATSIZE);
   list=(float*)calloc(Np,FLOATSIZE);
   sig=(float*)calloc(Np,FLOATSIZE);
   off=(float**)calloc(2*RPSF[img]+1,PTRSIZE);
   avpsf=(float**)calloc(2*RPSF[img]+1,PTRSIZE);
   if (!ssky || !list || !sig || !off || !avpsf) merr();
   off+=RPSF[img];
   avpsf+=RPSF[img];
   for (y=-RPSF[img];y<=RPSF[img];y++) {
      off[y]=(float*)calloc(2*RPSF[img]+1,FLOATSIZE);
      avpsf[y]=(float*)calloc(2*RPSF[img]+1,FLOATSIZE);
      if (!off[y] || !avpsf[y]) merr();
      off[y]+=RPSF[img];
      avpsf[y]+=RPSF[img];
      for (x=-RPSF[img];x<=RPSF[img];x++) avpsf[y][x]=0;
   }
   N=0;
#if defined(FIXPSF_DEBUG)
   if (img==FIXPSF_DEBUG) printf("**IMG %d; N=%d\n",FIXPSF_DEBUG,N);
#endif
   for (i=0;i<Np;i++) if (plist[i][img+1]) {
      j=plist[i][0];
      shift(img,stars[j].x,stars[j].y,&tx,&ty,1);
      clear_sky();
      if (GetParamInt(FitSky)==0 && fsky[img].lastoffset>=0) ssky[i]=skyval(img,(int)tx,(int)ty);
      else if (GetParamInt(FitSky)==3 || GetParamInt(FitSky)==4) {
	 // Sky is calculated with star added for these modes
	 int ix,iy,cx,cy;
	 imgadd1(img,j);
	 ix=(int)(tx+100)-100;
	 iy=(int)(ty+100)-100;
	 cx=(int)((tx-ix)*50+0.5);
	 if (cx<0) cx=0; if (cx>50) cx=50;
	 cy=(int)((ty-iy)*50+0.5);
	 if (cy<0) cy=0; if (cy>50) cy=50;
	 if (GetParamInt(FitSky)==3) ssky[i]=eval1_sky3(img,ix,iy,cx,cy);
	 else ssky[i]=eval1_sky4(img,ix,iy,cx,cy,&mx,&my);
	 imgsub1(img,j);
      }
      else {
	 // Sky is calculated with star subtracted
	 ssky[i]=getsky_norm(img,(int)tx,(int)ty,NULL,0);
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
	 if (img==FIXPSF_DEBUG) printf("    %d %g %g %g %d\n",i,tx,ty,ssky[i],GetParamInt(NegSky));
#endif
	 if (ssky[i]<0 && !GetParamInt(NegSky)) ssky[i]=0.0;
      }
      // AVPSF will be average PSF (no sky subtraction if FitSky==2)
      if ((is=stars[j].is[img]/stars[j].icm[img])>0 && stars[j].s>0 && stars[j].s>=stars[j].ss*GetParamDouble(SigFind)) {
	 calc1psf(img,tx,ty,RPSF[img],stars[j].pa[img],stars[j].pb[img],stars[j].pc[img],stars[j].type,0);
	 for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) avpsf[y][x]+=psf[img][y][x];
	 N++;
      }
   }
   if (N==0) return 0;
   for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) avpsf[y][x]/=N;
   for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) {
      N=0;
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
      if (img==FIXPSF_DEBUG) printf("  X,Y = %d,%d\n",x,y);
#endif
      for (i=0;i<Np;i++) if (plist[i][img+1]) {
	 int ii;
	 ii=plist[i][0];
	 if ((is=stars[ii].is[img]/stars[ii].icm[img])>0 && stars[ii].s>0 && stars[ii].s>=stars[ii].ss*GetParamDouble(SigFind)) {
	    double tx,ty;
	    shift(img,stars[ii].x,stars[ii].y,&tx,&ty,1);
	    xx=x+(int)tx;
	    yy=y+(int)ty;
	    // list[] is "residual - sky" divided by star brightness
	    // sig[] is star brightness divided by pixel noise estimate
	    if (posOK(img,xx,yy)) {
	       if (datafOK(img,xx,yy)) {
		  sig[N]=is/sqrt(noise(img,xx,yy,ssky[i]));
		  if (GetParamChar(psfstars)==0 && ssky[i]/iEXP[img]>0) sig[N]/=1+ssky[i]/iEXP[img]*100.;
		  list[N++]=(res[img][yy][xx]-ssky[i])/is;
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
		  if (img==FIXPSF_DEBUG) printf("    %g %g good (%g %g %g %g %g)\n",sig[N-1],list[N-1],is,res[img][yy][xx],noise(img,xx,yy,ssky[i]),ssky[i],iEXP[img]);
#endif
	       }
	       else {
		  float ns=0,rs=0;
		  int ct=0,dx,dy;
		  for (dy=-1;dy<=1;dy++) for (dx=-1;dx<=1;dx++) if (ppixfOK(img,xx+dx,yy+dy)) {
		     ns+=noise(img,xx+dx,yy+dy,ssky[i]);
		     rs+=res[img][yy+dy][xx+dx];
		     ct++;
		  }
		  if (ct) {
		     sig[N]=is/sqrt(ns/ct);
		     if (GetParamChar(psfstars)==0 && ssky[i]/iEXP[img]>0) sig[N]/=1+ssky[i]/iEXP[img]*100.;
		     list[N++]=(rs/ct-ssky[i])/is;
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
		     if (img==FIXPSF_DEBUG) printf("    %g %g bad (%g %d %g %g %g %g)\n",sig[N-1],list[N-1],is,ct,rs,ns,ssky[i],iEXP[img]);
#endif
		  }
	       }
	    }
	 }
      }
      av=sd=0.;
      xx=1;
      while (xx) {
	 xx=0;
	 av=sd=z=0.;
	 for (i=0;i<N;i++) {
	    av+=list[i]*sig[i];
	    z+=sig[i];
	 }
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
	 if (img==FIXPSF_DEBUG) printf("  totals %g %g %g %g\n",av,z,avpsf[y][x],poff[img][y][x]);
#endif
	 if (z>0) {
	    //delta = [ sum(list*sig*(avpsf+delta)) - prior*poff ] / [ sum(sig*(avpsf+delta)) + prior ];
	    //delta=(av*(avpsf+delta)-prior*poff)/(z*(avpsf+delta)+prior);
	    //delta*delta*z+delta*(z*avpsf+prior-av)+prior*poff-av*avpsf
	    //delta = 0.5*(av-z*avpsf-prior+sqrt(z*z*avpsf*avpsf+prior*prior+av*av+2*prior*z*avpsf+2*z*avpsf*av-2*prior*av-4*prior*z*poff))/z;
	    d = z*z*avpsf[y][x]*avpsf[y][x]+100.+av*av+20.*z*avpsf[y][x]+2.*z*avpsf[y][x]*av-20.*av-40.*z*poff[img][y][x];
	    if (d<0.) d=0.;
	    av=0.5*(av-z*avpsf[y][x]-10.+sqrt(d))/z;
	    for (i=0;i<N;i++) sd+=(list[i]-av)*(list[i]-av)*sig[i];
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
	    if (img==FIXPSF_DEBUG) printf("  av=%g, sig=%g",av,sd);
#endif
	    sd=sqrt(sd/z)*3.5;
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
	    if (img==FIXPSF_DEBUG) printf(", 3.5sig=%g\n",sd);
#endif
	    for (i=0;i<N;) {
	       if (fabs(list[i]-av)>sd) {
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
		  if (img==FIXPSF_DEBUG) printf("    deleting %g %g\n",list[i],sig[i]);
#endif
		  list[i]=list[--N];
		  sig[i]=sig[N];
		  xx=1;
	       }
	       else i++;
	    }
	 }
	 else av=-poff[img][y][x];
      }
      off[y][x]=av;
      toff+=avpsf[y][x];
      toffd+=avpsf[y][x]+off[y][x];
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
      if (img==FIXPSF_DEBUG) printf("  final %g %g (%g %g)\n",av,avpsf[y][x],toff,toffd);
#endif
   }
   toff/=toffd;
#if defined(FIXPSF_DEBUG) && !defined(FIXPSF_DEBUG_VERB)
   if (img==FIXPSF_DEBUG) {
      printf("Previous AVPSF\n");
      for (y=RPSF[img];y>=-RPSF[img];y--) {
	 for (x=-RPSF[img];x<=RPSF[img];x++) printf("%8.4f",avpsf[y][x]);
	 printf("\n");
      }
      printf("Average Residual\n");
      for (y=RPSF[img];y>=-RPSF[img];y--) {
	 for (x=-RPSF[img];x<=RPSF[img];x++) printf("%8.4f",off[y][x]);
	 printf("\n");
      }
      printf("Mean AVPSF / Mean (AVPSF+Residual) = %f\n",toff);
      fflush(stdout);
   }
#endif
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
   if (img==FIXPSF_DEBUG) printf("toff = %g\n",toff);
#endif
   //for (i=0;i<Nstars;i++) imgadd1(img,i);
   imgaddall(img);
#if defined(FIXPSF_DEBUG)
   if (img==FIXPSF_DEBUG) for (y=0;y<dataim[img].Y;y++) for (x=0;x<dataim[img].X;x++) if (dataOK(img,x,y) && fabs(data[img][y][x]-res[img][y][x])>1.e-5 && fabs(data[img][y][x]-res[img][y][x])>(1.e-5)*data[img][y][x]) printf("ERROR: data[%d][%d][%d]=%f; res[%d][%d][%d]=%f\n",img,y,x,data[img][y][x],img,y,x,res[img][y][x]);
#endif
   for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) {
      off[y][x]=toff*off[y][x]+(toff-1)*avpsf[y][x];
      if (it>4) off[y][x]/=1+it*0.1;
      poff[img][y][x]+=off[y][x];
#if defined(FIXPSF_DEBUG) && defined(FIXPSF_DEBUG_VERB)
      if (img==FIXPSF_DEBUG) printf("%d,%d: off = %g (%g %g %g)\n",x,y,poff[img][y][x],off[y][x],toff,avpsf[y][x]);
      if (img==FIXPSF_DEBUG) exit(0);
#endif
   }
   poffreset[img]=1;
   //for (i=0;i<Nstars;i++) imgsub1(img,i);
   imgsuball(img);
   poffreset[img]=0;
   free(ssky);
   free(list);
   free(sig);
   i=0;
   for (y=-RPSF[img];y<=RPSF[img] && !i;y++) for (x=-RPSF[img];x<=RPSF[img] && !i;x++) if (fabs(off[y][x])>=0.001+0.0001*it) i=1;
   for (y=-RPSF[img];y<=RPSF[img];y++) {
      free(off[y]-RPSF[img]);
      free(avpsf[y]-RPSF[img]);
   }
   free(off-RPSF[img]);
   free(avpsf-RPSF[img]);
   return i;
}

int fixpsf(int Np,int**plist,int it) {
   int img,rval=0;

#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 if (fix1psf(img,Np,plist,it)) {
	    rval=1;
	 }
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 if (fix1psf(img,Np,plist,it)) {
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    rval=1;
	 }
      }
   return rval;
}
#undef FIXPSF_DEBUG
#undef FIXPSF_DEBUG_VERB

void calcpsf(int ext,int fld) {
#define minNpsf 50
#define goodNpsf 150
#define maxNpsf 300
   int **plist,*tlist;
   int Np=0,Nt=0,i,pe,pc,bi,j,it=0,img,cont=1;
   double px,py,r,br;
   FILE *f;
   char str[161],*ptr,*ptr2;

   if (GetParamInt(PSFsol)<0 && !GetParamInt(PSFres)) return;
   plist=(int**)calloc(Nstars,PTRSIZE);
   tlist=(int*)calloc(Nstars,INTSIZE);
   if (!plist || !tlist) merr();
   for (i=0;i<Nstars;i++) {
      plist[i]=(int*)calloc(Nimg+1,INTSIZE);
      if (!plist[i]) merr();
   }
   sortstars();
   setindx(-1);
   for (i=0;i<Nstars;i++) {
      indx[(int)(stars[i].y*GetParamInt(SubResRef))][(int)(stars[i].x*GetParamInt(SubResRef))]=i;
      stars[i].flag=0;
   }
   if (GetParamChar(psfstars)!=0) {
      if ((f=fopen(GetParamString(psfstars),"r"))==NULL) {
	 printf("**%s not found\n",GetParamString(psfstars));
	 fflush(stdout);
      }
      else {
	 while (fgets(str,161,f)) {
	    pe=strtol(str,&ptr,10);
	    pc=strtol(ptr,&ptr,10);
	    px=strtod(ptr,&ptr)-GetParamDouble(psfoff);
	    py=strtod(ptr,&ptr2)-GetParamDouble(psfoff);
	    if (pe==ext && pc==fld+1 && ptr!=ptr2) {
	       bi=-1;
	       br=GetParamDouble(RCombine)*GetParamDouble(RCombine)/(GetParamInt(SubResRef)*GetParamInt(SubResRef)); // units of physical pixels
	       if (br<1) br=1;
	       for (i=0;i<Nstars;i++) if ((r=(px-stars[i].x)*(px-stars[i].x)+(py-stars[i].y)*(py-stars[i].y))<br && goodstar(-1,i,GetGlobalInt(rPhotPlusPSFmax),-1,0)) {
		  br=r;
		  bi=i;
	       }
	       if (bi>=0) { // star found
		  for (j=0;j<Np && bi!=plist[j][0];j++);
		  if (j==Np) { // star not already in list
		     for (j=Np;j>0 && bi<plist[j-1][0];j--) plist[j][0]=plist[j-1][0];
		     plist[j][0]=bi;
		     Np++;
		  }
	       }
	    }
	 }
	 fclose(f);
	 int NpMin = 0;
	 int *NpImg = (int*)calloc(Nimg,sizeof(int)); if (NpImg==0) merr();
	 for (i=0;i<Np;i++) if (stars[plist[i][0]].s>=stars[plist[i][0]].ss*50 || (stars[plist[i][0]].s>=stars[plist[i][0]].ss*10 && NpMin<goodNpsf) || NpMin<minNpsf) {
	    bi = plist[i][0];
	    fprintf(fpsfs,"%d %d %7.2f %7.2f",ext,fld+1,stars[bi].x,stars[bi].y);
	    for (j=0;j<Nimg;j++) if (igoodstar(j,bi,RPSF[j],5)) {
	       plist[i][j+1]=1;
	       fprintf(fpsfs," 1");
	       NpImg[j]++;
	    }
	    else {
	       plist[i][j+1]=0;
	       fprintf(fpsfs," 0");
	    }
	    if (j==0) NpMin = NpImg[j];
	    else if (NpImg[j]<NpMin) NpMin=NpImg[j];
	    fprintf(fpsfs,"\n");
	    markstars((int)stars[bi].x,(int)stars[bi].y);
	 }
	 Np=i;
	 free(NpImg);
      }
   }
   else {
      int NpMin = 0;
      int *NpImg = (int*)calloc(Nimg,sizeof(int)); if (NpImg==0) merr();
      for (i=0;i<Nstars;i++) if (((stars[i].s>=stars[i].ss*50 && NpMin<maxNpsf) || (stars[i].s>=stars[i].ss*10 && NpMin<goodNpsf) || NpMin<minNpsf) && goodstar(-1,i,GetGlobalInt(rPhotPlusPSFmax),-1,1)) {
	 plist[Np][0]=i;
	 fprintf(fpsfs,"%d %d %6.2f %6.2f",ext,fld+1,stars[i].x,stars[i].y);
	 NpMin=0;
	 for (j=0;j<Nimg;j++) {
	    if (((stars[i].is[j]>=stars[i].iss[j]*50 && NpImg[j]<maxNpsf) || (stars[i].is[j]>=stars[i].iss[j]*10 && NpImg[j]<goodNpsf) || NpImg[j]<minNpsf) && igoodstar(j,i,RPSF[j],5)) {
	       plist[Np][j+1]=1;
	       fprintf(fpsfs," 1");
	       NpImg[j]++;
	    }
	    else {
	       plist[Np][j+1]=0;
	       fprintf(fpsfs," 0");
	    }
	    if (j==0) NpMin = NpImg[j];
	    else if (NpImg[j]<NpMin) NpMin=NpImg[j];
	 }
	 fprintf(fpsfs,"\n");
	 Np++;
      }
      free(NpImg);
      while (Np>0 && cont) {
	 double mn=0,sd=0;

	 cont=0;
	 for (i=0;i<Np;i++) mn+=stars[plist[i][0]].sh;
	 mn/=Np;
	 for (i=0;i<Np;i++) sd+=(stars[plist[i][0]].sh-mn)*(stars[plist[i][0]].sh-mn);
	 sd=2.00*sqrt(sd/Np);
	 for (i=0;i<Np;i++) if (fabs(stars[plist[i][0]].sh-mn)>sd) {
	    Np--;
	    if (i!=Np) memcpy(plist[i],plist[Np],INTSIZE*(Nimg+1));
	    i--;
	    cont=1;
	 }
      }
      for (i=0;i<Np;i++) markstars((int)stars[plist[i][0]].x,(int)stars[plist[i][0]].y);
   }
   fflush(fpsfs);
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 imgaddall(img);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 imgaddall(img);
      }
   SetGlobal_NOT_THREADSAFE(POSPSF,0);
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 imgsuball(img);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 imgsuball(img);
      }
   for (i=0;i<Nstars;i++) if (stars[i].flag) tlist[Nt++]=i;
   printf("%d PSF stars; %d neighbors\n",Np,Nt-Np);
   fflush(stdout);
   if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
   {
      fprintf(fverb,"PSF: %d %d\n",Np,Nt-Np);
      fflush(fverb);
   }
   if (GetParamInt(PSFsol)>=0 && Np) solvepsf(Np,plist);
   if (GetParamInt(PSFres) && Np) while (fixpsf(Np,plist,++it)) {
      for (i=0;i<Nt;i++) {
	 imgadd(tlist[i]);
	 if (stars[tlist[i]].s>0) indx[(int)(stars[tlist[i]].y*GetParamInt(SubResRef))][(int)(stars[tlist[i]].x*GetParamInt(SubResRef))]=-1;
	 phot((int)stars[tlist[i]].x,(int)stars[tlist[i]].y,tlist[i],0);
	 imgsub(tlist[i]);
      }
   }
   if (it==20) printf("Poorly converging PSF\n");
   printf("Central pixel PSF adjustments:\n");
   fflush(stdout);
   if (Np<50) {
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
      {
	 fprintf(fwarn,"Only %d stars for PSF measurement\n",Np);
	 fflush(fwarn);
      }
   }
   for (img=0;img<Nimg;img++) {
      for (i=0,j=0;i<Np;i++) if (plist[i][img+1]) j++;
      printf("image %d: %d stars, %f\n",img+1,j,poff[img][0][0]);
      fflush(stdout);
      if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
      {
	 fprintf(fverb,"PSF image %d: %d %f\n",img+1,j,poff[img][0][0]);
	 fflush(fverb);
      }
      if (j<25 && j<Np) {
	 char imgstr[161];
	 getimgstr(img,imgstr);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	 {
	    fprintf(fwarn,"Only %d stars for PSF measurement in image %d, %s\n",j,img+1,imgstr);
	    fflush(fwarn);
	 }
      }
   }
   for (i=0;i<Nstars;i++) free(plist[i]);
   free(plist);
   free(tlist);
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 imgaddall(img);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 imgaddall(img);
      }
   SetGlobal_NOT_THREADSAFE(POSPSF,1);
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 imgsuball(img);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 imgsuball(img);
      }
  fflush(stdout);
#ifdef PGPLOT
   if (GetParamChar(DiagPlotType)!=0) {
      for (img=0;img<Nimg;img++) {
	 int x,y,i=0;
	 psfDiagData[img][DIAG_EXT][DIAG_Z].N = 2*RPSF[img]+1;
	 psfDiagData[img][DIAG_EXT][DIAG_Z].mid = poff[img][0][0];
	 psfDiagData[img][DIAG_EXT][DIAG_Z].data = (float*)calloc((RPSF[img]*2+1)*(RPSF[img]*2+1),sizeof(float));
	 if (!psfDiagData[img][DIAG_EXT][DIAG_Z].data) merr();
	 for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) psfDiagData[img][DIAG_EXT][DIAG_Z].data[i++]=poff[img][y][x];
      }
   }
#endif
   return;
}

int clean(int it,int show) {
   int i,ii,j,b=0,ct=0,xx,yy,rad,x3,y3;
   float x,y,r,br,RCeff,R2Ceff;

   if (GetGlobalInt(WARMSTART)) return 0;
   //if (it*2<=GetParamInt(MaxIT)) RCeff=GetParamDouble(RCombine);
   //else RCeff=GetParamDouble(RCombine)/(1.+(it*2.-GetParamInt(MaxIT))/GetParamInt(MaxIT));
   RCeff=GetParamDouble(RCombine);
   R2Ceff=RCeff*RCeff;
   for (i=1;i<Nstars;i++) if (stars[i].x>=0) {
      rad=(int)(RCeff+0.999);
      br=R2Ceff;
      for (xx=-rad;xx<=rad;xx++) if ((x3=(int)(stars[i].x*GetParamInt(SubResRef))+xx)>=GetGlobalInt(XMIN)*GetParamInt(SubResRef) && x3<GetGlobalInt(XMAX)*GetParamInt(SubResRef)) for (yy=-rad;yy<=rad;yy++) if ((xx || yy) && (y3=(int)(stars[i].y*GetParamInt(SubResRef))+yy)>=GetGlobalInt(YMIN)*GetParamInt(SubResRef) && y3<GetGlobalInt(YMAX)*GetParamInt(SubResRef) && (j=indx[y3][x3])>=0 && j<i && (x=fabs(stars[i].x-stars[j].x)*GetParamInt(SubResRef))<=RCeff && (y=fabs(stars[i].y-stars[j].y)*GetParamInt(SubResRef))<=RCeff) {
	 r=x*x+y*y;
	 if (r<br) {
	    br=r;
	    b=j;
	 }
      }
      if (br<R2Ceff) {
	 imgadd(i);
	 imgadd(b);
	 if (stars[i].s>0) indx[(int)(stars[i].y*GetParamInt(SubResRef))][(int)(stars[i].x*GetParamInt(SubResRef))]=-1;
	 if (stars[b].s>0) indx[(int)(stars[b].y*GetParamInt(SubResRef))][(int)(stars[b].x*GetParamInt(SubResRef))]=-1;
	 markstars((int)stars[i].x,(int)stars[i].y);
	 markstars((int)stars[b].x,(int)stars[b].y);
	 avgstar(b,stars+i);
	 stars[b].s*=2;
	 stars[b].ss*=2;
	 phot((int)stars[b].x,(int)stars[b].y,b,0);
	 imgsub(b);
	 stars[i].x=-1;
      }
   }
   // AEDDEBUG testing clean() section
   for (i=1;i<Nstars;i++) if (stars[i].x>=0 && (stars[i].s<stars[i].ss*GetParamDouble(SigFind) || (GetGlobalInt(fitpsf)==0 && stars[i].s<stars[i].ss*(GetParamDouble(SigFinal)-0.15)))) {
      imgadd(i);
      if (stars[i].s>0) indx[(int)(stars[i].y*GetParamInt(SubResRef))][(int)(stars[i].x*GetParamInt(SubResRef))]=-1;
      markstars((int)stars[i].x,(int)stars[i].y);
      stars[i].x=-1;
   }
   // AEDDEBUG testing clean() section
   ii=0;
   for (i=0;i<Nstars;i++) if (stars[i].x>=0 && stars[i].s>0) {
      if (ii<i) {
	 stype temp;
	 memcpy(&temp,stars+ii,sizeof(stype));
	 memcpy(stars+ii,stars+i,sizeof(stype));
	 memcpy(stars+i,&temp,sizeof(stype));
	 indx[(int)(stars[ii].y*GetParamInt(SubResRef))][(int)(stars[ii].x*GetParamInt(SubResRef))]=ii;
      }
      ii++;
   }
   ct=Nstars-ii;
   Nstars=ii;
   if (ct && show) {
      printf("%d stars eliminated\n",ct);
      fflush(stdout);
   }
   return ct;
}

float imginterp(int img,float x,float y) {
   int ix,iy;
   double v=0.,tw=0.,w,mx,my;

   ix=(int)(x-0.5);
   iy=(int)(y-0.5);
   mx=x-0.5-ix;
   my=y-0.5-iy;
   if (ppixfOK(img,ix,iy)) {
      tw+=(w=(1-mx)*(1-my));
      v+=w*data[img][iy][ix];
   }
   if (ppixfOK(img,ix+1,iy)) {
      tw+=(w=mx*(1-my));
      v+=w*data[img][iy][ix+1];
   }
   if (ppixfOK(img,ix,iy+1)) {
      tw+=(w=(1-mx)*my);
      v+=w*data[img][iy+1][ix];
   }
   if (ppixfOK(img,ix+1,iy+1)) {
      tw+=(w=mx*my);
      v+=w*data[img][iy+1][ix+1];
   }
   if (tw<=0.) return safedown(iDMIN[img]);
   return v/tw;
}

// GLOBAL (PARALLEL-SAFE) VARIABLES - not accessed within parallel code
float**convv;
int RPSFsub;

#define DECON_MULT 0.001
float convfunc(float*c,int IGNORE_IMG_NUMBER) {
   int x,y,i,j,n;
   float vv=0.,v;

   n=2*RPSFsub+1;
   // prior to reduce ringing
   for (i=1;i<=n*n;i++) vv+=c[i]*c[i]*DECON_MULT;
   // minimize difference between convv (img PSF) and convolution of c and psf (reference PSF)
   for (j=-RPSFsub;j<=RPSFsub;j++) for (i=-RPSFsub;i<=RPSFsub;i++) {
      v=-convv[j][i];
      for (y=-RPSFsub;y<=RPSFsub;y++) if (j-y>=-RPSFsub && j-y<=RPSFsub) for (x=-RPSFsub;x<=RPSFsub;x++) if (i-x>=-RPSFsub && i-x<=RPSFsub) {
	 v+=c[1+x+RPSFsub+(y+RPSFsub)*n]*psf[Nimg][j-y][i-x];
      }
      vv+=v*v;
   }
   return vv;
}

void dconvfunc(float*c,float*g) {
   int x,y,i,j,n;
   float v,dv;

   n=2*RPSFsub+1;
   for (i=1;i<=n*n;i++) g[i]=2*c[i]*DECON_MULT;
   for (j=-RPSFsub;j<=RPSFsub;j++) for (i=-RPSFsub;i<=RPSFsub;i++) {
      v=-convv[j][i];
      for (y=-RPSFsub;y<=RPSFsub;y++) if (j-y>=-RPSFsub && j-y<=RPSFsub) for (x=-RPSFsub;x<=RPSFsub;x++) if (i-x>=-RPSFsub && i-x<=RPSFsub) v+=c[1+x+RPSFsub+(y+RPSFsub)*n]*psf[Nimg][j-y][i-x];
      dv=2*v;
      for (y=-RPSFsub;y<=RPSFsub;y++) if (j-y>=-RPSFsub && j-y<=RPSFsub) for (x=-RPSFsub;x<=RPSFsub;x++) if (i-x>=-RPSFsub && i-x<=RPSFsub) g[1+x+RPSFsub+(y+RPSFsub)*n]+=dv*psf[Nimg][j-y][i-x];
   }
   return;
}
#undef DECON_MULT

#define ITMAX 2000
#define EPS 1.0e-10
#define FREEALL free_vector(xi,1,n);free_vector(h,1,n);free_vector(g,1,n);
void frprmn(float p[], int n, float ftol, int *iter, float *fret,
	float (*func)(float [],int), void (*dfunc)(float [], float []), int img)
{
	void linmin(float p[], float xi[], int n, float *fret,
	       float (*func)(float [], int), int img);
	int j,its;
	float gg,gam,fp,dgg;
	float *g,*h,*xi;

	g=vector(1,n);
	h=vector(1,n);
	xi=vector(1,n);
	fp=(*func)(p,img);
	(*dfunc)(p,xi);
	for (j=1;j<=n;j++) {
		g[j] = -xi[j];
		xi[j]=h[j]=g[j];
	}
	for (its=1;its<=ITMAX;its++) {
		*iter=its;
		linmin(p,xi,n,fret,func,img);
		if (2.0*fabs(*fret-fp) <= ftol*(fabs(*fret)+fabs(fp)+EPS)) {
			FREEALL
			return;
		}
		fp=(*func)(p,img);
		(*dfunc)(p,xi);
		dgg=gg=0.0;
		for (j=1;j<=n;j++) {
			gg += g[j]*g[j];
			dgg += (xi[j]+g[j])*xi[j];
		}
		if (gg == 0.0) {
			FREEALL
			return;
		}
		gam=dgg/gg;
		for (j=1;j<=n;j++) {
			g[j] = -xi[j];
			xi[j]=h[j]=g[j]+gam*h[j];
		}
	}
	nrerror("Too many iterations in frprmn");
	return;
}
#undef ITMAX
#undef EPS
#undef FREEALL

float medianim(chiptype im,int img,float min,float max) {
   float*list,xx[3]={0,0,0};
   int n=0,i,x,y,b,nm,np;

   list=(float*)calloc(10000,FLOATSIZE);
   b=(dataim[img].X*dataim[img].Y)/10000;
   if (b<1) b=1;
   for (y=0;y<dataim[img].Y && n<10000;y+=b) for (x=0;x<dataim[img].X && n<10000;x+=b) if (im[y][x]>min && im[y][x]<max) {
      list[n]=im[y][x];
      if (n==0 || list[n]<xx[0]) xx[0]=list[n];
      if (n==0) xx[1]=list[n];
      if (n==0 || list[n]>xx[2]) xx[2]=list[n];
      n++;
   }
   while (1) {
      nm=np=0;
      for (i=0;i<n;i++) {
	 if (list[i]<xx[1]-0.000001) nm++;
	 else if (list[i]>xx[1]+0.000001) np++;
      }
      if (nm<=n/2 && np<=n/2) {
	 free(list);
	 return xx[1];
      }
      if (nm>np) xx[2]=xx[1];
      else xx[0]=xx[1];
      xx[1]=0.5*(xx[0]+xx[2]);
   }
   free(list);
   return 0.;
}

void subreference(int img) {
   int x,y,xx,yy,i;
   double tx,ty;
   float mlt,rbg,newMAX,newMIN,fim=0.,fdif=0.;
   float s,ss,c,ndof,sh,ssky,mx,my,cs,cm,tmp;
   double E=0.,R=0.;
   float *p;
   chiptype tmpres;
   int n;

   if (Timg==Nimg) {
      printf("External reference image required for difference photometry\n");
      fflush(stdout);
      return;
   }
   // Set RPSFsub equal to smaller of RPSF[Nimg] and RPSF[img]
   RPSFsub=RPSF[img];
   if (RPSFsub>RPSF[Nimg]) RPSFsub=RPSF[Nimg];
   // Set res[] and tmpres[] equal to reference image, shifted to match img
   tmpres=allocchip(dataim[img].X,dataim[img].Y);
   for (y=0;y<dataim[img].Y;y++) for (x=0;x<dataim[img].X;x++) {
      shift(img,x+0.5,y+0.5,&tx,&ty,-1);
      if (tx<0.5 || ty<0.5 || tx>dataim[Nimg].X-0.5 || ty>dataim[Nimg].Y-0.5 || data[Nimg][(int)ty][(int)tx]<=iDMIN[Nimg]) res[img][y][x]=tmpres[y][x]=iDMIN[Nimg]-1.;
      else if (data[Nimg][(int)ty][(int)tx]>=GetParamDouble(FSat)*iDMAX[Nimg]) res[img][y][x]=tmpres[y][x]=iDMAX[Nimg]+1.;
      else res[img][y][x]=tmpres[y][x]=imginterp(Nimg,tx,ty);
   }
   // Set convv equal to img PSF, at center of chip; fim = sum of PSF
   convv=matrix(-RPSFsub,RPSFsub,-RPSFsub,RPSFsub);
   calc1psf(img,dataim[img].X/2+0.5,dataim[img].Y/2+0.5,RPSFsub,apsf[img][0][0],apsf[img][1][0],apsf[img][2][0],1,0);
   for (y=-RPSFsub;y<=RPSFsub;y++) for (x=-RPSFsub;x<=RPSFsub;x++) {
      convv[y][x]=psf[img][y][x];
      fim+=psf[img][y][x];
   }
   // Set psf equal to reference PSF at center of chip
   calc1psf(Nimg,dataim[Nimg].X/2+0.5,dataim[Nimg].Y/2+0.5,RPSFsub,apsf[Nimg][0][0],apsf[Nimg][1][0],apsf[Nimg][2][0],1,0);
   n=2*RPSFsub+1;
   // Add PSF offset image (if defined)
   if (refpsfimg) for (y=0;y<n;y++) for (x=0;x<n;x++) psf[Nimg][y-RPSFsub][x-RPSFsub]+=refpsfimg[y][x];

   // Compute kernel that convolves with reference PSF to get img PSF
   p=vector(1,n*n);
   for (y=1;y<=n*n;y++) p[y]=0.;
   frprmn(p,n*n,1.e-3,&y,&tmp,&convfunc,&dconvfunc,Nimg);
   // fdif will equal sum of convolved (ref*kernel) PSF divided by sum of kernel
   for (y=-RPSFsub;y<=RPSFsub;y++) for (x=-RPSFsub;x<=RPSFsub;x++) {
      // ty = convolution of kernel and reference PSF
      // tx = sum of kernel
      tx=ty=0.;
      for (yy=-RPSFsub;yy<=RPSFsub;yy++) if (y+yy>=-RPSFsub && y+yy<RPSFsub) for (xx=-RPSFsub;xx<=RPSFsub;xx++) if (x+xx>=-RPSFsub && x+xx<RPSFsub) {
	 ty+=psf[Nimg][y+yy][x+xx]*p[1+RPSFsub-xx+(RPSFsub-yy)*n];
	 tx+=p[1+RPSFsub-xx+(RPSFsub-yy)*n];
      }
      if (tx) fdif+=ty/tx;
   }
   /*
   tx=0.; for (y=RPSFsub;y>=-RPSFsub;y--) for (x=-RPSFsub;x<=RPSFsub;x++) tx+=psf[y][x]; printf("Template PSF sum: %f\n",tx);
   printf("Image PSF sum: %f\nConv. ref PSF sum: %f\n",fim,fdif);
   tx=0.;
   for (y=RPSFsub;y>=-RPSFsub;y--) {
      for (x=-RPSFsub;x<=RPSFsub;x++) tx+=p[1+x+RPSFsub+(y+RPSFsub)*n];
      for (x=-RPSFsub;x<=RPSFsub;x++) printf(" %3d",(int)(10000*p[1+x+RPSFsub+(y+RPSFsub)*n]));
      printf("\n");
   }
   printf("kernel sum: %f\n",tx);
   */
   free_matrix(convv,-RPSFsub,RPSFsub,-RPSFsub,RPSFsub);
   // tmpres equals convolved shifted reference image.  Note that "res" right now is the shifted reference image
   for (y=0;y<dataim[img].Y;y++) for (x=0;x<dataim[img].X;x++) {
      if (res[img][y][x]>iDMIN[Nimg] && res[img][y][x]<iDMAX[Nimg]) {
	 tx=ty=0.;
	 for (yy=-RPSFsub;yy<=RPSFsub;yy++) if (y+yy>=0 && y+yy<dataim[img].Y) for (xx=-RPSFsub;xx<=RPSFsub;xx++) if (x+xx>=0 && x+xx<dataim[img].X && res[img][y+yy][x+xx]>iDMIN[Nimg] && res[img][y+yy][x+xx]<iDMAX[Nimg]) {
	    ty+=res[img][y+yy][x+xx]*p[1+RPSFsub-xx+(RPSFsub-yy)*n];
	    tx+=p[1+RPSFsub-xx+(RPSFsub-yy)*n];
	 }
	 if (tx) tmpres[y][x]=ty/tx;
	 else tmpres[y][x]=iDMIN[Nimg]-1.;
      }

      // invalidate any bad points
      if (data[img][y][x]<iDMAX[img] && (tmpres[y][x]<=iDMIN[Nimg] || tmpres[y][x]>=GetParamDouble(FSat)*iDMAX[Nimg])) data[img][y][x] = safedown(iDMIN[img]);
   }
   free_vector(p,1,n*n);
   // E = sum(imgcounts*sqrt(refcts))
   // R = sum(refcts^1.5)
   E=R=0.;
   for (i=0;i<Nstars;i++) if (stars[i].type==1 && stars[i].icm[img]>0 && refcts[i]>0) {
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      x=(int)(tx+1)-1; y=(int)(ty+1)-1;
      if (ppixfOK(img,x,y)) {
	 E+=stars[i].is[img]/stars[i].icm[img];
	 R+=refcts[i];
      }
   }
   // refmult = average ratio of reference/image counts
   refmult[img]=R/E;
   // mlt = average ratio of (imageCounts*imgPSFsum)/(referenceCounts*convolvedPSFsum/kernelSum) ~= imageCountsInRPSF / referenceCountsInRPSF
   mlt=E*fim/R/fdif;
   // rbg = rough background of tmpres, using median
   rbg=medianim(tmpres,img,iDMIN[Nimg],iDMAX[Nimg]);
   if (R==0. || mlt<0.) {
      printf("Math error in subreference\n");
      return;
   }
   //printf("image multiple: %f %f\n",mlt,rbg);
   SetGlobal_NOT_THREADSAFE(IMDIFF,1);
   while (1) {
      // compute DMIN and DMAX and create subtracted reference in res[]
      newMIN=iDMIN[img]-mlt*(iDMAX[Nimg]-rbg)+mlt*mlt*(iDMIN[Nimg]+iRN[Nimg]);
      newMAX=iDMAX[img]-mlt*(iDMIN[Nimg]-rbg)+mlt*mlt*(iDMAX[Nimg]+iRN[Nimg]);
      for (y=0;y<dataim[img].Y;y++) for (x=0;x<dataim[img].X;x++) if (datafOK(img,x,y) && tmpres[y][x]>iDMIN[Nimg] && tmpres[y][x]<GetParamDouble(FSat)*iDMAX[Nimg]) res[img][y][x]=data[img][y][x]-mlt*(tmpres[y][x]-rbg);
      else if (data[img][y][x]<iDMAX[img]*GetParamDouble(FSat)) data[img][y][x]=res[img][y][x]=newMIN-1.;
      else data[img][y][x]=res[img][y][x]=newMAX+1.;
      // E = sum(imgcounts*sqrt(refcts)) -- unchanged from previous
      // R = sum of subtracted photometry * sqrt(refcts) -- hopefully zero
      E=R=0.;
      for (i=0;i<Nstars;i++) if (stars[i].type==1 && stars[i].icm[img]>0 && refcts[i]>0) {
	 clear_sky();
	 shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
	 x=(int)(tx+1)-1; y=(int)(ty+1)-1;
	 if (ppixfOK(img,x,y)) {
	    eval1(img,stars[i].x,stars[i].y,stars[i].pa[img],stars[i].pb[img],stars[i].pc[img],&s,&ss,&c,&ndof,&sh,&ssky,&mx,&my,&cs,&cm,1,0,i,1,0,0,0);
	    E+=stars[i].is[img]/stars[i].icm[img];
	    R+=s/cm;
	 }
      }
      refmult[img]*=1.-R/E;
      mlt/=1.-R/E;
      if (R>=E) {
	 printf("**Fatal Math error in subreference**\n");
	 return;
      }
      //printf("%f %f\n",mlt,1./(1.-R/E));
      if (fabs(R/E)<=0.001) break;
   }
   // increase noise to sqrt( imgNoise^2 + mlt^2*refNoise^2 )
   for (y=0;y<dataim[img].Y;y++) for (x=0;x<dataim[img].X;x++) if (datafOK(img,x,y)) {
      if (data[Nimg][y][x]>0.) data[img][y][x]+=mlt*mlt*data[Nimg][y][x];
      data[img][y][x]+=mlt*mlt*iRN[Nimg];
   }
   iDMIN[img]=newMIN;
   iDMAX[img]=newMAX;
   for (x=0;x<Nstars;x++) stars[x].s=stars[x].is[img]=0.;
   freechip(tmpres,dataim[img].X,dataim[img].Y);
   printf("Subtraction made; scale factor=%6.4f\n",mlt);
   fflush(stdout);
   return;
}

int solve(int ext,int fld,int it,int show) {
   int i,reit=0,c;

   setindx(-1);
   for (i=0;i<Nstars;i++) indx[(int)(stars[i].y*GetParamInt(SubResRef))][(int)(stars[i].x*GetParamInt(SubResRef))]=i;
   for (i=0;i<Nstars;i++) if (stars[i].flag) {
      reit=1;
      imgadd(i);
      indx[(int)(stars[i].y*GetParamInt(SubResRef))][(int)(stars[i].x*GetParamInt(SubResRef))]=-1;
      phot((int)stars[i].x,(int)stars[i].y,i,it);
      imgsub(i);
   }
   c=clean(it,show);
   if (it<3 || (c && it<GetParamInt(MaxIT)/2)) return 2;
   if (GetGlobalInt(fitpsf)) {
      calcpsf(ext,fld);
      SetGlobal_NOT_THREADSAFE(fitpsf,0);
      clean(it,show);
      if (GetGlobalInt(WARMSTART)==2) {
	 for (i=0;i<Nstars;i++) stars[i].flag=1;
	 solve(ext,fld,it,show);
	 for (i=0;i<Nstars;i++) stars[i].flag=1;
	 solve(ext,fld,it,show);
	 for (i=0;i<Nimg;i++) subreference(i);
      }
      for (i=0;i<Nstars;i++) stars[i].flag=1;
      return 2;
   }
   if (reit) return 1;
   return 0;
}

void clearsnmap(void) {
   int y;
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) {
      memset(snmap[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,FLOATSIZE*(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
      memset(snmmap[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,FLOATSIZE*(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
      memset(snmapflag[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
   }
   return;
}

void clearsnmapflag(void) {
   int y;
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) {
      memset(snmapflag[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
   }
   return;
}

void setsnmaprow(int img,int x0,int x1,int y,int pa,int pb,int pc) {
   int x;
   float s,ss,c,ndof,sh,ssky,mx,my,cs,cm;

   for (x=x0*GetParamInt(SubResRef);x<x1*GetParamInt(SubResRef);x++) if (snmapflag[y][x]<2) {
      //clear_sky();
      eval1(img,(x+0.5)/(double)GetParamInt(SubResRef),(y+0.5)/(double)GetParamInt(SubResRef),pa,pb,pc,&s,&ss,&c,&ndof,&sh,&ssky,&mx,&my,&cs,&cm,1,1,-1,0,0,1,1);
      if (snmapflag[y][x]==0) {
	 snmap[y][x]=snmmap[y][x]=0.;
	 snmapflag[y][x]=1;
      }
      if (ss>0) {
	 snmap[y][x]+=s*fabs(s)/(ss*ss);
	 snmmap[y][x]+=s*fabs(s)/(ss*ss*(c*c+0.25));
      }
   }
}

void setsnmap(int IMG,int x0,int x1,int y0,int y1) {
   int x,y,img;
   float pa,pb,pc;

   if (y0<GetGlobalInt(YMIN)) y0=GetGlobalInt(YMIN);
   if (y1>GetGlobalInt(YMAX)) y1=GetGlobalInt(YMAX);
   if (x0<GetGlobalInt(XMIN)) x0=GetGlobalInt(XMIN);
   if (x1>GetGlobalInt(XMAX)) x1=GetGlobalInt(XMAX);
   for (img=0;img<Nimg || (img==Nimg && IMG==Nimg);img++) if (IMG<0 || img==IMG) {
      pa=apsf[img][0][0]+(apsf[img][0][3]+apsf[img][0][4])/12.;
      pb=apsf[img][1][0]+(apsf[img][1][3]+apsf[img][1][4])/12.;
      pc=apsf[img][2][0]+(apsf[img][2][3]+apsf[img][2][4])/12.;
      calc1psf(img,0.5+dataim[img].X/2,0.5+dataim[img].Y/2,rphot[img],pa,pb,pc,1,2);
#ifdef DOLPHOT_THREADED
      if (omp_get_max_threads()<2) {
	 for (y=y0*GetParamInt(SubResRef);y<y1*GetParamInt(SubResRef);y++) {
	    setsnmaprow(img,x0,x1,y,pa,pb,pc);
	 }
      }
      else
#pragma omp parallel for schedule(guided)
#endif
	 for (y=y0*GetParamInt(SubResRef);y<y1*GetParamInt(SubResRef);y++) {
	    setsnmaprow(img,x0,x1,y,pa,pb,pc);
	 }
   }
   for (y=y0*GetParamInt(SubResRef);y<y1*GetParamInt(SubResRef);y++) for (x=x0*GetParamInt(SubResRef);x<x1*GetParamInt(SubResRef);x++) if (snmapflag[y][x]<2) {
      if (snmap[y][x]<=0) snmap[y][x]=0.;
      else snmap[y][x]=sqrt(snmap[y][x]);
      if (snmmap[y][x]>0.) snmmap[y][x]=sqrt(snmmap[y][x]);
      else snmmap[y][x]=-sqrt(-snmmap[y][x]);
      snmapflag[y][x]=2;
   }
   return;
}

// Note peaksn should be called in subresref-scaled pixel units (pixel x SubResRef), not native pixels
static inline float peaksn(int img,int x,int y) {
   if (x<=GetGlobalInt(XMIN)*GetParamInt(SubResRef) || y<=GetGlobalInt(YMIN)*GetParamInt(SubResRef) || x>=GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1 || y>=GetGlobalInt(YMAX)*GetParamInt(SubResRef)-1) return 0.;
   if (img>=0 && (x<=0 || y<=0 || x>=dataim[img].X*GetParamInt(SubResRef)-1 || y>=dataim[img].Y*GetParamInt(SubResRef)-1)) return 0.;
   if (snmmap[y][x]<snmmap[y-1][x-1] || snmmap[y][x]<snmmap[y-1][x] || snmmap[y][x]<snmmap[y-1][x+1] || snmmap[y][x]<snmmap[y][x+1] || snmmap[y][x]<snmmap[y+1][x+1] || snmmap[y][x]<snmmap[y+1][x] || snmmap[y][x]<snmmap[y+1][x-1] || snmmap[y][x]<snmmap[y][x-1]) return 0.;
   return snmap[y][x];
}

// Note qpeak should be called in subresref-scaled pixel units (pixel x SubResRef), not native pixels
static inline float qpeak(int x,int y,int strict) {
   if (x<=GetGlobalInt(XMIN)*GetParamInt(SubResRef) || y<=GetGlobalInt(YMIN)*GetParamInt(SubResRef) || x>=GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1 || y>=GetGlobalInt(YMAX)*GetParamInt(SubResRef)-1) return 0.;
   if (strict && ran[y][x]) return 0;
   return peaksn(-1,x,y);
}

float alphot(int img,double x0,double y0,double*x,double*y,int finding,int strictpos) {
   float s,ss,sky,chi,sh;
   int i,j,cl;
   static float pa[MAXNIMG],pb[MAXNIMG],pc[MAXNIMG];

   if (finding!=0) {
      if (Nstars>=MAXNSTARS) return 0;
      centroid(img,(int)x0,(int)y0,x,y);
      cl=1;
      s=ss=0.0;
      getpsfpars(img,*x,*y,pa,pb,pc);
      if (photsearch(img,x,y,pa,pb,pc,&s,&ss,&chi,&sh,&sky,NULL,NULL,NULL,&cl,1,-1)<=0) return 0;
      if (finding==1) {
	 if (cl>1 || s<ss*10 || s<ss*GetParamDouble(SigAlign) || s<=0 || fabs(sh)>0.3) return 0;
	 getpsfpars(img,*x,*y,pa,pb,pc);
	 if (photsearch(img,x,y,pa,pb,pc,&s,&ss,&chi,&sh,&sky,NULL,NULL,NULL,&cl,-1,-1)<=0) return 0;
	 if (cl>1 || s<ss*10 || s<ss*GetParamDouble(SigAlign) || s<=0 || fabs(sh)>0.3) return 0;
      }
      else {
	 if (cl>1 || s<ss*GetParamDouble(SigAlign) || s<=0) return 0;
	 getpsfpars(img,*x,*y,pa,pb,pc);
	 if (photsearch(img,x,y,pa,pb,pc,&s,&ss,&chi,&sh,&sky,NULL,NULL,NULL,&cl,-1,-1)<=0) return 0;
	 if (cl>1 || s<ss*GetParamDouble(SigAlign) || s<=0) return 0;
      }
      if (*x<GetGlobalInt(XMIN) || *x>=GetGlobalInt(XMAX) || *y<GetGlobalInt(YMIN) || *y>=GetGlobalInt(YMAX) || indx[(int)(*y*GetParamInt(SubResRef))][(int)(*x*GetParamInt(SubResRef))]>=0) return 0;
      indx[(int)(*y*GetParamInt(SubResRef))][(int)(*x*GetParamInt(SubResRef))]=Nstars;
      for (j=-(int)GetParamDouble(RCombine);j<=(int)GetParamDouble(RCombine);j++) for (i=-(int)GetParamDouble(RCombine);i<=(int)GetParamDouble(RCombine);i++) if ((int)(*y*GetParamInt(SubResRef))+j>=0 && (int)(*y*GetParamInt(SubResRef))+j<dataim[img].Y && (int)(*x*GetParamInt(SubResRef))+i>=0 && (int)(*x*GetParamInt(SubResRef))+i<dataim[img].X && i*i+j*j<(GetParamDouble(RCombine)-0.5)*(GetParamDouble(RCombine)-0.5)) ran[(int)(*y*GetParamInt(SubResRef))+j][(int)(*x*GetParamInt(SubResRef))+i]=1;
      stars[Nstars].x=*x;
      stars[Nstars].y=*y;
      stars[Nstars].passID=-1;
      Nstars++;
      return 1;
   }
   else {
      *x=x0;
      *y=y0;
      cl=1;
      s=ss=0.0;
      getpsfpars(img,*x,*y,pa,pb,pc);
      if (photsearch(img,x,y,pa,pb,pc,&s,&ss,&chi,&sh,&sky,NULL,NULL,NULL,&cl,-1,-1)<=0) return 0;
      if (cl>1 || s<ss*GetParamDouble(SigAlign) || s<=0 || chi>10. || fabs(sh)>0.4) return 0;
      if (strictpos) {
	 double sx0,sy0,sx1,sy1;
	 shift(img,x0,y0,&sx0,&sy0,1);
	 shift(img,*x,*y,&sx1,&sy1,1);
	 if ((sx1-sx0)*(sx1-sx0)+(sy1-sy0)*(sy1-sy0)>1+RAper[img]*RAper[img]*0.5) return 0;
      }
      return s/ss;
   }
}

// GLOBAL (PARALLEL-SAFE) VARIABLES - each thread has different image
typedef float dptype[5];
int NFREE=5,Nalign[MAXNIMG]; // Note NFREE set outside of parallel code
dptype *Alist[MAXNIMG];
static const double Align45scale[21]={1,
   1e3,1e3,
   1e6,1e6,1e6,
   1e9,1e9,1e9,1e9,
   1e12,1e12,1e12,1e12,1e12,
   1e15,1e15,1e15,1e15,1e15,1e15};

float aminfunc(float*c,int img) {
   double x,y;
   float rval=0,r2;
   int i;

   if (GetParamInt(Align)<4) {
      for (i=0;i<=GetParamInt(Align);i++) dpos[img][i]=c[i+1];
      if (GetParamInt(Rotate)) dpos[img][4]=c[GetParamInt(Align)+2];
   }
   else {
      for (i=0;i<NFREE;i++) dpos[img][i]=c[i+1]/Align45scale[i%10];
   }
   for (i=0;i<Nalign[img];i++) {
      shift(img,Alist[img][i][0],Alist[img][i][1],&x,&y,1);
      r2=((x-Alist[img][i][2])*(x-Alist[img][i][2])+(y-Alist[img][i][3])*(y-Alist[img][i][3]))*25;
      //rval+=r2;
      rval+=log(1+0.5*r2); // Cauchy with sig^2 = 1/12.5
   }
   return rval/Nalign[img];
}

float aminfunc4(float *c,int img) {
   int i,x0,y0,N=0,xstep,ystep;
   double x,y;
   float rval=0;

   xstep = dataim[img].X/16;
   ystep = dataim[img].Y/16;
   for (i=0;i<20;i++) dpos[img][i+20] = c[i+1]/Align45scale[i%10];
   for (x0=0;x0<=dataim[img].X;x0+=xstep) {
      for (y0=0;y0<=dataim[img].Y;y0+=ystep) {
	 shift(img,x0,y0,&x,&y,-1);
	 shift(img,x,y,&x,&y,1);
	 rval += ((x0-x)*(x0-x)+(y0-y)*(y0-y))*25;
	 N++;
      }
   }
   return rval/N;
}

float aminfuncR2I(float *c,int img) {
   int i,x0,y0,N=0,xstep,ystep;
   double x,y;
   float rval=0;

   xstep = dataim[img].X/16;
   ystep = dataim[img].Y/16;
   for (i=0;i<20;i++) ref2img[img][i+20] = c[i+1]/Align45scale[i%10];
   for (x0=0;x0<=dataim[img].X;x0+=xstep) {
      for (y0=0;y0<=dataim[img].Y;y0+=ystep) {
	 shift(img,x0,y0,&x,&y,-2);
	 shift(img,x,y,&x,&y,2);
	 rval += ((x0-x)*(x0-x)+(y0-y)*(y0-y))*25;
	 N++;
      }
   }
   return rval/N;
}

//powell() and affiliated code from NR;
// GLOBAL (PARALLEL-SAFE) VARIABLES - each thread has different image
int ncom[MAXNIMG];
float *pcom[MAXNIMG],*xicom[MAXNIMG],(*nrfunc[MAXNIMG])(float [],int);

float f1dim(float x,int img)
{
	int j;
	float f,*xt;

	xt=vector(1,ncom[img]);
	for (j=1;j<=ncom[img];j++) xt[j]=pcom[img][j]+x*xicom[img][j];
	f=(*nrfunc[img])(xt,img);
	free_vector(xt,1,ncom[img]);
	return f;
}

#define GOLD 1.618034
#define GLIMIT 100.0
#define TINY 1.0e-20
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);
void mnbrak(float *ax, float *bx, float *cx, float *fa, float *fb, float *fc,
	float (*func)(float,int),int img)
{
	float ulim,u,r,q,fu,dum;

	*fa=(*func)(*ax,img);
	*fb=(*func)(*bx,img);
	if (*fb > *fa) {
		SHFT(dum,*ax,*bx,dum)
		SHFT(dum,*fb,*fa,dum)
	}
	*cx=(*bx)+GOLD*(*bx-*ax);
	*fc=(*func)(*cx,img);
	while (*fb > *fc) {
	        float abs_qr;
		r=(*bx-*ax)*(*fb-*fc);
		q=(*bx-*cx)*(*fb-*fa);
		abs_qr = fabs(q-r);
		u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/
			(2.0*SIGN(FMAX(abs_qr,TINY),q-r));
		ulim=(*bx)+GLIMIT*(*cx-*bx);
		if ((*bx-u)*(u-*cx) > 0.0) {
		        fu=(*func)(u,img);
			if (fu < *fc) {
				*ax=(*bx);
				*bx=u;
				*fa=(*fb);
				*fb=fu;
				return;
			} else if (fu > *fb) {
				*cx=u;
				*fc=fu;
				return;
			}
			u=(*cx)+GOLD*(*cx-*bx);
			fu=(*func)(u,img);
		} else if ((*cx-u)*(u-ulim) > 0.0) {
   		        fu=(*func)(u,img);
			if (fu < *fc) {
				SHFT(*bx,*cx,u,*cx+GOLD*(*cx-*bx))
			        SHFT(*fb,*fc,fu,(*func)(u,img))
			}
		} else if ((u-ulim)*(ulim-*cx) >= 0.0) {
			u=ulim;
			fu=(*func)(u,img);
		} else {
			u=(*cx)+GOLD*(*cx-*bx);
			fu=(*func)(u,img);
		}
		SHFT(*ax,*bx,*cx,u)
		SHFT(*fa,*fb,*fc,fu)
	}
	return;
}
#undef GOLD
#undef GLIMIT
#undef TINY
#undef SHFT

#define ITMAX 100
#define CGOLD 0.3819660
#define ZEPS 1.0e-10
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);
float brent(float ax, float bx, float cx, float (*f)(float,int), float tol,
	    float *xmin,int img)
{
	int iter;
	float a,b,d=0,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	float e=0.0;

	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=(*f)(x,img);
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (fabs(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=fabs(q);
			etemp=e;
			e=d;
			if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}
		u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		fu=(*f)(u,img);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	nrerror("Too many iterations in brent");
	*xmin=x;
	return fx;
}
#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SHFT

#define TOL 2.0e-4
void linmin(float p[], float xi[], int n, float *fret, float (*func)(float [],int),int img)
{
	int j;
	float xx,xmin,fx,fb,fa,bx,ax;

	ncom[img]=n;
	pcom[img]=vector(1,n);
	xicom[img]=vector(1,n);
	nrfunc[img]=func;
	for (j=1;j<=n;j++) {
		pcom[img][j]=p[j];
		xicom[img][j]=xi[j];
	}
	ax=0.0;
	xx=1.0;
	mnbrak(&ax,&xx,&bx,&fa,&fx,&fb,f1dim,img);
	*fret=brent(ax,xx,bx,f1dim,TOL,&xmin,img);
	for (j=1;j<=n;j++) {
		xi[j] *= xmin;
		p[j] += xi[j];
	}
	free_vector(xicom[img],1,n);
	free_vector(pcom[img],1,n);
	return;
}
#undef TOL

#define ITMAX 10000
void powell(float p[], float **xi, int n, float ftol, int *iter, float *fret,
	    float (*func)(float [],int), int img)
{
	int i,ibig,j;
	float del,fp,fptt,t,*pt,*ptt,*xit;

	pt=vector(1,n);
	ptt=vector(1,n);
	xit=vector(1,n);
	*fret=(*func)(p,img);
	for (j=1;j<=n;j++) pt[j]=p[j];
	for (*iter=1;;++(*iter)) {
		fp=(*fret);
		ibig=0;
		del=0.0;
		for (i=1;i<=n;i++) {
			for (j=1;j<=n;j++) xit[j]=xi[j][i];
			fptt=(*fret);
			linmin(p,xit,n,fret,func,img);
			if (fabs(fptt-(*fret)) > del) {
				del=fabs(fptt-(*fret));
				ibig=i;
			}
		}
		if (2.0*fabs(fp-(*fret)) <= ftol*(fabs(fp)+fabs(*fret))) {
			free_vector(xit,1,n);
			free_vector(ptt,1,n);
			free_vector(pt,1,n);
			return;
		}
		if (*iter == ITMAX) nrerror("powell exceeding maximum iterations.");
		for (j=1;j<=n;j++) {
			ptt[j]=2.0*p[j]-pt[j];
			xit[j]=p[j]-pt[j];
			pt[j]=p[j];
		}
		fptt=(*func)(ptt,img);
		if (fptt < fp) {
		   t=2.0*(fp-2.0*(*fret)+fptt)*(fp-(*fret)-del)*(fp-(*fret)-del)-del*(fp-fptt)*(fp-fptt);
			if (t < 0.0) {
			   linmin(p,xit,n,fret,func,img);
				for (j=1;j<=n;j++) {
					xi[j][ibig]=xi[j][n];
					xi[j][n]=xit[j];
				}
			}
		}
	}
	return;
}
#undef ITMAX
//end of NR stuff;

// Compute distortion correction from WCS
void headerWCS(int img,float* imgfwd) {
   char str[81];
   int sip[2],ord,i;
   for (i=0;i<40;i++) imgfwd[i]=0.0;

   // parse coordinate types
   strcpy(str,getcardval(dataim+img,"CTYPE1",1));
   if (!strcmp(str,"RA---TAN")) sip[0]=0;
   else if (!strcmp(str,"RA---TAN-SIP")) sip[0]=1;
   else {printf("Error: CTYPE1 of %s not understood\n",str); exit(-1);}
   strcpy(str,getcardval(dataim+img,"CTYPE2",1));
   if (!strcmp(str,"DEC--TAN")) sip[1]=0;
   else if (!strcmp(str,"DEC--TAN-SIP")) sip[1]=1;
   else {printf("Error: CTYPE2 of %s not understood\n",str); exit(-1);}

   // define reference pixel
   wcsref[img][0] = atof(getcardval(dataim+img,"CRPIX1",1))-0.5;
   wcsref[img][1] = atof(getcardval(dataim+img,"CRPIX2",1))-0.5;
   wcsref[img][2] = atof(getcardval(dataim+img,"CRVAL1",1));
   wcsref[img][3] = atof(getcardval(dataim+img,"CRVAL2",1));

   // Use PC matrix if CD not available
   if (!sip[0] && !sip[1]) {
      int all=0;
      char *rval;
      rval = getcardval(dataim+img,"CD1_1",0);
      if (*rval!=0) {
	 rval = getcardval(dataim+img,"CD1_2",0);
	 if (*rval!=0) {
	    rval = getcardval(dataim+img,"CD2_1",0);
	    if (*rval!=0) {
	       rval = getcardval(dataim+img,"CD2_2",0);
	       if (*rval!=0) all=1;
	    }
	 }
      }
      if (all==0) {
	 float xmult,ymult;
	 printf("CD matrix not found; using PC matrix instead\n");
	 xmult = atof(getcardval(dataim+img,"CDELT1",1));
	 ymult = atof(getcardval(dataim+img,"CDELT2",1));
	 imgfwd[0] = atof(getcardval(dataim+img,"PC1_1",1))*xmult;
	 imgfwd[1] = atof(getcardval(dataim+img,"PC1_2",1))*xmult;
	 imgfwd[20] = atof(getcardval(dataim+img,"PC2_1",1))*ymult;
	 imgfwd[21] = atof(getcardval(dataim+img,"PC2_2",1))*ymult;
	 //printf("%g %g %g %g\n",imgfwd[0],imgfwd[1],imgfwd[20],imgfwd[21]);
	 return;
      }
   }

   // parse first transformation
   imgfwd[0] = atof(getcardval(dataim+img,"CD1_1",1));
   imgfwd[1] = atof(getcardval(dataim+img,"CD1_2",1));
   if (sip[0]) {
      ord = atoi(getcardval(dataim+img,"A_ORDER",1));
      if (ord>1) {
	 imgfwd[2] = atof(getcardval(dataim+img,"A_2_0",1));
	 imgfwd[3] = atof(getcardval(dataim+img,"A_1_1",1));
	 imgfwd[4] = atof(getcardval(dataim+img,"A_0_2",1));
      }
      if (ord>2) {
	 imgfwd[5] = atof(getcardval(dataim+img,"A_3_0",1));
	 imgfwd[6] = atof(getcardval(dataim+img,"A_2_1",1));
	 imgfwd[7] = atof(getcardval(dataim+img,"A_1_2",1));
	 imgfwd[8] = atof(getcardval(dataim+img,"A_0_3",1));
      }
      if (ord>3) {
	 imgfwd[9] = atof(getcardval(dataim+img,"A_4_0",1));
	 imgfwd[10] = atof(getcardval(dataim+img,"A_3_1",1));
	 imgfwd[11] = atof(getcardval(dataim+img,"A_2_2",1));
	 imgfwd[12] = atof(getcardval(dataim+img,"A_1_3",1));
	 imgfwd[13] = atof(getcardval(dataim+img,"A_0_4",1));
      }
      if (ord>4) {
	 imgfwd[14] = atof(getcardval(dataim+img,"A_5_0",1));
	 imgfwd[15] = atof(getcardval(dataim+img,"A_4_1",1));
	 imgfwd[16] = atof(getcardval(dataim+img,"A_3_2",1));
	 imgfwd[17] = atof(getcardval(dataim+img,"A_2_3",1));
	 imgfwd[18] = atof(getcardval(dataim+img,"A_1_4",1));
	 imgfwd[19] = atof(getcardval(dataim+img,"A_0_5",1));
      }
   }

   // parse second transformation
   imgfwd[20] = atof(getcardval(dataim+img,"CD2_1",1));
   imgfwd[21] = atof(getcardval(dataim+img,"CD2_2",1));
   if (sip[1]) {
      ord = atoi(getcardval(dataim+img,"B_ORDER",1));
      if (ord>1) {
	 imgfwd[22] = atof(getcardval(dataim+img,"B_2_0",1));
	 imgfwd[23] = atof(getcardval(dataim+img,"B_1_1",1));
	 imgfwd[24] = atof(getcardval(dataim+img,"B_0_2",1));
      }
      if (ord>2) {
	 imgfwd[25] = atof(getcardval(dataim+img,"B_3_0",1));
	 imgfwd[26] = atof(getcardval(dataim+img,"B_2_1",1));
	 imgfwd[27] = atof(getcardval(dataim+img,"B_1_2",1));
	 imgfwd[28] = atof(getcardval(dataim+img,"B_0_3",1));
      }
      if (ord>3) {
	 imgfwd[29] = atof(getcardval(dataim+img,"B_4_0",1));
	 imgfwd[30] = atof(getcardval(dataim+img,"B_3_1",1));
	 imgfwd[31] = atof(getcardval(dataim+img,"B_2_2",1));
	 imgfwd[32] = atof(getcardval(dataim+img,"B_1_3",1));
	 imgfwd[33] = atof(getcardval(dataim+img,"B_0_4",1));
      }
      if (ord>4) {
	 imgfwd[34] = atof(getcardval(dataim+img,"B_5_0",1));
	 imgfwd[35] = atof(getcardval(dataim+img,"B_4_1",1));
	 imgfwd[36] = atof(getcardval(dataim+img,"B_3_2",1));
	 imgfwd[37] = atof(getcardval(dataim+img,"B_2_3",1));
	 imgfwd[38] = atof(getcardval(dataim+img,"B_1_4",1));
	 imgfwd[39] = atof(getcardval(dataim+img,"B_0_5",1));
      }
   }
   //printf("%g %g %g %g\n",imgfwd[0],imgfwd[1],imgfwd[20],imgfwd[21]);
}

// GLOBAL (PARALLEL-SAFE) VARIABLES - each thread has different image
int REFIMG; // REFIMG set outside of parallel code
double refpix[MAXNIMG][2]; // position of standard pixel on reference image
double refra[MAXNIMG][2];  // RA/DEC of standard pixel

float aminfuncWCSref(float *c,int ASSUMING_REFIMG_CONSTANT) {
   int i,x0,y0,N=0,xstep,ystep;
   double x,y;
   float rval=0;

   xstep = dataim[REFIMG].X/16;
   ystep = dataim[REFIMG].Y/16;
   for (i=0;i<40;i++) wcs[REFIMG][i+40] = c[i+1]/Align45scale[i%20+1];
   for (x0=0;x0<=dataim[REFIMG].X;x0+=xstep) {
      for (y0=0;y0<=dataim[REFIMG].Y;y0+=ystep) {
	 shift(REFIMG,x0,y0,&x,&y,-3);  // REF pixel to REF tangent
	 shift(REFIMG,x,y,&x,&y,3);     // REF tangent to REF pixel
	 rval += ((x0-x)*(x0-x)+(y0-y)*(y0-y))*25;
	 N++;
      }
   }
   return rval/N;
}

float aminfuncWCSrev(float *c,int img) {
   int i,x0,y0,N=0,xstep,ystep;
   double x,y;
   float rval=0;

   xstep = dataim[img].X/16;
   ystep = dataim[img].Y/16;
   for (i=0;i<40;i++) wcs[img][i+40] = c[i+1]/Align45scale[i%20+1];
   for (x0=0;x0<=dataim[img].X;x0+=xstep) {
      for (y0=0;y0<=dataim[img].Y;y0+=ystep) {
	 wcsref[img][2]=refra[img][0]; wcsref[img][3]=refra[img][1];
	 shift(img,x0,y0,&x,&y,-3); // IMG pixel to IMG tangent
	 fixRA(img,x,y,&x,&y,-3);      // IMG tangent to RA/DEC
	 fixRA(REFIMG,x,y,&x,&y,3);      // RA/DEC to REF tangent
	 shift(REFIMG,x,y,&x,&y,3);   // REF tangent to REF pixel
	 wcsref[img][2]=refpix[img][0]; wcsref[img][3]=refpix[img][1];
	 shift(img,x,y,&x,&y,3);    // refpix to imgpix
	 rval += ((x0-x)*(x0-x)+(y0-y)*(y0-y))*25;
	 N++;
      }
   }
   return rval/N;
}

float aminfuncWCSfwd(float *c,int img) {
   int i,x0,y0,N=0,xstep,ystep;
   double x,y;
   float rval=0;

   xstep = dataim[img].X/16;
   ystep = dataim[img].Y/16;
   for (i=0;i<40;i++) wcs[img][i] = c[i+1]/Align45scale[i%20+1];
   for (x0=0;x0<=dataim[img].X;x0+=xstep) {
      for (y0=0;y0<=dataim[img].Y;y0+=ystep) {
	 shift(img,x0,y0,&x,&y,-3);    // imgpix to refpix
	 shift(img,x,y,&x,&y,3);       // refpix to imgpix
	 rval += ((x0-x)*(x0-x)+(y0-y)*(y0-y))*25;
	 N++;
      }
   }
   return rval/N;
}

void convertWCS2_img(int img,char *outstr) {
   int i,j;
   double r,d;
   float p[40],**xi,rv;
   double m[2][2];
   char *outptr=outstr;

   xi=matrix(1,40,1,40);
   headerWCS(img,wcs[img]);
   // location of reference pixel in RA/DEC and in reference img pixels
   refra[img][0]=wcsref[img][2];
   refra[img][1]=wcsref[img][3];
   fixRA(REFIMG,refra[img][0],refra[img][1],&r,&d,3); // convert RA/DEC to REF tangent
   shift(REFIMG,r,d,refpix[img],refpix[img]+1,3);

   // solve for reverse
   m[0][0] = wcs[REFIMG][40]*wcs[img][0] + wcs[REFIMG][41]*wcs[img][20];
   m[0][1] = wcs[REFIMG][40]*wcs[img][1] + wcs[REFIMG][41]*wcs[img][21];
   m[1][0] = wcs[REFIMG][60]*wcs[img][0] + wcs[REFIMG][61]*wcs[img][20];
   m[1][1] = wcs[REFIMG][60]*wcs[img][1] + wcs[REFIMG][61]*wcs[img][21];
   d = m[0][0]*m[1][1] - m[0][1]*m[1][0];
   if (d==0.0) {
      printf("ERROR: determinant of WCS is zero\n");
      exit(-1);
   }
   memset(p,0,sizeof(p));
   p[0] = m[1][1]/d*Align45scale[1];
   p[1] = -m[0][1]/d*Align45scale[2];
   p[20] = -m[1][0]/d*Align45scale[1];
   p[21] = m[0][0]/d*Align45scale[2];
   outptr += sprintf(outptr,"reverse rmsi = %f, ",0.2*sqrt(aminfuncWCSrev(p-1,img)));
   for (i=1;i<=40;i++) {
      for (j=1;j<=40;j++) xi[i][j]=0.;
      xi[i][i]=1.;
   }
   powell(p-1,xi,40,1.e-8,&i,&rv,&aminfuncWCSrev,img);
   outptr += sprintf(outptr,"rmsf = %f\n",0.2*sqrt(aminfuncWCSrev(p-1,img)));
   for (i=0;i<40;i++) wcs[img][40+i] = p[i]/Align45scale[i%20+1];

   // solve for forward
   wcsref[img][2]=refpix[img][0];
   wcsref[img][3]=refpix[img][1];
   d = wcs[img][40]*wcs[img][61] - wcs[img][41]*wcs[img][60];
   if (d==0.0) {
      printf("ERROR: determinant of WCS is zero\n");
      exit(-1);
   }
   memset(p,0,sizeof(p));
   p[0] = wcs[img][61]/d*Align45scale[1];
   p[1] = -wcs[img][41]/d*Align45scale[2];
   p[20] = -wcs[img][60]/d*Align45scale[1];
   p[21] = wcs[img][40]/d*Align45scale[2];
   outptr += sprintf(outptr,"forward rmsi = %f, ",0.2*sqrt(aminfuncWCSfwd(p-1,img)));
   for (i=1;i<=40;i++) {
      for (j=1;j<=40;j++) xi[i][j]=0.;
      xi[i][i]=1.;
   }
   powell(p-1,xi,40,1.e-8,&i,&rv,&aminfuncWCSfwd,img);
   outptr += sprintf(outptr,"rmsf = %f\n",0.2*sqrt(aminfuncWCSfwd(p-1,img)));
   for (i=0;i<40;i++) wcs[img][i] = p[i]/Align45scale[i%20+1];
   free_matrix(xi,1,40,1,40);
}

void convertWCS2(void) {
   int img,i,j;
   double d;
   float p[40],**xi,rv;
   char outstr[MAXNIMG][1024];

   xi=matrix(1,40,1,40);

   // compute reverse transform for reference image
   REFIMG=0;
   if (Timg>Nimg) REFIMG=Nimg;

   // read header, populate imgfwd and imgpix
   headerWCS(REFIMG,wcs[REFIMG]);

   // solve for first-order reverse transformation
   d = wcs[REFIMG][0]*wcs[REFIMG][21] - wcs[REFIMG][1]*wcs[REFIMG][20];
   if (d==0.0) {
      printf("ERROR: determinant of WCS is zero\n");
      exit(-1);
   }
   memset(p,0,sizeof(p));
   p[0] = wcs[REFIMG][21]/d*Align45scale[1];
   p[1] = -wcs[REFIMG][1]/d*Align45scale[2];
   p[20] = -wcs[REFIMG][20]/d*Align45scale[1];
   p[21] = wcs[REFIMG][0]/d*Align45scale[2];
   printf("reference rmsi = %f, ",0.2*sqrt(aminfuncWCSref(p-1,REFIMG)));
   for (i=1;i<=40;i++) {
      for (j=1;j<=40;j++) xi[i][j]=0.;
      xi[i][i]=1.;
   }
   powell(p-1,xi,40,1.e-8,&i,&rv,&aminfuncWCSref,REFIMG);
   printf("rmsf = %f\n",0.2*sqrt(aminfuncWCSref(p-1,REFIMG)));
   fflush(stdout);
   for (i=0;i<40;i++) wcs[REFIMG][40+i] = p[i]/Align45scale[i%20+1];

   // compute transformations for non-reference images
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 if (img!=REFIMG) convertWCS2_img(img,outstr[img]);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 if (img!=REFIMG) convertWCS2_img(img,outstr[img]);
      }
   for (img=0;img<Nimg;img++) if (img!=REFIMG) printf("%s",outstr[img]);
   fflush(stdout);

   free_matrix(xi,1,40,1,40);

   // set wcs[] for reference image
   memset(wcs[REFIMG],0,sizeof(wcstype));
   wcsref[REFIMG][0] = GetGlobalInt(X)*0.5;
   wcsref[REFIMG][1] = GetGlobalInt(Y)*0.5;
   wcsref[REFIMG][2] = GetGlobalInt(X)*0.5;
   wcsref[REFIMG][3] = GetGlobalInt(Y)*0.5;
   wcs[REFIMG][0] = 1.0;
   wcs[REFIMG][21] = 1.0;
   wcs[REFIMG][40] = 1.0;
   wcs[REFIMG][61] = 1.0;
}

// Delta to alignment info based on WCS 1st order solution
void convertWCS1_img(int img,int refimg,double tref,double sref,double refm[2][2],char *outstr) {
   double x,y,xr,yr,dx,dy,t[2],s[2],ct,st;
   float wcsdata[40];

   headerWCS(img,wcsdata);

   // Target location of reference pixel on reference image
   fixRA(refimg,wcsref[img][2],wcsref[img][3],&x,&y,3); // RA/DEC to REF tangent
   xr = wcsref[refimg][0] + refm[0][0]*(x-wcsref[refimg][2]) + refm[0][1]*(y-wcsref[refimg][3]);
   yr = wcsref[refimg][1] + refm[1][0]*(x-wcsref[refimg][2]) + refm[1][1]*(y-wcsref[refimg][3]);

   // compute scale and rotation
   shift(img,wcsref[img][0],wcsref[img][1],&x,&y,-3);
   shift(img,wcsref[img][0]+1.0,wcsref[img][1],&dx,&dy,-3);
   dx-=x; dy-=y;
   s[0] = sqrt(dx*dx+dy*dy)/sqrt(wcsdata[0]*wcsdata[0]+wcsdata[20]*wcsdata[20]);
   t[0] = atan2(dy,dx)-atan2(wcsdata[20],-wcsdata[0]);
   shift(img,wcsref[img][0],wcsref[img][1]+1.0,&dx,&dy,-3);
   dx-=x; dy-=y;
   s[1] = sqrt(dx*dx+dy*dy)/sqrt(wcsdata[1]*wcsdata[1]+wcsdata[21]*wcsdata[21]);
   t[1] = atan2(dy,dx)-atan2(wcsdata[21],-wcsdata[1]);

   s[0] = 0.5*(s[0]+s[1]);
   if (t[0]<t[1]-M_PI) t[0] += 2*M_PI;
   else if (t[0]>t[1]+M_PI) t[0] -= 2*M_PI;
   t[0] = 0.5*(t[0]+t[1]);
   if (t[0]<tref-M_PI) t[0] += 2*M_PI;
   else if (t[0]>tref+M_PI) t[0] -= 2*M_PI;
   s[0] /= sref;
   t[0] -= tref;

   // additional rotation due to delta RA
   t[0] += (wcsref[img][2]-wcsref[refimg][2])*M_PI/180.0 * sin(wcsref[refimg][3]*M_PI/180.0);

   dpos[img][2] *= s[0];
   dpos[img][4] += t[0]*180.0/M_PI;

   // compute shift
   shift(img,wcsref[img][0],wcsref[img][1],&x,&y,-3); // Output from *shift (no rotation, scaling, or translation) plus (X,Y)*0.5
   x-=GetGlobalInt(X)*0.5;
   y-=GetGlobalInt(Y)*0.5;

   // Solve for dpos[img][0] and dpos[img][1]
   ct = cos(t[0]);
   st = sin(t[0]);
   xr=(xr-GetGlobalInt(X)*0.5)*s[0];
   yr=(yr-GetGlobalInt(Y)*0.5)*s[0];
   dpos[img][0] += x-ct*xr+st*yr;
   dpos[img][1] += y-st*xr-ct*yr;
   sprintf(outstr,"image %d: shift %4.2f %4.2f, scale %8.6f, rotate %5.3f\n",img+1,x-ct*xr+st*yr,y-st*xr-ct*yr,s[0],t[0]*180.0/M_PI);
   if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
   {
      fprintf(fverb,"WCS image %d: %f %f %f %f\n",img+1,x-ct*xr+st*yr,y-st*xr-ct*yr,s[0],t[0]*180.0/M_PI);
      fflush(fverb);
   }

   // Verify
   //shift(img,wcsref[img][0],wcsref[img][1],&x,&y,-1);
}

void convertWCS1(void) {
   int img,refimg;
   double x,y,dx,dy,t[2],s[2],tref,sref,refm[2][2];
   float wcsdata[40];
   char outstr[MAXNIMG][256];

   // read WCS data for reference image
   refimg=0;
   if (Timg>Nimg) refimg=Nimg;
   headerWCS(refimg,wcsdata);
   shift(refimg,wcsref[refimg][0],wcsref[refimg][1],&x,&y,-3);
   shift(refimg,wcsref[refimg][0]+1.0,wcsref[refimg][1],&dx,&dy,-3);
   dx-=x; dy-=y;
   s[0] = sqrt(dx*dx+dy*dy)/sqrt(wcsdata[0]*wcsdata[0]+wcsdata[20]*wcsdata[20]);
   t[0] = atan2(dy,dx)-atan2(wcsdata[20],-wcsdata[0]);
   shift(refimg,wcsref[refimg][0],wcsref[refimg][1]+1.0,&dx,&dy,-3);
   dx-=x; dy-=y;
   s[1] = sqrt(dx*dx+dy*dy)/sqrt(wcsdata[1]*wcsdata[1]+wcsdata[21]*wcsdata[21]);
   t[1] = atan2(dy,dx)-atan2(wcsdata[21],-wcsdata[1]);
   sref = 0.5*(s[0]+s[1]);
   if (t[0]<t[1]-M_PI) t[0] += 2*M_PI;
   else if (t[0]>t[1]+M_PI) t[0] -= 2*M_PI;
   tref = 0.5*(t[0]+t[1]);

   // RA/DEC-to-X/Y
   x = wcsdata[0]*wcsdata[21] - wcsdata[1]*wcsdata[20];
   if (x==0.0) {
      printf("ERROR: determinant of reference WCS is zero\n");
      exit(-1);
   }
   refm[0][0] = wcsdata[21]/x;  // dxref/dRA
   refm[0][1] = -wcsdata[1]/x;  // dxref/dDec
   refm[1][0] = -wcsdata[20]/x; // dyref/dRA
   refm[1][1] = wcsdata[0]/x;   // dyref/dDec

   // compute transformations for non-reference images
   printf("Alignment estimates from WCS header\n");
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) {
	 if (img!=refimg) convertWCS1_img(img,refimg,tref,sref,refm,outstr[img]);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) {
	 if (img!=refimg) convertWCS1_img(img,refimg,tref,sref,refm,outstr[img]);
      }
   for (img=0;img<Nimg;img++) if (img!=refimg) printf("%s",outstr[img]);
   fflush(stdout);
}

// Convert from parameterized alignment to plain cubic coefficients
double fitFwdAlign4(int img) {
   float p[20],**xi,rv,rval;
   int i,j;

   xi=matrix(1,20,1,20);
   for (i=0;i<20;i++) p[i] = dpos[img][i+20]*Align45scale[i%10];
   //printf("rmsi = %f, ",0.2*sqrt(aminfunc4(p-1)));
   for (i=1;i<=20;i++) {
      for (j=1;j<=20;j++) xi[i][j]=0.;
      xi[i][i]=1.;
   }
   powell(p-1,xi,20,1.e-8,&i,&rv,&aminfunc4,img);
   rval = aminfunc4(p-1,img);
   //printf("rmsf = %f\n",0.2*sqrt(aminfunc4(p-1)));
   for (i=0;i<20;i++) dpos[img][i+20] = p[i]/Align45scale[i%10];
   free_matrix(xi,1,20,1,20);

   return 0.2*sqrt(rval);
}

void convertAlign4_img(int img) {
   dpostype tmpdpos;
   double c,s,d2,m,u,v,w;

   memcpy(tmpdpos,dpos[img],sizeof(dpostype));
   d2 = 4.*dpos[img][3]/3./(GetGlobalInt(X)*GetGlobalInt(X)+GetGlobalInt(Y)*GetGlobalInt(Y));
   c=cos(M_PI/180.*dpos[img][4]);
   s=sin(M_PI/180.*dpos[img][4]);
	 
   // reference -> frame (reverse), X calculation
   dpos[img][0] = tmpdpos[0]; // constant
   dpos[img][1] = c*tmpdpos[2]; // x term
   dpos[img][2] = -s*tmpdpos[2]; // y term
   dpos[img][3] = 0; // xx term
   dpos[img][4] = 0; // xy term
   dpos[img][5] = 0; // yy term
   dpos[img][6] = c*tmpdpos[2]*d2; // xxx term
   dpos[img][7] = -s*tmpdpos[2]*d2; // xxy term
   dpos[img][8] = c*tmpdpos[2]*d2; // xyy term
   dpos[img][9] = -s*tmpdpos[2]*d2; // yyy term
   // reference -> frame (reverse), Y calculation
   dpos[img][10] = tmpdpos[1]; // constant
   dpos[img][11] = s*tmpdpos[2]; // x term
   dpos[img][12] = c*tmpdpos[2]; // y term
   dpos[img][13] = 0; // xx term
   dpos[img][14] = 0; // xy term
   dpos[img][15] = 0; // yy term
   dpos[img][16] = s*tmpdpos[2]*d2; // xxx term
   dpos[img][17] = c*tmpdpos[2]*d2; // xxy term
   dpos[img][18] = s*tmpdpos[2]*d2; // xyy term
   dpos[img][19] = c*tmpdpos[2]*d2; // yyy term

   m = d2/tmpdpos[2]/tmpdpos[2]/tmpdpos[2];
   u=tmpdpos[0]*tmpdpos[0]+tmpdpos[1]*tmpdpos[1];
   v=c*tmpdpos[0]+s*tmpdpos[1];
   w=-s*tmpdpos[0]+c*tmpdpos[1];
   dpos[img][20] = -v*(1/tmpdpos[2]-u*m); // const
   dpos[img][21] = c/tmpdpos[2]-(2*v*tmpdpos[0]+c*u)*m; // *x
   dpos[img][22] = s/tmpdpos[2]-(2*v*tmpdpos[1]+s*u)*m; // *y
   dpos[img][23] = (s*tmpdpos[1]+3*c*tmpdpos[0])*m; // *xx
   dpos[img][24] = (2*s*tmpdpos[0]+2*c*tmpdpos[1])*m; // *xy
   dpos[img][25] = (c*tmpdpos[0]+3*s*tmpdpos[1])*m; // *yy
   dpos[img][26] = -c*m; // *xxx
   dpos[img][27] = -s*m; // *xxy
   dpos[img][28] = -c*m; // *xyy
   dpos[img][29] = -s*m; // *yyy
   dpos[img][30] = -w*(1/tmpdpos[2]-m*u); // const
   dpos[img][31] = -s/tmpdpos[2]-(2*w*tmpdpos[0]-s*u)*m; // *x
   dpos[img][32] = c/tmpdpos[2]-(2*w*tmpdpos[1]+c*u)*m; // *y
   dpos[img][33] = (w-2*s*tmpdpos[0])*m; // *xx
   dpos[img][34] = (2*c*tmpdpos[0]-2*s*tmpdpos[1])*m; // *xy
   dpos[img][35] = (w+2*c*tmpdpos[1])*m; // *yy
   dpos[img][36] = s*m; // *xxx
   dpos[img][37] = -c*m; // *xxy
   dpos[img][38] = s*m; // *xyy
   dpos[img][39] = -c*m; // *yyy

   // Refine solution
   fitFwdAlign4(img);
}

void convertAlign4(void) {
   int img;

#ifdef DOLPHOT_THREADED
   if (Timg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Timg;img++) convertAlign4_img(img);
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Timg;img++) convertAlign4_img(img);
}

void setRef2Img_img(int img) {
   int i,j;
   float p[20],**xi,rv,d;

   xi=matrix(1,20,1,20);
   d = ref2img[img][1]*ref2img[img][12] - ref2img[img][2]*ref2img[img][11];
   memset(p,0,sizeof(p));
   p[1] = ref2img[img][12]/d*Align45scale[1];
   p[2] = -ref2img[img][2]/d*Align45scale[2];
   p[11] = -ref2img[img][11]/d*Align45scale[1];
   p[12] = ref2img[img][1]/d*Align45scale[2];
   //printf("rmsi = %f, ",0.2*sqrt(aminfuncR2I(p-1)));
   for (i=1;i<=20;i++) {
      for (j=1;j<=20;j++) xi[i][j]=0.;
      xi[i][i]=1.;
   }
   powell(p-1,xi,20,1.e-8,&i,&rv,&aminfuncR2I,img);
   //printf("rmsf = %f\n",0.2*sqrt(aminfuncR2I(p-1)));
   for (i=0;i<20;i++) ref2img[img][i+20] = p[i]/Align45scale[i%10];
   free_matrix(xi,1,20,1,20);
}

void setRef2Img(void) {
   int img;

#ifdef DOLPHOT_THREADED
   if (Timg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Timg;img++) setRef2Img_img(img);
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Timg;img++) setRef2Img_img(img);
}

#define ALIGN_SEARCH_MODE 1
void alignimg(int ext,int fld,int img,int nStep,char*outstr) {
   int it,i,j,x,dx0,dy0,Nalign0;
   float xoff=0,yoff=0,sig=0;
   float **xi,*v;
   double dx,dy,rsig=0.0;
   char *outptr=outstr;

   // find any large offset
   if (nStep>0) {
      int nlist=0,nn;
      float *xlist,*ylist;
      int r,n,nmax=-1,xmax=0,ymax=0;
      xlist = (float*)calloc(Nstars*(2*nStep+1)*(2*nStep+1),FLOATSIZE);
      ylist = (float*)calloc(Nstars*(2*nStep+1)*(2*nStep+1),FLOATSIZE);
      for (i=0;i<Nstars;i++) {
	 double sx0,sy0;
	 shift(img,stars[i].x,stars[i].y,&sx0,&sy0,1);
	 if (posOK(img,sx0,sy0)) {
#if ALIGN_SEARCH_MODE==1
	    int err;
	    float best=0,snr;
	    for (err=0;err<=nStep && best==0;err++) {
	       for (dx0=-err;dx0<=err;dx0++) {
		  dy0=err-abs(dx0);
		  while (1) {
		     shift(img,sx0+dx0*GetParamDouble(AlignStep),sy0+dy0*GetParamDouble(AlignStep),&dx,&dy,-1);
		     snr = alphot(img,dx,dy,&dx,&dy,0,0);
		     if (snr>0.0) {
			shift(img,dx,dy,&dx,&dy,1);
			dx -= sx0;
			dy -= sy0;
			snr *= exp( -0.5*(dx*dx+dy*dy)/(GetParamDouble(AlignStep)*GetParamDouble(AlignStep)) );
			if (snr>best) {
			   best = snr;
			   xlist[nlist] = dx;
			   ylist[nlist] = dy;
			}
		     }
		     if (dy0<=0) break;
		     dy0 = -dy0;
		  }
	       }
	    }
	    if (best>0) nlist++;
#else
	    int nlist0;
	    nlist0=nlist;
	    for (dx0=-nStep;dx0<=nStep;dx0++) for (dy0=-nStep;dy0<=nStep;dy0++) {
	       int ix,iy;
	       ix = (int)(sx0+dx0+100)-100;
	       iy = (int)(sy0+dy0+100)-100;
	       if (posOK(img,ix,iy) && peak2(img,ix,iy)) {
		  shift(img,sx0+dx0,sy0+dy0,&dx,&dy,-1);
		  if (alphot(img,dx,dy,&dx,&dy,0,0)) {
		     int newStar=1;
		     shift(img,dx,dy,&dx,&dy,1);
		     dx -= sx0;
		     dy -= sy0;
		     for (j=nlist0;j<nlist && newStar==1;j++) {
			if (fabs(xlist[j]-dx)<0.5 && fabs(ylist[j]-dy)<0.5) newStar=0;
		     }
		     if (newStar==1) {
			xlist[nlist] = dx;
			ylist[nlist] = dy;
			nlist++;
		     }
		  }
	       }
	    }
#endif	
	 }
      }
      r = (int)(nStep*GetParamDouble(AlignStep)+3.5);
      for (dy0=-r;dy0<=r;dy0++) for (dx0=-r;dx0<=r;dx0++) {
	 n=0;
	 for (i=0;i<nlist;i++) {
	    if (fabs(xlist[i]-dx0)<=0.5 && fabs(ylist[i]-dy0)<=0.5) n+=4;
	    else if (fabs(xlist[i]-dx0)<=1.5 && fabs(ylist[i]-dy0)<=1.5) n++;
	 }
	 if (n>nmax) {
	    xmax=dx0;
	    ymax=dy0;
	    nmax=n;
	 }
	 //printf("%3d ",n); if (dx0==r) printf("\n");
      }
      nn=0;
      for (i=0;i<nlist;i++) {
	 if (xlist[i]>=xmax-1.5 && xlist[i]<=xmax+1.5 && ylist[i]>=ymax-1.5 && ylist[i]<=ymax+1.5) {
	    float z;
	    z = xlist[i];
	    for (j=nn;j>0 && z<xlist[j-1];j--) xlist[j]=xlist[j-1];
	    xlist[j] = z;
	    z = ylist[i];
	    for (j=nn;j>0 && z<ylist[j-1];j--) ylist[j]=ylist[j-1];
	    ylist[j] = z;
	    nn++;
	 }
      }
      if (nn%2==1) {
	 xoff = xlist[nn/2];
	 yoff = ylist[nn/2];
      }
      else if (nn>0) {
	 xoff = 0.5*(xlist[nn/2-1]+xlist[nn/2]);
	 yoff = 0.5*(ylist[nn/2-1]+ylist[nn/2]);
      }
      //printf("Mode %d %d (n=%d); median %f %f (n=%d)\n",xmax,ymax,nlist,xoff,yoff,nn);
      outptr += sprintf(outptr,"image %d:\n",img+1);
      outptr += sprintf(outptr,"  coarse alignment = %4.2f %4.2f\n",xoff,yoff);
      free(xlist);
      free(ylist);
      if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
      {
	 fprintf(fverb,"Coarse align image %d: %f %f\n",img+1,xoff,yoff);
	 fflush(fverb);
      }
   }

   // fine alignment
#if ALIGN_SEARCH_MODE==1
   Alist[img]=(dptype*)calloc(Nstars,sizeof(dptype));
#else
   Alist[img]=(dptype*)calloc(Nstars*(2*nStep+1)*(2*nStep+1),sizeof(dptype));
#endif
   if (!Alist[img]) merr();
   v=vector(1,NFREE);
   xi=matrix(1,NFREE,1,NFREE);
   for (it=0;it<GetParamInt(AlignIter);it++) {
      Nalign[img]=0;
      for (i=0;i<Nstars;i++) {
	 shift(img,stars[i].x,stars[i].y,&dx,&dy,1);
	 dx+=xoff;
	 dy+=yoff;
	 if (posOK(img,dx,dy)) {
	    shift(img,dx,dy,&dx,&dy,-1);
	    if (alphot(img,dx,dy,&dx,&dy,0,1)>0) {
	       shift(img,dx,dy,&dx,&dy,1);
	       Alist[img][Nalign[img]][0]=stars[i].x;
	       Alist[img][Nalign[img]][1]=stars[i].y;
	       Alist[img][Nalign[img]][2]=dx;
	       Alist[img][Nalign[img]][3]=dy;
	       Nalign[img]++;
	    }
	 }
      }
      if (it==GetParamInt(AlignIter)-1) {
	 if (nStep==0) outptr += sprintf(outptr,"image %d: %d matched, ",img+1,Nalign[img]);
	 else outptr += sprintf(outptr,"  %d matched, ",Nalign[img]);
      }
      Nalign0 = Nalign[img];
      if (Nalign[img]) {
	 do {
	    int nit=0;
	    float rval=0.;

	    for (i=1;i<=NFREE;i++) {
	       for (j=1;j<=NFREE;j++) xi[i][j]=0.;
	       xi[i][i]=1.;
	    }
	    if (GetParamInt(Align)==4) {
	       for (i=0;i<NFREE;i++) v[i+1]=dpos[img][i]*Align45scale[i%10];
	    }
	    else {
	       for (i=0;i<NFREE;i++) v[i+1]=dpos[img][i];
	       if (GetParamInt(Rotate)) v[NFREE]=dpos[img][4];
	    }
	    powell(v,xi,NFREE,1.e-5,&nit,&rval,&aminfunc,img);
	    if (GetParamInt(Align)==4) {
	       for (i=0;i<NFREE;i++) dpos[img][i]=v[i+1]/Align45scale[i%10];
	    }
	    else {
	       for (i=0;i<=GetParamInt(Align);i++) dpos[img][i]=v[i+1];
	       if (GetParamInt(Rotate)) dpos[img][4]=v[GetParamInt(Align)+2];
	    }
	    sig=0;
	    for (i=0;i<Nalign[img];i++) {
	       shift(img,Alist[img][i][0],Alist[img][i][1],&dx,&dy,1);
	       sig+=(Alist[img][i][4]=(Alist[img][i][2]-dx)*(Alist[img][i][2]-dx)+(Alist[img][i][3]-dy)*(Alist[img][i][3]-dy));
	    }
	    sig/=Nalign[img];
	    if (sig<0.0001) sig=0.0001;
	    x=0;
	    for (i=0;i<Nalign[img];i++) {
	       if (Alist[img][i][4]>sig*5.0625) {
		  Nalign[img]--;
		  memcpy(Alist[img][i],Alist[img][Nalign[img]],sizeof(dptype));
		  x=1;
		  i--;
	       }
	    }
	 } while (Nalign[img] && x);
      }
      if (GetParamInt(Align)>=4) rsig=fitFwdAlign4(img);
      if (it==GetParamInt(AlignIter)-1) {
	 if (Nalign[img]==0) {
	    char imgstr[161];
	    getimgstr(img,imgstr);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    {
	       fprintf(fwarn,"No alignment stars matched for image %d, %s\n",img+1,imgstr);
	       fflush(fwarn);
	    }
	 }
	 else if (Nalign[img]==Nalign0) {
	    char imgstr[161];
	    getimgstr(img,imgstr);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    {
	       fprintf(fwarn,"No alignment stars trimmed for image %d, %s\n",img+1,imgstr);
	       fflush(fwarn);
	    }
	 }
	 else if (Nalign[img]<50 && Nalign[img]<Nstars) {
	    char imgstr[161];
	    getimgstr(img,imgstr);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    {
	       fprintf(fwarn,"Only %d alignment stars used for image %d, %s\n",Nalign[img],img+1,imgstr);
	       fflush(fwarn);
	    }
	 }
	 if (hstmode[img].inst!=NONE && sqrt(sig)>=0.5) {
	    char imgstr[161];
	    getimgstr(img,imgstr);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    {
	       fprintf(fwarn,"Large alignment scatter (%5.3f pix) for image %d, %s\n",sqrt(sig),img+1,imgstr);
	       fflush(fwarn);
	    }
	 }
	 if (GetParamInt(Align)<4) {
	    outptr += sprintf(outptr,"%d used, %4.2f %4.2f %8.6f %7.5f %5.3f, sig=%4.2f\n",Nalign[img],dpos[img][0],dpos[img][1],dpos[img][2],dpos[img][3],dpos[img][4],sqrt(sig));
	    if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    {
	       fprintf(fverb,"Align image %d: %d ",img+1,Nalign0);
	       fprintf(fverb,"%d %f %f %f %f %f %f\n",Nalign[img],dpos[img][0],dpos[img][1],dpos[img][2],dpos[img][3],dpos[img][4],sqrt(sig));
	       fflush(fverb);
	    }
	 }
	 else {
	    outptr += sprintf(outptr,"%d used, sig=%4.2f, rsig=%5.3f\n",Nalign[img],sqrt(sig),rsig);
	    if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	    {
	       fprintf(fverb,"Align image %d: %d ",img+1,Nalign0);
	       fprintf(fverb,"%d %f %f\n",Nalign[img],sqrt(sig),rsig);
	       fflush(fverb);
	    }
	 }
	 if (falign!=NULL) {
	    for (i=0;i<Nalign[img];i++) {
	       shift(img,Alist[img][i][0],Alist[img][i][1],&dx,&dy,1);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	       {
		  fprintf(falign,"%2d %d %d %7.2f %7.2f %8.3f %8.3f %8.3f %8.3f\n",img+1,ext,fld+1,Alist[img][i][0],Alist[img][i][1],dx,dy,Alist[img][i][2],Alist[img][i][3]);
		  fflush(falign);
	       }
	    }
	 }
#ifdef PGPLOT
	 if (GetParamChar(DiagPlotType)!=0 && Nalign[img]>0) {
	    alignDiagData[img][DIAG_EXT][DIAG_Z].N = Nalign[img];
	    alignDiagData[img][DIAG_EXT][DIAG_Z].XMIN = GetGlobalInt(XMIN);
	    alignDiagData[img][DIAG_EXT][DIAG_Z].XMAX = GetGlobalInt(XMAX);
	    alignDiagData[img][DIAG_EXT][DIAG_Z].YMIN = GetGlobalInt(YMIN);
	    alignDiagData[img][DIAG_EXT][DIAG_Z].YMAX = GetGlobalInt(YMAX);
	    alignDiagData[img][DIAG_EXT][DIAG_Z].sig = sqrt(sig);
	    alignDiagData[img][DIAG_EXT][DIAG_Z].data = (alignDiagDataType*)calloc(Nalign[img],sizeof(alignDiagDataType));
	    if (!alignDiagData[img][DIAG_EXT][DIAG_Z].data) merr();
	    for (i=0;i<Nalign[img];i++) {
	       shift(img,Alist[img][i][0],Alist[img][i][1],&dx,&dy,1);
	       alignDiagData[img][DIAG_EXT][DIAG_Z].data[i].x = Alist[img][i][0];
	       alignDiagData[img][DIAG_EXT][DIAG_Z].data[i].y = Alist[img][i][1];
	       alignDiagData[img][DIAG_EXT][DIAG_Z].data[i].dx = Alist[img][i][2]-dx;
	       alignDiagData[img][DIAG_EXT][DIAG_Z].data[i].dy = Alist[img][i][3]-dy;
	    }
	 }
#endif
      }
   }
   free_matrix(xi,1,NFREE,1,NFREE);
   free_vector(v,1,NFREE);
   free(Alist[img]);
}

void align(int ext,int fld) {
   int img,x,y,WARMSAVE,FORCE1SAVE,SEARCHMODESAVE,e,c,t,nStep;
   float p,sn,sigmin,sigmax;
   double dx,dy;
   FILE *f;
   char str[1024];
   char outstr[MAXNIMG][1024];

   WARMSAVE=GetGlobalInt(WARMSTART);
   SetGlobal_NOT_THREADSAFE(WARMSTART,0);
   FORCE1SAVE=GetParamInt(Force1);
   SetParam_NOT_THREADSAFE(Force1,0);
   SEARCHMODESAVE=GetParamInt(SearchMode);
   SetParam_NOT_THREADSAFE(SearchMode,0);
   if (Timg>Nimg) img=Nimg;
   else img=0;
   Nstars=0;
   if (GetParamChar(alignstars)!=0) {
      if ((f=fopen(GetParamString(alignstars),"r"))!=NULL) {
	 setindx(-1);
	 for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) {
	    memset(ran[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
	 }
 	 while (fscanf(f,"%d %d %lf %lf",&e,&c,&dx,&dy)!=EOF && Nstars<MAXNSTARS) {
	    if (e==ext && c==fld+1) alphot(img,dx,dy,&dx,&dy,2,1);
	 }
      }
      if (Nstars==0) printf("No stars in alignstars list found; using default search\n");
   }
   else {
      sigmax=1.e20;
      sigmin=200.;
      if (Nstars==0 && GetParamChar(xytfile)!=0) while (sigmin>GetParamDouble(SigAlign) && Nstars<1000) {
	 sigmin*=0.5;
	 if (sigmin<GetParamDouble(SigAlign)) sigmin=GetParamDouble(SigAlign);
	 if ((f=fopen(GetParamString(xytfile),"r"))!=NULL) {
	    while (fscanf(f,"%d %d %lf %lf %d %f",&e,&c,&dx,&dy,&t,&sn)!=EOF && Nstars<MAXNSTARS) {
	       if (e==ext && c==fld+1 && t==1 && dx>=GetGlobalInt(XMIN) && dx<GetGlobalInt(XMAX) && dy>=GetGlobalInt(YMIN) && dy<GetGlobalInt(YMAX) && sn>=sigmin && sn<sigmax) {
		  stars[Nstars].x=dx;
		  stars[Nstars].y=dy;
		  stars[Nstars].passID=-1;
		  Nstars++;
	       }
	       fgets(str,1024,f);
	    }
	    fclose(f);
	 }
	 sigmax=sigmin;
       }
   }
   SetGlobal_NOT_THREADSAFE(NO_SKY,1);
   if (Nstars==0) {
      int y,first=1;
      setindx(-1);
      for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) {
	 memset(ran[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
      }
      clearsnmap();
      setsnmap(img,GetGlobalInt(XMIN),GetGlobalInt(XMAX),GetGlobalInt(YMIN),GetGlobalInt(YMAX));
      sigmax=1.e20;
      sigmin=200.;
      while ((first || sigmin>GetParamDouble(SigAlign)) && Nstars<1000) {
	 first=0;
	 sigmin*=0.7;
	 if (sigmin<GetParamDouble(SigAlign)) sigmin=GetParamDouble(SigAlign);
	 for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef)+1;y<GetGlobalInt(YMAX)*GetParamInt(SubResRef)-1;y++) for (x=GetGlobalInt(XMIN)*GetParamInt(SubResRef)+1;x<GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1;x++) if ((p=peaksn(img,x,y))>=sigmin && p<sigmax) alphot(img,(x+0.5)/GetParamInt(SubResRef),(y+0.5)/GetParamInt(SubResRef),&dx,&dy,1,1);
	 sigmax=sigmin;
      }
   }
   if (Nstars<500) {
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
      {
	 fprintf(fwarn,"Only %d stars for alignment\n",Nstars);
	 fflush(fwarn);
      }
   }
   printf("%d stars for alignment\n",Nstars);
   fflush(stdout);
   if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
   {
      fprintf(fverb,"Align: %d\n",Nstars);
      fflush(fverb);
   }
   setindx(-1);
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) {
      memset(ran[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
   }
   if (GetParamInt(Align)==4) NFREE=20;
   else {
      NFREE=1+GetParamInt(Align);
      if (GetParamInt(Rotate)) NFREE++;
   }
#if ALIGN_SEARCH_MODE==1
   nStep=(int)(GetParamDouble(AlignTol)/GetParamDouble(AlignStep)+0.5);
#else
   GetParamDouble(AlignStep)=1.0;
   nStep=(int)(GetParamDouble(AlignTol)+0.5);
#endif
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) if (img || Timg>Nimg) {
	    alignimg(ext,fld,img,nStep,outstr[img]);
      }
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) if (img || Timg>Nimg) {
	    alignimg(ext,fld,img,nStep,outstr[img]);
      }
   for (img=0;img<Nimg;img++) if (img || Timg>Nimg) printf("%s",outstr[img]);
   fflush(stdout);
   SetGlobal_NOT_THREADSAFE(NO_SKY,0);
   SetGlobal_NOT_THREADSAFE(WARMSTART,WARMSAVE);
   SetParam_NOT_THREADSAFE(Force1,FORCE1SAVE);
   SetParam_NOT_THREADSAFE(SearchMode,SEARCHMODESAVE);
   return;
}

static inline double getscale(int img,double X,double Y) {
   double x0,y0,x,y;
   float scale,maxscale;
   shift(img,X,Y,&x0,&y0,-1);
   shift(img,X+1,Y,&x,&y,-1);
   maxscale = sqrt((x-x0)*(x-x0)+(y-y0)*(y-y0));
   shift(img,X,Y+1,&x,&y,-1);
   scale = sqrt((x-x0)*(x-x0)+(y-y0)*(y-y0));
   if (scale>maxscale) maxscale=scale;
   return maxscale;
}

void setRMark(void) {
   int img,r,xmax,ymax;
   float scale,maxscale;
   rMark=0;
   rMarkPSF=0;
   for (img=0;img<Nimg;img++) {
      xmax=dataim[img].X-1;
      ymax=dataim[img].Y-1;
      maxscale=getscale(img,0,0);
      scale=getscale(img,xmax,0); if (scale>maxscale) maxscale=scale;
      scale=getscale(img,0,ymax); if (scale>maxscale) maxscale=scale;
      scale=getscale(img,xmax,ymax); if (scale>maxscale) maxscale=scale;
      scale=getscale(img,xmax*0.5,ymax*0.5); if (scale>maxscale) maxscale=scale;
      r=(int)((rphot[img]+RPSF[img])*maxscale+0.9999);
      if (r>rMark) rMark=r;
      r=(int)(RPSF[img]*maxscale+0.9999);
      if (r>rMarkPSF) rMarkPSF=r;
   }
}

void find(int show) {
   int xx,x,y,nf=0,nf0=0,i;
   float p,sigmax=1.e30,sigmin=200,fsigm;

   int SEARCHMODESAVE=GetParamInt(SearchMode);
   SetParam_NOT_THREADSAFE(SearchMode,0);
   if (show) {printf("Finding stars: "); fflush(stdout);}
   setindx(-1);
   fsigm=GetParamDouble(SigFind)*GetParamDouble(SigFindMult);
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) memset(ran[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
   clearsnmap();
   SetGlobal_NOT_THREADSAFE(NO_SKY,1);
   setsnmap(-1,GetGlobalInt(XMIN),GetGlobalInt(XMAX),GetGlobalInt(YMIN),GetGlobalInt(YMAX));
   SetGlobal_NOT_THREADSAFE(NO_SKY,0);
   if (show) {printf("."); fflush(stdout);}
   while (sigmin>fsigm) {
      sigmin*=0.5;
      if (sigmin<fsigm) sigmin=fsigm;
      for (x=GetGlobalInt(XMIN)*GetParamInt(SubResRef)+1;x<GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1;x++) for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef)+1;y<GetGlobalInt(YMAX)*GetParamInt(SubResRef)-1;y++) if ((p=qpeak(x,y,1))>=sigmin && p<sigmax) {
	 phot((x+0.5)/(double)GetParamInt(SubResRef),(y+0.5)/(double)GetParamInt(SubResRef),-1,0);
	 for (;nf<Nstars;nf++) imgsub(nf);
      }
      sigmax=sigmin;
   }
   for (i=0;i<GetParamInt(SecondPass);i++) {
      if (show) {printf("."); fflush(stdout);}
      for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) {
	 memset(ran[y]+GetGlobalInt(XMIN)*GetParamInt(SubResRef),0,(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef));
      }
      for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) for (x=GetGlobalInt(XMIN)*GetParamInt(SubResRef);x<GetGlobalInt(XMAX)*GetParamInt(SubResRef);x++) if (indx[y][x]>=0) {
	 ran[y][x]=1;
	 if (GetParamDouble(RCombine)>=1.9) {
	    if (y>GetGlobalInt(YMIN)*GetParamInt(SubResRef)) ran[y-1][x]=1;
	    if (y<GetGlobalInt(YMAX)*GetParamInt(SubResRef)-1) ran[y+1][x]=1;
	    if (x>GetGlobalInt(XMIN)*GetParamInt(SubResRef)) ran[y][x-1]=1;
	    if (x<GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1) ran[y][x+1]=1;
	 }
	 if (GetParamDouble(RCombine)>=2.4) {
	    if (y>GetGlobalInt(YMIN)*GetParamInt(SubResRef)) {
	       if (x>GetGlobalInt(XMIN)*GetParamInt(SubResRef)) ran[y-1][x-1]=1;
	       if (x<GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1) ran[y-1][x+1]=1;
	    }
	    if (y<GetGlobalInt(YMAX)*GetParamInt(SubResRef)-1) {
	       if (x>GetGlobalInt(XMIN)*GetParamInt(SubResRef)) ran[y+1][x-1]=1;
	       if (x<GetGlobalInt(XMAX)*GetParamInt(SubResRef)-1) ran[y+1][x+1]=1;
	    }
	 }
      }
      clearsnmapflag();
      for (xx=nf0;xx<nf;xx++) {
	 SetGlobal_NOT_THREADSAFE(NO_SKY,1);
	 setsnmap(-1,(int)stars[xx].x-rMark-1,(int)stars[xx].x+rMark+2,(int)stars[xx].y-rMark-2,(int)stars[xx].y+rMark+1);
	 SetGlobal_NOT_THREADSAFE(NO_SKY,0);
	 for (y=(int)((stars[xx].y-rMarkPSF)*GetParamInt(SubResRef));y<=(int)((stars[xx].y+rMarkPSF)*GetParamInt(SubResRef));y++) if (y>=GetGlobalInt(YMIN)*GetParamInt(SubResRef) && y<GetGlobalInt(YMAX)*GetParamInt(SubResRef)) for (x=(int)((stars[xx].x-rMarkPSF)*GetParamInt(SubResRef));x<=(int)((stars[xx].x+rMarkPSF)*GetParamInt(SubResRef));x++) if (x>=GetGlobalInt(XMIN)*GetParamInt(SubResRef) && x<GetGlobalInt(XMAX)*GetParamInt(SubResRef) && qpeak(x,y,1)>=fsigm) {
	    phot((x+0.5)/(double)GetParamInt(SubResRef),(y+0.5)/(double)GetParamInt(SubResRef),-2-i,0);
	    ran[y][x]=1;
	 }
      }
      //printf("%d\n",Nstars-nf);
      nf0=nf;
      for (;nf<Nstars;nf++) imgsub(nf);
   }
   if (show) {printf(".\n%d found\n",Nstars); fflush(stdout);}
   for (x=0;x<Nstars;x++) {
      y=stars[x].flag;
      markstars(stars[x].x,stars[x].y);
      stars[x].flag=y;
   }
   clean(1,show);
   SetParam_NOT_THREADSAFE(SearchMode,SEARCHMODESAVE);
   if (GetParamInt(SearchMode)==1) for (i=0;i<Nstars;i++) stars[i].flag=1;
   return;
}

void readwarm(int ext,int fld) {
   FILE *f;
   int e,c,t;
   float x,y,z;
   char str[1024];

   setindx(-1);
   if ((f=fopen(GetParamString(xytfile),"r"))==NULL) {
      printf("Cannot open %s\n",GetParamString(xytfile));
      fflush(stdout);
      return;
   }
   printf("Reading %s ...",GetParamString(xytfile));
   fflush(stdout);
   while (fscanf(f,"%d %d %f %f %d %f",&e,&c,&x,&y,&t,&z)!=EOF) {
      if (e==ext && c==fld+1) {
	 if (t==2) t=1;
	 stars[Nstars].x=x;
	 stars[Nstars].y=y;
	 stars[Nstars].type=t;
	 stars[Nstars].passID=0;
	 getpsfpars(-1,x,y,stars[Nstars].pa,stars[Nstars].pb,stars[Nstars].pc);
	 if (Timg>Nimg) getpsfpars(Nimg,x,y,stars[Nstars].pa,stars[Nstars].pb,stars[Nstars].pc);
	 phot((int)x,(int)y,Nstars,0);
	 imgsub(Nstars);
	 if (GetGlobalInt(WARMSTART)==2) fscanf(f,"%f %f",refcts+Nstars,refmag+Nstars);
	 Nstars++;
      }
      fgets(str,1024,f);
   }
   fclose(f);
   printf(" %d stars\n",Nstars);
   fflush(stdout);
   return;
   /* should readwarm() also search for new stars (pass2)? */
}

void cleansat(int img) {
   int x,y,cont=1,x0,x1,y0,y1;
   int NPIX;
   float DDMAX;
   double tx,ty;
   char **fixed,**last;
   void *ptr;

   return;
   NPIX=dataim[img].X*dataim[img].Y;
   fixed=(char**)calloc(dataim[img].Y,PTRSIZE);
   if (!fixed) merr();
   ptr=malloc(NPIX); if (!ptr) merr();
   for (y=0;y<dataim[img].Y;y++) fixed[y]=ptr+dataim[img].X*y;
   last=(char**)calloc(dataim[img].Y,PTRSIZE);
   if (!last) merr();
   ptr=malloc(NPIX); if (!ptr) merr();
   for (y=0;y<dataim[img].Y;y++) last[y]=ptr+dataim[img].X*y;
   x0=dataim[img].X;
   x1=0;
   y0=dataim[img].Y;
   y1=0;
   for (x=GetGlobalInt(XMIN);x<GetGlobalInt(XMAX);x++) {
      shift(img,x,GetGlobalInt(YMIN),&tx,&ty,1);
      if (tx<x0) x0=(int)tx;
      if (tx>x1) x1=(int)tx;
      if (ty<y0) y0=(int)ty;
      if (ty>y1) y1=(int)ty;
      shift(img,x,GetGlobalInt(YMAX),&tx,&ty,1);
      if (tx<x0) x0=(int)tx;
      if (tx>x1) x1=(int)tx;
      if (ty<y0) y0=(int)ty;
      if (ty>y1) y1=(int)ty;
   }
   for (y=GetGlobalInt(YMIN);y<GetGlobalInt(YMAX);y++) {
      shift(img,GetGlobalInt(XMIN),y,&tx,&ty,1);
      if (tx<x0) x0=(int)tx;
      if (tx>x1) x1=(int)tx;
      if (ty<y0) y0=(int)ty;
      if (ty>y1) y1=(int)ty;
      shift(img,GetGlobalInt(XMAX),y,&tx,&ty,1);
      if (tx<x0) x0=(int)tx;
      if (tx>x1) x1=(int)tx;
      if (ty<y0) y0=(int)ty;
      if (ty>y1) y1=(int)ty;
   }
   if (x0<0) x0=0;
   if (x1>dataim[img].X) x1=dataim[img].X;
   if (y0<0) y0=0;
   if (y1>dataim[img].Y) y1=dataim[img].Y;
   DDMAX=iDMAX[img]*GetParamDouble(FSat);
   memset(fixed[0],0,NPIX);
   for (y=y0;y<y1;y++) for (x=x0;x<x1;x++) if (data[img][y][x]>=DDMAX) {
      data[img][y][x]=iDMAX[img];
      if (x>0) fixed[y][x-1]=1;
      if (x<dataim[img].X-1) fixed[y][x+1]=1;
      if (y>0) fixed[y-1][x]=1;
      if (y<dataim[img].Y-1) fixed[y+1][x]=1;
   }
   while (cont) {
      cont=0;
      memcpy(last[0],fixed[0],NPIX);
      memset(fixed[0],0,NPIX);
      for (y=y0;y<y1;y++) for (x=x0;x<x1;x++) if (dataOK(img,x,y) && last[y][x] && peak2(img,x,y)) {
	 data[img][y][x]=iDMAX[img];
	 if (x>0 && data[img][y][x-1]<iDMAX[img]) fixed[y][x-1]=1;
	 if (x<dataim[img].X-1 && data[img][y][x+1]<iDMAX[img]) fixed[y][x+1]=1;
	 if (y>0 && data[img][y-1][1]<iDMAX[img]) fixed[y-1][x]=1;
	 if (y<dataim[img].Y-1 && data[img][y+1][1]<iDMAX[img]) fixed[y+1][x]=1;
	 cont=1;
      }
   }
   free(last[0]);
   free(last);
   free(fixed[0]);
   free(fixed);
   return;
}

float minfunc(float x,int N,float3*apc,float3*sig,int*flagUse,int idx) {
   int i;
   float q=0;
   int nGood=0;
   for (i=0;i<N;i++) if (flagUse[i]) {
      q+=log(1+(x-apc[i][idx])*(x-apc[i][idx])*0.5*sig[i][idx]*sig[i][idx]);
      nGood++;
   }
   return q/nGood;
}

float dminfunc(float x,int N,float3*apc,float3*sig,int*flagUse,int idx) {
   int i;
   float q=0;
   int nGood=0;
   for (i=0;i<N;i++) if (flagUse[i]) {
      q+=(x-apc[i][idx])/(1/sig[i][idx]/sig[i][idx]+0.5*(x-apc[i][idx])*(x-apc[i][idx]));
      nGood++;
   }
   return q/nGood;
}

//adapted from Numerical Recipes;
#define ITMAX 100
#define ZEPS 1.0e-10
#define MOV3(a,b,c, d,e,f) (a)=(d);(b)=(e);(c)=(f);
#define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))
float dbrent(float ax, float bx, float cx, float (*f)(float,int,float3*,float3*,int*,int),
	float (*df)(float,int,float3*,float3*,int*,int), float tol, float *xmin,int fN,float3*fapc,float3*fsig,int*flagUse,int idx)
{
	int iter,ok1,ok2;
	float a,b,d=0,d1,d2,du,dv,dw,dx,e=0.0;
	float fu,fv,fw,fx,olde,tol1,tol2,u,u1,u2,v,w,x,xm;

	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=(*f)(x,fN,fapc,fsig,flagUse,idx);
	dw=dv=dx=(*df)(x,fN,fapc,fsig,flagUse,idx);
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol1=tol*fabs(x)+ZEPS;
		tol2=2.0*tol1;
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (fabs(e) > tol1) {
			d1=2.0*(b-a);
			d2=d1;
			if (dw != dx) d1=(w-x)*dx/(dx-dw);
			if (dv != dx) d2=(v-x)*dx/(dx-dv);
			u1=x+d1;
			u2=x+d2;
			ok1 = (a-u1)*(u1-b) > 0.0 && dx*d1 <= 0.0;
			ok2 = (a-u2)*(u2-b) > 0.0 && dx*d2 <= 0.0;
			olde=e;
			e=d;
			if (ok1 || ok2) {
				if (ok1 && ok2)
					d=(fabs(d1) < fabs(d2) ? d1 : d2);
				else if (ok1)
					d=d1;
				else
					d=d2;
				if (fabs(d) <= fabs(0.5*olde)) {
					u=x+d;
					if (u-a < tol2 || b-u < tol2)
						d=SIGN(tol1,xm-x);
				} else {
					d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
				}
			} else {
				d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
			}
		} else {
			d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
		}
		if (fabs(d) >= tol1) {
			u=x+d;
			fu=(*f)(u,fN,fapc,fsig,flagUse,idx);
		} else {
			u=x+SIGN(tol1,d);
			fu=(*f)(u,fN,fapc,fsig,flagUse,idx);
			if (fu > fx) {
				*xmin=x;
				return fx;
			}
		}
		du=(*df)(u,fN,fapc,fsig,flagUse,idx);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			MOV3(v,fv,dv, w,fw,dw)
			MOV3(w,fw,dw, x,fx,dx)
			MOV3(x,fx,dx, u,fu,du)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				MOV3(v,fv,dv, w,fw,dw)
				MOV3(w,fw,dw, u,fu,du)
			} else if (fu < fv || v == x || v == w) {
				MOV3(v,fv,dv, u,fu,du)
			}
		}
	}
	printf("**Aperture corrections failed to converge!\n");
	fflush(stdout);
	*xmin=u;
	return fu;
}
#undef SIGN
#undef ITMAX
#undef ZEPS
#undef MOV3

#define APMAX 200
#define SIGCAP 10

static inline float getapsize(int img,int i) {
#ifdef USEWFPC2
   if (hstmode[img].inst==WFPC2) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return wfpc2_apsize(img,tx,ty);
   }
#endif
#ifdef USEACS
   if (hstmode[img].inst==ACS) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return acs_apsize(img,tx,ty);
   }
#endif
#ifdef USEWFC3
   if (hstmode[img].inst==WFC3) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return wfc3_apsize(img,tx,ty);
   }
#endif
#ifdef USEROMAN
   if (hstmode[img].inst==ROMAN) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return roman_apsize(img,tx,ty);
   }
#endif
#ifdef USENIRCAM
   if (hstmode[img].inst==NIRCAM) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return nircam_apsize(img,tx,ty);
   }
#endif
#ifdef USENIRISS
   if (hstmode[img].inst==NIRISS) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return niriss_apsize(img,tx,ty);
   }
#endif
#ifdef USEMIRI
   if (hstmode[img].inst==MIRI) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return miri_apsize(img,tx,ty);
   }
#endif
#ifdef USEEUCLID
   if (hstmode[img].inst==NISP) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return nisp_apsize(img,tx,ty);
   }
   if (hstmode[img].inst==VIS) {
      double tx,ty;
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      return vis_apsize(img,tx,ty);
   }
#endif
   return apsize[img];
}

/*
float calcapsky(int img,float x0,float y0,float rap,float*sigsky,float*slist) {
   int x,y,ix,iy,Nsky=0;
   float r;
   ix=(int)x0; iy=(int)y0;
   for (y=iy-(int)apsky[img][1]-1;y<=iy+(int)apsky[img][1]+1;y++) for (x=ix-(int)apsky[img][1]-1;x<=ix+(int)apsky[img][1]+1;x++) if (ppixOK(img,x,y)) {
      r=sqrt((x+0.5-x0)*(x+0.5-x0)+(y+0.5-y0)*(y+0.5-y0));
      if (r>rap+0.5 && r>apsky[img][0] && r<=apsky[img][1]) slist[Nsky++]=res[img][y][x];
   }
   return calcskyval(Nsky,slist,img,sigsky);
}
 */
float calcapsky(int img,float x0,float y0,float rap,float*sigsky,float*slist,float*vslist) {
   int x,y,ix,iy,Nsky=0;
   float r2;
   ix=(int)x0; iy=(int)y0;
   const float r2min1 = SQR(rap+0.5);
   const float r2min2 = SQR(apsky[img][0]);
   const float r2min = (r2min1>r2min2)?r2min1:r2min2;
   const float r2max = SQR(apsky[img][1]);
   x0-=0.5;
   y0-=0.5;
   for (y=iy-(int)apsky[img][1]-1;y<=iy+(int)apsky[img][1]+1;y++) for (x=ix-(int)apsky[img][1]-1;x<=ix+(int)apsky[img][1]+1;x++) if (ppixOK(img,x,y)) {
      r2=SQR(x-x0)+SQR(y-y0);
      if (r2>r2min && r2<=r2max) {
#ifdef USESKYNOISE
	 vslist[Nsky]=bgnoise(img,x,y);
#else
	 vslist[Nsky]=1.;
#endif
	 slist[Nsky++]=res[img][y][x];
      }
   }
   return calcskyval(Nsky,slist,vslist,img,sigsky);
}

void calcapcor(int ext,int fld,int img,FILE *f,int i,float rap,int*N,float3*apc,float3*sig,float*sq,int*id,float*slist,float*vslist) {
   double tx,ty;
   float xc,yc,r,ssky,sigsky,isq,spsf=0;
   float3 iapc,isig;
   int x,y,ix,iy,sOK=1;

   imgadd1(img,i);
   shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
   ssky=calcapsky(img,tx,ty,rap,&sigsky,slist,vslist);
   ix=(int)tx; xc=tx-0.5;
   iy=(int)ty; yc=ty-0.5;
   iapc[0]=iapc[1]=isig[0]=isig[1]=0;
   if (ssky<1.e30) {
      shift(img,stars[i].x,stars[i].y,&tx,&ty,1);
      calc1psf(img,tx,ty,RPSF[img],stars[i].pa[img],stars[i].pb[img],stars[i].pc[img],1,0); // want real PSF for this, not FitSky=2 subtracted
      for (x=ix-(int)(rap)-1;x<=ix+(int)(rap)+1;x++) for (y=iy-(int)(rap)-1;y<=iy+(int)(rap)+1;y++) {
	 r=0.465+rap-sqrt((x-xc)*(x-xc)+(y-yc)*(y-yc));
	 if (r>1) r=1;
	 if (r>0) {
	    // AEDDEBUG - should have "rtot", sum of "r" values.  This can be used in place of pi*rap^2 below.
	    if (abs(x-ix)<=RPSF[img] && abs(y-iy)<=RPSF[img]) spsf+=psf[img][y-iy][x-ix]*r;
	    if (ppixfOK(img,x,y)) {
	       iapc[1]+=(res[img][y][x]-ssky)*r;
	       // AEDDEBUG - should noise be incremented by r^2 instead?
	       isig[1]+=noise(img,x,y,ssky)*r;
	    }
	    // AEDDEBUG - not sure why doing this - why not skip pixel if bad, both for counting iapc[1] / isig[1] and for incrementing spsf used for iapc[1] calculation?
	    else {
	       float ns=0,rs=0;
	       int ct=0,dx,dy;
	       for (dx=-1;dx<=1;dx++) for (dy=-1;dy<=1;dy++) if (ppixfOK(img,x+dx,y+dy)) {
		  ns+=noise(img,x+dx,y+dy,ssky);
		  rs+=res[img][y+dy][x+dx];
		  ct++;
	       }
	       if (ct) {
		  iapc[1]+=(rs/ct-ssky)*r;
		  isig[1]+=ns/ct*r;
	       }
	    }
	 }
      }
   }
   if (isig[1]>0 && iapc[1]>0 && stars[i].is[img]>0) {
#ifdef SINGLE_APCOR
      isig[0]=1.0;
      iapc[0]=0.0;
      isig[1]=sqrt(isig[1]/iapc[1]/iapc[1]+stars[i].iss[img]*stars[i].iss[img]/stars[i].is[img]/stars[i].is[img])*1.0857+0.01;
      iapc[1]=-2.5*log10(iapc[1]/stars[i].is[img]*stars[i].icm[img]);
      isig[2]=0.01;
      iapc[2]=0.0;
#else
      float s,ss,ssky_dum,cs;
      int ix,iy,cx,cy;
      sky_set[img]=1;
      sky_val[img]=ssky;
      sky_unc[img]=1;
      ix=(int)(tx+100)-100;
      iy=(int)(ty+100)-100;
      cx=(int)((tx-ix)*50+0.5);
      if (cx<0) cx=0; if (cx>50) cx=50;
      cy=(int)((ty-iy)*50+0.5);
      if (cy<0) cy=0; if (cy>50) cy=50;
      eval1_apphot(img,ix,iy,cx,cy,&s,&ss,&ssky_dum,&cs,0,1,0);
      if (s<=0) sOK=0;
      else {
	 //printf("APCOR %6.2f %6.2f %f %f %f %f %f\n",stars[i].x,stars[i].y,stars[i].is[img]/stars[i].icm[img],iapc[1],spsf,ssky,rap);
	 // aperture@RAper vs. PSF-fit (both weighted by fraction of PSF)
	 isig[0]=sqrt(ss*ss/s/s+stars[i].iss[img]*stars[i].iss[img]/stars[i].is[img]/stars[i].is[img])*1.0857+0.01;
	 iapc[0]=-2.5*log10(s/stars[i].is[img]*stars[i].icm[img]);
	 // aperture@aprad vs. aperture@RAper (both weighted by fraction of PSF)
	 isig[1]=sqrt((isig[1]+9.8696044*rap*rap*rap*rap*sigsky*sigsky)/iapc[1]/iapc[1]+ss*ss/s/s)*1.0857+0.01;
	 iapc[1]=-2.5*log10(iapc[1]/spsf/s);
	 // fraction of PSF within aperture
	 isig[2]=0.01;
	 iapc[2]=-2.5*log10(spsf);
      }
#endif
      if (sOK==1) {
	 isq=1/isig[1];
	 if (stars[i].sky>=0) isq/=1+stars[i].sky*0.1;
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	 {
	    fprintf(f,"%2d %d %d %7.2f %7.2f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.2f\n",img+1,ext,fld+1,stars[i].x,stars[i].y,-2.5*log10(stars[i].is[img])+GetParamDouble(Zero),iapc[0],isig[0],iapc[1],isig[1],iapc[2],isig[2],isq);
	    fflush(f);
	 }
	 if (fabs(iapc[0])<=0.25 && fabs(iapc[1])<=0.25) {
	    isig[0]=1./isig[0];
	    isig[1]=1./isig[1];
	    isig[2]=1./isig[2];
	    for (x=(*N);x>0 && isq>sq[x-1];x--) {
	       apc[x][0]=apc[x-1][0];
	       sig[x][0]=sig[x-1][0];
	       apc[x][1]=apc[x-1][1];
	       sig[x][1]=sig[x-1][1];
	       apc[x][2]=apc[x-1][2];
	       sig[x][2]=sig[x-1][2];
	       sq[x]=sq[x-1];
	       id[x]=id[x-1];
	    }
	    apc[x][0]=iapc[0];
	    sig[x][0]=isig[0];
	    apc[x][1]=iapc[1];
	    sig[x][1]=isig[1];
	    apc[x][2]=iapc[2];
	    sig[x][2]=isig[2];
	    sq[x]=isq;
	    id[x]=i;
	    if ((*N)<APMAX) (*N)++;
	 }
      }
   }
   imgsub1(img,i);
   return;
}

void startapcor(void) {
   int i;

   printf("Aperture corrections:\n");
   fflush(stdout);
   sortstars();
   setindx(-1);
   for (i=0;i<Nstars;i++) {
      indx[(int)(stars[i].y*GetParamInt(SubResRef))][(int)(stars[i].x*GetParamInt(SubResRef))]=i;
      stars[i].flag=0;
   }
   return;
}

void getapcor(int ext,int fld,int img,FILE *f,char *outstr) {
   int N=0,i,j,x;
   float3 apc[APMAX+1],sig[APMAX+1];
   float sq[APMAX+1],mn[3],hisig[SIGCAP],rap,gmn,gsd,gsg,w;
   int flagUse[APMAX+1],id[APMAX+1];
   char *outptr=outstr;
   float *slist,*vslist;

   slist=(float*)calloc((int)(4*apsky[img][1]*apsky[img][1]+4*apsky[img][1])+1,FLOATSIZE);
   vslist=(float*)calloc((int)(4*apsky[img][1]*apsky[img][1]+4*apsky[img][1])+1,FLOATSIZE);
   if (!slist || !vslist) merr();
   if (GetParamChar(psfstars)!=0) {
      FILE *fpsf;
      int *apused=(int*)calloc(Nstars,INTSIZE); if (!apused) merr();
      if ((fpsf=fopen(GetParamString(psfstars),"r"))==NULL) {
	 outptr += sprintf(outptr,"**%s not found\n",GetParamString(psfstars));
      }
      else {
	 int pe,pc,bi;
	 double px,py,r,br,brap;
	 char str[161],*ptr,*ptr2;

	 while (fgets(str,161,fpsf)) {
	    pe=strtol(str,&ptr,10);
	    pc=strtol(ptr,&ptr,10);
	    px=strtod(ptr,&ptr)-GetParamDouble(psfoff);
	    py=strtod(ptr,&ptr2)-GetParamDouble(psfoff);
	    if (pe==ext && pc==fld+1 && ptr!=ptr2) {
	       bi=-1;
	       br=GetParamDouble(RCombine)*GetParamDouble(RCombine)/(double)(GetParamInt(SubResRef)*GetParamInt(SubResRef)); // br in units of physical pixels
	       brap=10.;
	       if (br<1) br=1;
	       for (i=0;i<Nstars;i++) if ((r=(px-stars[i].x)*(px-stars[i].x)+(py-stars[i].y)*(py-stars[i].y))<br) {
		  rap=getapsize(img,i);
		  if (goodstar(img,i,(int)rap+1,(int)rap+1,0)) {
		     br=r;
		     bi=i;
		     brap=rap;
		  }
	       }
	       if (bi>=0 && !apused[bi]) {
		  calcapcor(ext,fld,img,f,bi,brap,&N,apc,sig,sq,id,slist,vslist);
		  apused[bi]=1;
	       }
	    }
	 }
	 fclose(fpsf);
      }
      free(apused);
   }
   else for (i=0;i<Nstars;i++) {
      rap=getapsize(img,i);
      if (goodstar(img,i,(int)rap+1,(int)rap+1,2)) calcapcor(ext,fld,img,f,i,rap,&N,apc,sig,sq,id,slist,vslist);
   }
   free(slist);
   free(vslist);

   if (N<50) {
      char imgstr[161];
      getimgstr(img,imgstr);
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
      {
	 fprintf(fwarn,"Only %d aperture stars in image %d, %s\n",N,img+1,imgstr);
	 fflush(fwarn);
      }
   }
   outptr += sprintf(outptr,"image %d: %d total aperture stars\n",img+1,N);
   char verbstr[321],*verbptr=verbstr;
   if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
   {
      verbptr += sprintf(verbptr,"Apcor image %d: %d",img+1,N);
   }
   for (j=0;j<3;j++) {
      int nGood=0;
      // truncate sigma values
      for (i=0;i<SIGCAP;i++) hisig[i]=0.0;
      for (i=0;i<N;i++) {
	 for (x=SIGCAP;x>0 && sig[i][j]>hisig[x-1];x--) if (x<SIGCAP) hisig[x]=hisig[x-1];
	 if (x<SIGCAP) hisig[x]=sig[i][j];
      }
      for (x=SIGCAP-1;x>=0 && hisig[x]==0;x--);
      if (x>=0) for (i=0;i<N;i++) if (sig[i][j]>hisig[x]) sig[i][j]=hisig[x];

      // iterate apcor solution
      for (i=0;i<N;i++) {
	 flagUse[i]=1;
	 nGood++;
      }
      i=1;
      while (nGood>0 && i==1) {
	 dbrent(-0.75,0.,0.75,&minfunc,&dminfunc,0.01,mn+j,N,apc,sig,flagUse,j);
	 i=0;
	 for (x=0;x<N;x++) if (flagUse[x] && fabs(apc[x][j]-mn[j])>2./sig[x][j]+0.03) {
	    flagUse[x]=0;
	    i=1;
	    nGood--;
	 }
      }
      if (nGood<=0) mn[j]=0.;
      gmn=gsd=gsg=0.;
      if (nGood>0) {
	 for (x=0;x<N;x++) if (flagUse[x]) {
	    w=sig[x][j]*sig[x][j];
	    gmn+=apc[x][j]*w;
	    gsg+=w;
	    gsd+=apc[x][j]*apc[x][j]*w;
	 }
	 gmn/=gsg;
	 gsd/=gsg;
	 gsg=1./sqrt(gsg);
	 if (nGood>1) {
	    gsd -= gmn*gmn;
	    if (gsd<=0.) gsd=0.0;
	    else gsd = gsg*sqrt(gsd*nGood/(nGood-1.))*1.25;
	 }
	 else gsd=gsg;
      }
      outptr += sprintf(outptr,"  %d stars used, %6.3f (%5.3f +/- %5.3f, %5.3f)\n",nGood,mn[j],gmn,gsd,gsg);
      if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0) verbptr += sprintf(verbptr," %d %f %f %f %f",nGood,mn[j],gmn,gsd,gsg);
   }
   if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
   {
      fprintf(fverb,"%s\n",verbstr);
      fflush(fverb);
   }
   apcor[img]=pow(10,-0.4*(mn[0]+mn[1]+mn[2]));
#ifdef PGPLOT
   if (GetParamChar(DiagPlotType)!=0 && N>0) {
      apcorDiagData[img][DIAG_EXT][DIAG_Z].N = N;
      apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmean = mn[0]+mn[1]+mn[2];
      apcorDiagData[img][DIAG_EXT][DIAG_Z].m = (float*)calloc(N,sizeof(float));
      apcorDiagData[img][DIAG_EXT][DIAG_Z].dm = (float*)calloc(N,sizeof(float));
      getimgstr(img,apcorDiagData[img][DIAG_EXT][DIAG_Z].imgstr);
#ifdef USEWFPC2
      if (hstmode[img].inst==WFPC2) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"WFPC2 %s",WFPC2imagestring(img));
      else
#endif
#ifdef USEACS
      if (hstmode[img].inst==ACS) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"ACS %s",ACSimagestring(img));
      else
#endif
#ifdef USEWFC3
      if (hstmode[img].inst==WFC3) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"WFC3 %s",WFC3imagestring(img));
      else
#endif
#ifdef USEROMAN
      if (hstmode[img].inst==ROMAN) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"ROMAN %s",ROMANimagestring(img));
      else
#endif
#ifdef USENIRCAM
      if (hstmode[img].inst==NIRCAM) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"NIRCAM %s",NIRCAMimagestring(img));
      else
#endif
#ifdef USENIRISS
      if (hstmode[img].inst==NIRISS) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"NIRISS %s",NIRISSimagestring(img));
      else
#endif
#ifdef USEMIRI
      if (hstmode[img].inst==MIRI) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"MIRI %s",MIRIimagestring(img));
      else
#endif
#ifdef USEEUCLID
      if (hstmode[img].inst==NISP) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"NISP %s",NISPimagestring(img));
      else
      if (hstmode[img].inst==VIS) sprintf(apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr,"VIS %s",VISimagestring(img));
      else
#endif
	 apcorDiagData[img][DIAG_EXT][DIAG_Z].filtstr[0]=0;
      apcorDiagData[img][DIAG_EXT][DIAG_Z].inst = hstmode[img].inst;
      if (hstmode[img].inst==NONE) apcorDiagData[img][DIAG_EXT][DIAG_Z].filt=0;
      else apcorDiagData[img][DIAG_EXT][DIAG_Z].filt = hstmode[img].filt;
      if (!apcorDiagData[img][DIAG_EXT][DIAG_Z].m || !apcorDiagData[img][DIAG_EXT][DIAG_Z].dm) merr();
      for (i=0;i<N;i++) {
	 apcorDiagData[img][DIAG_EXT][DIAG_Z].m[i] = -2.5*log10(stars[id[i]].is[img])+GetParamDouble(Zero);
	 apcorDiagData[img][DIAG_EXT][DIAG_Z].dm[i] = apc[i][0]+apc[i][1]+apc[i][2];
	 if (i==0) {
	    apcorDiagData[img][DIAG_EXT][DIAG_Z].mmin = apcorDiagData[img][DIAG_EXT][DIAG_Z].mmax = apcorDiagData[img][DIAG_EXT][DIAG_Z].m[i];
	    apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmin = apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmax = apcorDiagData[img][DIAG_EXT][DIAG_Z].dm[i];
	 }
	 else {
	    if (apcorDiagData[img][DIAG_EXT][DIAG_Z].m[i]<apcorDiagData[img][DIAG_EXT][DIAG_Z].mmin) apcorDiagData[img][DIAG_EXT][DIAG_Z].mmin = apcorDiagData[img][DIAG_EXT][DIAG_Z].m[i];
	    else if (apcorDiagData[img][DIAG_EXT][DIAG_Z].m[i]>apcorDiagData[img][DIAG_EXT][DIAG_Z].mmax) apcorDiagData[img][DIAG_EXT][DIAG_Z].mmax = apcorDiagData[img][DIAG_EXT][DIAG_Z].m[i];
	    if (apcorDiagData[img][DIAG_EXT][DIAG_Z].dm[i]<apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmin) apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmin = apcorDiagData[img][DIAG_EXT][DIAG_Z].dm[i];
	    else if (apcorDiagData[img][DIAG_EXT][DIAG_Z].dm[i]>apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmax) apcorDiagData[img][DIAG_EXT][DIAG_Z].dmmax = apcorDiagData[img][DIAG_EXT][DIAG_Z].dm[i];
	 }
      }
   }
#endif
   return;
}
#undef APMAX

#define PASTEP 5
#define NPA 18
typedef float rndtype[NPA];
void get1round(int i,int img,rndtype rnd,rndtype wt) {
   int ix,iy,cx,cy,x1,y1,dx,dy,pa,NOSKY34=0;
   double x,y;
   float mr,mr2,trnd,ssky,s,ss,cs,fpa,s2,mx=0.,my=0.;

   shift(img,stars[i].x,stars[i].y,&x,&y,1);
   ix=(int)(x+100)-100;
   iy=(int)(y+100)-100;
   cx=(int)((x+10-(int)(x+10))*50+0.5);
   if (cx<0) cx=0; if (cx>50) cx=50;
   cy=(int)((y+10-(int)(y+10))*50+0.5);
   if (cy<0) cy=0; if (cy>50) cy=50;
   calc1psf(img,x,y,RPSF[img],stars[i].pa[img],stars[i].pb[img],stars[i].pc[img],1,1); // calculate PSF for eval() functions
   if (GetGlobalDouble(RBig)>0 && stars[i].pa[img]==GetGlobalDouble(RBig) && stars[i].pb[img]==GetGlobalDouble(RBig) && stars[i].pc[img]==0.) NOSKY34=1;
   clear_sky();
   if (GetParamInt(FitSky)==3 && !NOSKY34) {
      if (GetParamInt(PSFPhot)) eval1_psfphot_sky3(img,ix,iy,cx,cy,&s,&ss,&ssky,&cs,0);
      else eval1_apphot_sky3(img,ix,iy,cx,cy,&s,&ss,&ssky,&cs,0);
   }
   else if (GetParamInt(FitSky)==4 && !NOSKY34) {
      if (GetParamInt(PSFPhot)) eval1_psfphot_sky4(img,ix,iy,cx,cy,&s,&ss,&ssky,&mx,&my,&cs,0);
      else eval1_apphot_sky4(img,ix,iy,cx,cy,&s,&ss,&ssky,&mx,&my,&cs,0);
   }
   else if (GetParamInt(PSFPhot)) eval1_psfphot(img,ix,iy,cx,cy,&s,&ss,&ssky,&cs,0,0);
   else eval1_apphot(img,ix,iy,cx,cy,&s,&ss,&ssky,&cs,0,GetParamInt(FitSky),0);
   if (GetGlobalInt(WARMSTART)==2) s2=s+refcts[i]/refmult[img];
   else s2=s;
   if (s2<=0 || ss<=0) {
      for (pa=0;pa<NPA;pa++) {
	 if (wt) wt[pa]=0.;
	 rnd[pa]=0.;
      }
      return;
   }
   for (pa=0;pa<NPA;pa++) {
      fpa=pa*0.017453293*PASTEP;
      trnd=mr=mr2=0.;
      for (y1=iy-RPSF[img],dy=-RPSF[img];y1<=iy+RPSF[img];y1++,dy++) for (x1=ix-RPSF[img],dx=-RPSF[img];x1<=ix+RPSF[img];x1++,dx++) if ((dx!=0 || dy!=0) && dx*dx+dy*dy<(RPSF[img]+0.465)*(RPSF[img]+0.465)) {
	 if (ppixOK(img,x1,y1)) {
	    if (psf[img][dy][dx]>0) {
	       mr+=psf[img][dy][dx]*s2;
	       mr2+=res[img][y1][x1]-ssky+psf[img][dy][dx]*(s2-s);
	       trnd+=(res[img][y1][x1]-ssky-psf[img][dy][dx]*s)*cos(2*(atan2(dy,dx)-fpa));
	    }
	 }
	 else if (ppixOK(img,ix-dx,iy-dy)) {
	    if (psf[img][-dy][-dx]>0) {
	       mr+=psf[img][-dy][-dx]*s2;
	       mr2+=res[img][iy-dy][ix-dx]-ssky+psf[img][-dy][-dx]*(s2-s);
	       trnd+=(res[img][iy-dy][ix-dx]-ssky-psf[img][-dy][-dx]*s)*cos(2*(atan2(-dy,-dx)-fpa));
	    }
	 }
      }
      if (mr2>mr) mr=mr2;
      if (mr>0) {
	 rnd[pa]=trnd/mr;
	 if (wt) wt[pa]=s2/ss/ss;
      }
      else {
	 rnd[pa]=0.;
	 if (wt) wt[pa]=0.;
      }
   }
   return;
}

void getround(int i,int*pa,float*rnd,float*irnd) {
   int img,ip,pm=1;
   float tr,wt;
   rndtype *rsave,*wsave;

   rsave=(rndtype*)calloc(Nimg,sizeof(rndtype));
   wsave=(rndtype*)calloc(Nimg,sizeof(rndtype));
   if (!rsave || !wsave) merr();
   *pa=-1;
   *rnd=-1.;
   for (img=0;img<Nimg;img++) irnd[img]=-1.;
#ifdef DOLPHOT_THREADED
   if (Nimg==1 || omp_get_max_threads()<2) {
      for (img=0;img<Nimg;img++) get1round(i,img,rsave[img],wsave[img]);
   }
   else
#pragma omp parallel for schedule(guided)
#endif
      for (img=0;img<Nimg;img++) get1round(i,img,rsave[img],wsave[img]);
   for (ip=0;ip<NPA;ip++) {
      tr=wt=0.;
      for (img=0;img<Nimg;img++) {
	 tr+=rsave[img][ip]*wsave[img][ip];
	 wt+=wsave[img][ip];
      }
      if (wt>0 && (*pa<0 || fabs(tr/wt)>*rnd)) {
	 if (tr>=0) {
	    *rnd=tr/wt;
	    *pa=ip*PASTEP;
	    pm=1;
	 }
	 else {
	    *rnd=-tr/wt;
	    *pa=ip*PASTEP+90;
	    pm=-1;
	 }
	 for (img=0;img<Nimg;img++) {
	    irnd[img]=pm*rsave[img][ip];
	    if (irnd[img]<-9.999) irnd[img]=-9.999;
	    if (irnd[img]>9.999) irnd[img]=9.999;
	 }
      }
   }
   free(rsave);
   free(wsave);
   if (*rnd<-9.999) *rnd=-9.999;
   if (*rnd>9.999) *rnd=9.999;
   return;
}
#undef PASTEP
#undef NPA

void out1starinfo(void) {
   static int first=1;
   FILE *f;
   char str[161];
   int ct=0,i;

   if (!first) return;
   first=0;
   sprintf(str,"%s.columns",GetParamString(outfn));
   f=fopen(str,"w");
   if (f==0) return;
   fprintf(f,"%d. Extension (zero for base image)\n",++ct);
   fprintf(f,"%d. Chip (for three-dimensional FITS image)\n",++ct);
   fprintf(f,"%d. Object X position on reference image (or first image, if no reference)\n",++ct);
   fprintf(f,"%d. Object Y position on reference image (or first image, if no reference)\n",++ct);
   fprintf(f,"%d. Chi for fit\n",++ct);
   fprintf(f,"%d. Signal-to-noise\n",++ct);
   fprintf(f,"%d. Object sharpness\n",++ct);
   fprintf(f,"%d. Object roundness\n",++ct);
   fprintf(f,"%d. Direction of major axis (if not round)\n",++ct);
   fprintf(f,"%d. Crowding\n",++ct);
   fprintf(f,"%d. Object type (1=bright star, 2=faint, 3=elongated, 4=hot pixel, 5=extended)\n",++ct);
   if (GetParamInt(PrintPass)) fprintf(f,"%d. Pass Detected\n",++ct);
#ifdef USEWFPC2
   WFPC2outstarinfo(f,&ct);
#endif
#ifdef USEACS
   ACSoutstarinfo(f,&ct);
#endif
#ifdef USEWFC3
   WFC3outstarinfo(f,&ct);
#endif
#ifdef USEROMAN
   ROMANoutstarinfo(f,&ct);
#endif
#ifdef USENIRCAM
   NIRCAMoutstarinfo(f,&ct);
#endif
#ifdef USENIRISS
   NIRISSoutstarinfo(f,&ct);
#endif
#ifdef USEMIRI
   MIRIoutstarinfo(f,&ct);
#endif
#ifdef USEEUCLID
   NISPoutstarinfo(f,&ct);
   VISoutstarinfo(f,&ct);
#endif
   for (i=0;i<Nimg;i++) {
      char imgstr[161];
      getimgstr(i,imgstr);
      fprintf(f,"%d. Measured counts, %s\n",++ct,imgstr);
      fprintf(f,"%d. Measured sky level, %s\n",++ct,imgstr);
      fprintf(f,"%d. Normalized count rate, %s\n",++ct,imgstr);
      fprintf(f,"%d. Normalized count rate uncertainty, %s\n",++ct,imgstr);
#ifdef USEWFPC2
      if (hstmode[i].inst==WFPC2) {
	 fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USEACS
      if (hstmode[i].inst==ACS) {
	 fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USEWFC3
      if (hstmode[i].inst==WFC3) {
	 fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USEROMAN
      if (hstmode[i].inst==ROMAN) {
	 if (GetParamInt(ROMANvega)) fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 else fprintf(f,"%d. Instrumental ABMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USENIRCAM
      if (hstmode[i].inst==NIRCAM) {
	 if (GetParamInt(NIRCAMvega)) fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 else fprintf(f,"%d. Instrumental ABMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USENIRISS
      if (hstmode[i].inst==NIRISS) {
	 if (GetParamInt(NIRISSvega)) fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 else fprintf(f,"%d. Instrumental ABMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USEMIRI
      if (hstmode[i].inst==MIRI) {
	 if (GetParamInt(MIRIvega)) fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 else fprintf(f,"%d. Instrumental ABMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
#ifdef USEEUCLID
      if (hstmode[i].inst==NISP) {
	 if (GetParamInt(NISPvega)) fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 else fprintf(f,"%d. Instrumental ABMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
      if (hstmode[i].inst==VIS) {
	 fprintf(f,"%d. Instrumental VEGAMAG magnitude, %s\n",++ct,imgstr);
	 fprintf(f,"%d. Transformed UBVRI magnitude, %s\n",++ct,imgstr);
      }
      else
#endif
	 fprintf(f,"%d. Instrumental magnitude, %s\n",++ct,imgstr);
      fprintf(f,"%d. Magnitude uncertainty, %s\n",++ct,imgstr);
      fprintf(f,"%d. Chi, %s\n",++ct,imgstr);
      fprintf(f,"%d. Signal-to-noise, %s\n",++ct,imgstr);
      fprintf(f,"%d. Sharpness, %s\n",++ct,imgstr);
      fprintf(f,"%d. Roundness, %s\n",++ct,imgstr);
      fprintf(f,"%d. Crowding, %s\n",++ct,imgstr);
      if (hstmode[i].inst==NONE || hstmode[i].inst==NISP || hstmode[i].inst==VIS) { // AEDDEBUG temp
	 fprintf(f,"%d. PSF FWHM, %s\n",++ct,imgstr);
	 fprintf(f,"%d. PSF eccentricity, %s\n",++ct,imgstr);
	 fprintf(f,"%d. PSF a parameter, %s\n",++ct,imgstr);
	 fprintf(f,"%d. PSF b parameter, %s\n",++ct,imgstr);
	 fprintf(f,"%d. PSF c parameter, %s\n",++ct,imgstr);
      }
      fprintf(f,"%d. Photometry quality flag, %s\n",++ct,imgstr);
   }
   fclose(f);
}

int out1star(int ext,int fld,int i,FILE*of,int fake) {
   int j,SBAD=0,pa;
   stype t;
   double tx,ty;
   float rnd,s0;
   static float irnd[MAXNIMG],is[MAXNIMG],is0[MAXNIMG],iss[MAXNIMG],ic[MAXNIMG],ish[MAXNIMG],issky[MAXNIMG],icm[MAXNIMG];
   static photdatatype pdata[MAXNIMG];

   clear_sky();
   if (i<0) {
      t.x=t.y=0.;
      t.s=-1;
   }
   else {
      eval(-1,stars[i].x,stars[i].y,stars[i].pa,stars[i].pb,stars[i].pc,&stars[i].s,&s0,&stars[i].ss,&stars[i].chi,&stars[i].sh,&stars[i].sky,is,is0,iss,ic,ish,issky,icm,-1,i);
      getround(i,&pa,&rnd,irnd);
      memcpy(&t,stars+i,sizeof(stype));
   }
   if (i<0 || (GetGlobalInt(WARMSTART)!=2 && (stars[i].s<=0 || stars[i].s<GetParamDouble(SigFinal)*stars[i].ss)) || stars[i].x<GetParamInt(XMIN2) || stars[i].x>=GetParamInt(XMAX2) || stars[i].y<GetParamInt(YMIN2) || stars[i].y>=GetParamInt(YMAX2)) {
      if (fake || GetGlobalInt(WARMSTART)) SBAD=1;
      else return 0;
   }
   fprintf(of,"%d %d %7.2f %7.2f",ext,fld+1,t.x,t.y);
   if (t.s<=0 && !GetGlobalInt(WARMSTART)) fprintf(of,"   0.00     0.0  0.000  0.000   0 0.000 1");
   else fprintf(of," %6.2f %7.1f %6.3f %6.3f %3d %5.3f %d",t.chi,t.s/t.ss,t.sh,rnd,pa,s0,t.type);
   if (GetParamInt(PrintPass)) fprintf(of," %2d",t.passID);
   for (j=0;j<Nimg;j++) {
      pdata[j].ct0=pdata[j].ct=pdata[j].ctcorr=pdata[j].sky=pdata[j].dm=pdata[j].chi=pdata[j].sh=pdata[j].rnd=pdata[j].r=pdata[j].e=pdata[j].a=pdata[j].b=pdata[j].c=pdata[j].crowd=0.;
      pdata[j].dct=9999;
      pdata[j].dctcorr=9999;
      pdata[j].m=99.999;
      pdata[j].dm=9.999;
      shift(j,t.x,t.y,&tx,&ty,1);
      if (SBAD) pdata[j].flag=16;
      else pdata[j].flag=star_flag(stars[i].x,stars[i].y,j);
      if (!SBAD && posOK(j,tx,ty)) {
	 if (GetGlobalInt(WARMSTART) || iss[j]>0) {
	    pdata[j].ct=pdata[j].ct0=is[j]/icm[j];
	    pdata[j].dct=iss[j]/icm[j];
	    if (GetGlobalInt(WARMSTART)==2) {
	       pdata[j].ct0+=refcts[i]/refmult[j];
	       pdata[j].ct=is[j]=pdata[j].ct0*refmult[j];
	       pdata[j].dct*=refmult[j];
	       if (is[j]>0. && refcts[i]>0.) pdata[j].m=refmag[i]-2.5*log10(is[j]/refcts[i]);
	       else pdata[j].m=99.999;
	       pdata[j].ctcorr=pdata[j].ct/refcts[i]*pow(10,-0.4*refmag[i]);
	       pdata[j].dctcorr=pdata[j].dct/refcts[i]*pow(10,-0.4*refmag[i]);
	    }
	    else {
	       if (is[j]>0.) pdata[j].m=-2.5*log10(is[j])+GetParamDouble(Zero);
	       else pdata[j].m=99.999;
	       pdata[j].ctcorr=is[j]*pow(10,-0.4*GetParamDouble(Zero));
	       pdata[j].dctcorr=iss[j]*pow(10,-0.4*GetParamDouble(Zero));
	    }
	    pdata[j].crowd=is0[j];
	    pdata[j].sky=issky[j]/icm[j];
	    if (pdata[j].ct>0.) pdata[j].dm=pdata[j].dct/pdata[j].ct*1.0857362;
	    pdata[j].chi=ic[j];
	    pdata[j].sh=ish[j];
	    pdata[j].rnd=irnd[j];
	    pdata[j].a=t.pa[j];
	    pdata[j].b=t.pb[j];
	    pdata[j].c=t.pc[j];
	    getshape(pdata[j].a,pdata[j].b,pdata[j].c,&pdata[j].r,&pdata[j].e);
	 }
      }
   }
#ifdef USEWFPC2
   WFPC2outstar(of,t.x,t.y,pdata);
#endif
#ifdef USEACS
   ACSoutstar(of,t.x,t.y,pdata);
#endif
#ifdef USEWFC3
   WFC3outstar(of,t.x,t.y,pdata);
#endif
#ifdef USEROMAN
   ROMANoutstar(of,t.x,t.y,pdata);
#endif
#ifdef USENIRCAM
   NIRCAMoutstar(of,t.x,t.y,pdata);
#endif
#ifdef USENIRISS
   NIRISSoutstar(of,t.x,t.y,pdata);
#endif
#ifdef USEMIRI
   MIRIoutstar(of,t.x,t.y,pdata);
#endif
#ifdef USEEUCLID
   NISPoutstar(of,t.x,t.y,pdata);
   VISoutstar(of,t.x,t.y,pdata);
#endif
   for (j=0;j<Nimg;j++) {
      if (pdata[j].ct<999999.5) fprintf(of,"  %8.1f",pdata[j].ct);
      else fprintf(of,"  %8.2e",pdata[j].ct);
      if (pdata[j].sky<99999.95) fprintf(of," %8.2f",pdata[j].sky);
      else if (pdata[j].sky<999999.5) fprintf(of," %8.1f",pdata[j].sky);
      else fprintf(of," %8.1e",pdata[j].sky);

      if (pdata[j].ctcorr<0.0) fprintf(of," %8.1e",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<0.99) fprintf(of," %8.2e",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<9.999995) fprintf(of," %8.6f",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<99.99995) fprintf(of," %8.5f",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<999.9995) fprintf(of," %8.4f",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<9999.995) fprintf(of," %8.3f",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<99999.95) fprintf(of," %8.2f",pdata[j].ctcorr);
      else if (pdata[j].ctcorr<999999.5) fprintf(of," %8.1f",pdata[j].ctcorr);
      else fprintf(of," %8.1e",pdata[j].ctcorr);

      if (pdata[j].dctcorr<0.0) fprintf(of," %8.1e",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<0.99) fprintf(of," %8.2e",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<9.999995) fprintf(of," %8.6f",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<99.99995) fprintf(of," %8.5f",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<999.9995) fprintf(of," %8.4f",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<9999.995) fprintf(of," %8.3f",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<99999.95) fprintf(of," %8.2f",pdata[j].dctcorr);
      else if (pdata[j].dctcorr<999999.5) fprintf(of," %8.1f",pdata[j].dctcorr);
      else fprintf(of," %8.1e",pdata[j].dctcorr);

#ifdef USEWFPC2
      if (hstmode[j].inst==WFPC2) WFPC2outstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USEACS
      if (hstmode[j].inst==ACS) ACSoutstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USEWFC3
      if (hstmode[j].inst==WFC3) WFC3outstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USEROMAN
      if (hstmode[j].inst==ROMAN) ROMANoutstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USENIRCAM
      if (hstmode[j].inst==NIRCAM) NIRCAMoutstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USENIRISS
      if (hstmode[j].inst==NIRISS) NIRISSoutstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USEMIRI
      if (hstmode[j].inst==MIRI) MIRIoutstarimg(j,of,t.x,t.y,pdata);
      else
#endif
#ifdef USEEUCLID
      if (hstmode[j].inst==NISP) NISPoutstarimg(j,of,t.x,t.y,pdata);
      else
      if (hstmode[j].inst==VIS) VISoutstarimg(j,of,t.x,t.y,pdata);
      else
#endif
      {
	 fprintf(of," %6.3f",pdata[j].m);
	 if (pdata[j].dm>9.999) fprintf(of," 9.999");
	 else fprintf(of," %5.3f",pdata[j].dm);
      }

      fprintf(of," %6.2f %7.1f %6.3f %6.3f %5.3f",pdata[j].chi,pdata[j].ct/pdata[j].dct,pdata[j].sh,pdata[j].rnd,pdata[j].crowd);
      if (hstmode[j].inst==NONE || hstmode[j].inst==NISP || hstmode[j].inst==VIS) fprintf(of," %6.3f %6.3f %6.3f %6.3f %6.3f",pdata[j].r,pdata[j].e,pdata[j].a,pdata[j].b,pdata[j].c); // AEDDEBUG temp
      fprintf(of," %2d",pdata[j].flag);
   }
   fprintf(of,"\n");
   return 1;
}

typedef struct {double x,y,sn; int t;} realstartype;

// GLOBAL (PARALLEL-SAFE) VARIABLES
chiptype *data0; // data0 allocated outside of parallel code for each image
int Nrealstar; // Nrealstar set outside of parallel code
realstartype *realstar; // realstar set outside of parallel code

static inline int bestmatch(double x0,double y0) {
   int i=-1,j;
   float rmin,r;

   rmin=SQR(GetParamDouble(FakeMatch));
   for (j=0;j<Nstars;j++) {
      r=SQR(stars[j].x-x0)+SQR(stars[j].y-y0);
      if (r<rmin) {i=j; rmin=r;}
   }
   return i;
}

void fakestar(int ext,int z,double x0,double y0,double xw,double yw,int tw,double *ct0) {
   int img,ix,iy,x1,y1,XMIN0,XMAX0,YMIN0,YMAX0,XMIN20,XMAX20,YMIN20,YMAX20,i,j,it,cont,xmin,xmax,xsize,ymin,ymax,cx,cy,nbg,anyPosOK,wcsminmax_save,anyCornerOK;
   float pa,pb,pc;
   double x,y,r,rmin,SNRIN=0.;
   static float bg[MAXNIMG];

   // eliminate stars with illegal positions
   if (x0<GetParamInt(XMIN2) || x0>GetParamInt(XMAX2) || y0<GetParamInt(YMIN2) || y0>GetParamInt(YMAX2)) return;
   anyPosOK = 0;
   for (img=0;img<Nimg;img++) {
      shift(img,x0,y0,&x,&y,1);
      if (x>GetParamInt(FakePad) && x<dataim[img].X-GetParamInt(FakePad) && y>GetParamInt(FakePad) && y<dataim[img].Y-GetParamInt(FakePad)) anyPosOK = 1;
   }
   if (anyPosOK==0) return;

   XMIN0=GetGlobalInt(XMIN);
   XMAX0=GetGlobalInt(XMAX);
   YMIN0=GetGlobalInt(YMIN);
   YMAX0=GetGlobalInt(YMAX);
   XMIN20=GetParamInt(XMIN2);
   XMAX20=GetParamInt(XMAX2);
   YMIN20=GetParamInt(YMIN2);
   YMAX20=GetParamInt(YMAX2);
   SetParam_NOT_THREADSAFE(XMIN2,(int)x0-rMarkPSF);
   SetParam_NOT_THREADSAFE(XMAX2,(int)x0+rMarkPSF+1);
   SetParam_NOT_THREADSAFE(YMIN2,(int)y0-rMarkPSF);
   SetParam_NOT_THREADSAFE(YMAX2,(int)y0+rMarkPSF+1);
   // ensure solution is within original photsec
   if (GetParamInt(XMIN2)<XMIN20) SetParam_NOT_THREADSAFE(XMIN2,XMIN20);
   if (GetParamInt(XMAX2)>XMAX20) SetParam_NOT_THREADSAFE(XMAX2,XMAX20);
   if (GetParamInt(YMIN2)<YMIN20) SetParam_NOT_THREADSAFE(YMIN2,YMIN20);
   if (GetParamInt(YMAX2)>YMAX20) SetParam_NOT_THREADSAFE(YMAX2,YMAX20);
   SetGlobal_NOT_THREADSAFE(XMIN,GetParamInt(XMIN2)-rMark); if (GetGlobalInt(XMIN)<0) SetGlobal_NOT_THREADSAFE(XMIN,0);
   SetGlobal_NOT_THREADSAFE(XMAX,GetParamInt(XMAX2)+rMark); if (GetGlobalInt(XMAX)>GetGlobalInt(X)) SetGlobal_NOT_THREADSAFE(XMAX,GetGlobalInt(X));
   SetGlobal_NOT_THREADSAFE(YMIN,GetParamInt(YMIN2)-rMark); if (GetGlobalInt(YMIN)<0) SetGlobal_NOT_THREADSAFE(YMIN,0);
   SetGlobal_NOT_THREADSAFE(YMAX,GetParamInt(YMAX2)+rMark); if (GetGlobalInt(YMAX)>GetGlobalInt(Y)) SetGlobal_NOT_THREADSAFE(YMAX,GetGlobalInt(Y));
   for (img=0;img<Nimg;img++) {
      shift(img,x0,y0,&x,&y,1);
      ix=(int)(x+RPSF[img])-RPSF[img];
      iy=(int)(y+RPSF[img])-RPSF[img];
      getpsfpars(img,xw,yw,(&pa)-img,(&pb)-img,(&pc)-img);
      bg[img]=0.;
      nbg=0;
      for (y1=-RPSF[img];y1<=RPSF[img] && iy+y1<dataim[img].Y;y1++) if (iy+y1>=0) for (x1=-RPSF[img];x1<=RPSF[img] && ix+x1<dataim[img].X;x1++) if (ix+x1>=0 && dataOK(img,ix+x1,iy+y1) && y1>=-rphot[img] && y1<=rphot[img] && x1>=-rphot[img] && x1<=rphot[img]) {
	 bg[img]+=data[img][iy+y1][ix+x1];
	 nbg++;
      }
      if (nbg) bg[img]/=nbg;
      else bg[img]=skyval(img,ix,iy);
#ifdef USEWFPC2
      if (hstmode[img].inst==WFPC2) WFPC2fixfakemag(img,x0,y0,ct0,bg);
#endif
#ifdef USEACS
      if (hstmode[img].inst==ACS) ACSfixfakemag(img,x0,y0,ct0,bg);
#endif
#ifdef USEWFC3
      if (hstmode[img].inst==WFC3) WFC3fixfakemag(img,x0,y0,ct0,bg);
#endif
#ifdef USEROMAN
      if (hstmode[img].inst==ROMAN) ROMANfixfakemag(img,x0,y0,ct0);
#endif
#ifdef USENIRCAM
      if (hstmode[img].inst==NIRCAM) NIRCAMfixfakemag(img,x0,y0,ct0);
#endif
#ifdef USENIRISS
      if (hstmode[img].inst==NIRISS) NIRISSfixfakemag(img,x0,y0,ct0);
#endif
#ifdef USEMIRI
      if (hstmode[img].inst==MIRI) MIRIfixfakemag(img,x0,y0,ct0);
#endif
#ifdef USEEUCLID
      if (hstmode[img].inst==NISP) NISPfixfakemag(img,x0,y0,ct0);
      if (hstmode[img].inst==VIS) VISfixfakemag(img,x0,y0,ct0);
#endif
      if (ix>=-rphot[img] && ix<dataim[img].X+rphot[img] && iy>=-rphot[img] && iy<dataim[img].Y+rphot[img]) {
	 cx=(int)((x-ix)*50+0.5);
	 if (cx<0) cx=0; if (cx>50) cx=50;
	 cy=(int)((y-iy)*50+0.5);
	 if (cy<0) cy=0; if (cy>50) cy=50;
	 calc1psf(img,x,y,rphot[img],pa,pb,pc,1,1); // PSF for eval()
	 SNRIN+=SQR(eval1_snr(img,ix,iy,cx,cy,ct0[img]));
      }
      calc1psf(img,x,y,RPSF[img],pa,pb,pc,1,0); // PSF for image addition
      for (y1=-RPSF[img];y1<=RPSF[img] && iy+y1<dataim[img].Y;y1++) if (iy+y1>=0) for (x1=-RPSF[img];x1<=RPSF[img] && ix+x1<dataim[img].X;x1++) if (ix+x1>=0) {
	 data0[img][y1+RPSF[img]][x1+RPSF[img]]=data[img][iy+y1][ix+x1];
	 if (dataOK(img,ix+x1,iy+y1)) {
	    if (GetParamInt(RandomFake)) data[img][iy+y1][ix+x1]+=poiss(ct0[img]*psf[img][y1][x1]*iGAIN[img])*invGAIN[img]; // AEDDEBUG need to test this works
	    else data[img][iy+y1][ix+x1]+=ct0[img]*psf[img][y1][x1];
	 }
      }
      // ensure at least one corner is in the WCS-defined position
      anyCornerOK = 0;
      shift(img,GetGlobalInt(XMIN),GetGlobalInt(YMIN),&x,&y,1); if (x>-999.0) anyCornerOK = 1;
      if (!anyCornerOK) {
	 shift(img,GetGlobalInt(XMIN),GetGlobalInt(YMAX),&x,&y,1); if (x>-999.0) anyCornerOK = 1;
      }
      if (!anyCornerOK) {
	 shift(img,GetGlobalInt(XMAX),GetGlobalInt(YMIN),&x,&y,1); if (x>-999.0) anyCornerOK = 1;
      }
      if (!anyCornerOK) {
	 shift(img,GetGlobalInt(XMAX),GetGlobalInt(YMAX),&x,&y,1); if (x>-999.0) anyCornerOK = 1;
      }
      if (anyCornerOK) {
	 wcsminmax_save = GetGlobalInt(wcsminmax_set);
	 SetGlobal_NOT_THREADSAFE(wcsminmax_set,0); // temporary override to determine surrounding box
	 shift(img,GetGlobalInt(XMIN),GetGlobalInt(YMIN),&x,&y,1);
	 xmin=xmax=(int)(x+RPSF[img])-RPSF[img];
	 ymin=ymax=(int)(y+RPSF[img])-RPSF[img];
	 shift(img,GetGlobalInt(XMIN),GetGlobalInt(YMAX),&x,&y,1);
	 ix=(int)(x+RPSF[img])-RPSF[img]; if (ix<xmin) xmin=ix; if (ix>xmax) xmax=ix;
	 iy=(int)(y+RPSF[img])-RPSF[img]; if (iy<ymin) ymin=iy; if (iy>ymax) ymax=iy;
	 shift(img,GetGlobalInt(XMAX),GetGlobalInt(YMIN),&x,&y,1);
	 ix=(int)(x+RPSF[img])-RPSF[img]; if (ix<xmin) xmin=ix; if (ix>xmax) xmax=ix;
	 iy=(int)(y+RPSF[img])-RPSF[img]; if (iy<ymin) ymin=iy; if (iy>ymax) ymax=iy;
	 shift(img,GetGlobalInt(XMAX),GetGlobalInt(YMAX),&x,&y,1);
	 ix=(int)(x+RPSF[img])-RPSF[img]; if (ix<xmin) xmin=ix; if (ix>xmax) xmax=ix;
	 iy=(int)(y+RPSF[img])-RPSF[img]; if (iy<ymin) ymin=iy; if (iy>ymax) ymax=iy;
	 xmin-=RPSF[img];
	 ymin-=RPSF[img];
	 xmax+=RPSF[img];
	 ymax+=RPSF[img];
	 if (xmin<0) xmin=0;
	 if (xmax>=dataim[img].X) xmax=dataim[img].X-1;
	 if (ymin<0) ymin=0;
	 if (ymax>=dataim[img].Y) ymax=dataim[img].Y-1;
	 xsize=1+xmax-xmin;
	 if (xsize>0) for (y1=ymin;y1<=ymax;y1++) memcpy(res[img][y1]+xmin,data[img][y1]+xmin,xsize*FLOATSIZE);
	 SetGlobal_NOT_THREADSAFE(wcsminmax_set,wcsminmax_save);
      }
   }
   SNRIN=sqrt(SNRIN);
   SetGlobal_NOT_THREADSAFE(fitpsf,1);
   Nstars=0;
   if (GetGlobalInt(WARMSTART)) {
      setindx(-1);
      for (i=0;i<Nrealstar;i++) if (realstar[i].x>=GetGlobalInt(XMIN) && realstar[i].x<GetGlobalInt(XMAX) && realstar[i].y>=GetGlobalInt(YMIN) && realstar[i].y<GetGlobalInt(YMAX)) {
	 stars[Nstars].x=realstar[i].x;
	 stars[Nstars].y=realstar[i].y;
	 stars[Nstars].type=realstar[i].t;
	 stars[Nstars].passID=0;
	 getpsfpars(-1,xw,yw,stars[Nstars].pa,stars[Nstars].pb,stars[Nstars].pc);
	 if (Timg>Nimg) getpsfpars(Nimg,xw,yw,stars[Nstars].pa,stars[Nstars].pb,stars[Nstars].pc);
	 phot((int)xw,(int)yw,Nstars,0);
	 imgsub(Nstars);
	 Nstars++;
      }
      stars[Nstars].x=xw;
      stars[Nstars].y=yw;
      if (tw==2) stars[Nstars].type=1;
      else stars[Nstars].type=tw;
      stars[Nstars].passID=1;
      getpsfpars(-1,xw,yw,stars[Nstars].pa,stars[Nstars].pb,stars[Nstars].pc);
      if (Timg>Nimg) getpsfpars(Nimg,xw,yw,stars[Nstars].pa,stars[Nstars].pb,stars[Nstars].pc);
      phot((int)xw,(int)yw,Nstars,0);
      imgsub(Nstars);
      Nstars++;
   }
   else find(0);
   sortstars();
   if (GetParamInt(FitSky)==1) for (i=0;i<Nstars;i++) stars[i].flag=1;
   it=0;
   cont=1;
   SetGlobal_NOT_THREADSAFE(fitpsf,0);
   while (cont && it<GetParamInt(MaxIT)) {
      cont=solve(ext,z,++it,0);
      if (cont>1 && it>GetParamInt(MaxIT)/2) it=GetParamInt(MaxIT)/2;
   }
   sortstars();
   if (GetGlobalInt(WARMSTART)) i=Nstars-1;
   else i=bestmatch(x0,y0);
   if (i>=0 && (stars[i].x<GetParamInt(XMIN2) || stars[i].x>=GetParamInt(XMAX2) || stars[i].y<GetParamInt(YMIN2) || stars[i].y>=GetParamInt(YMAX2))) i=-1;
   if (i>=0 && !GetGlobalInt(WARMSTART) && (stars[i].s<=0 || stars[i].s<GetParamDouble(SigFinal)*stars[i].ss)) i=-1;
   if (i>=0 && !GetGlobalInt(WARMSTART)) {
      rmin=SNRIN*exp(-0.5*(SQR(stars[i].x-x0)+SQR(stars[i].y-y0))/GetParamDouble(FakePSF));
      for (j=0;j<Nrealstar && i>=0;j++) {
	 r=realstar[j].sn*exp(-0.5*(SQR(stars[i].x-realstar[j].x)+SQR(stars[i].y-realstar[j].y))/GetParamDouble(FakePSF));
	 if (r>rmin && bestmatch(realstar[j].x,realstar[j].y)==i) i=-1;
      }
   }
   fprintf(ffakeout,"%d %d %7.2f %7.2f ",ext,z+1,x0,y0);
   for (img=0;img<Nimg;img++) {
      if (ct0[img]<999999.5) fprintf(ffakeout,"%8.1f ",ct0[img]);
      else fprintf(ffakeout,"%8.2e ",ct0[img]);
#ifdef USEWFPC2
      if (hstmode[img].inst==WFPC2) fprintf(ffakeout,"%6.3f ",WFPC2calcmag(img,x0,y0,ct0[img],bg[img],GetParamInt(WFPC2useCTE)));
      else
#endif
#ifdef USEACS
      if (hstmode[img].inst==ACS) fprintf(ffakeout,"%6.3f ",ACScalcmag(img,x0,y0,ct0[img],bg[img],GetParamInt(ACSuseCTE)));
      else
#endif
#ifdef USEWFC3
      if (hstmode[img].inst==WFC3) fprintf(ffakeout,"%6.3f ",WFC3calcmag(img,x0,y0,ct0[img],bg[img],GetParamInt(WFC3useCTE)));
      else
#endif
#ifdef USEROMAN
      if (hstmode[img].inst==ROMAN) fprintf(ffakeout,"%6.3f ",ROMANcalcmag(img,x0,y0,ct0[img]));
      else
#endif
#ifdef USENIRCAM
      if (hstmode[img].inst==NIRCAM) fprintf(ffakeout,"%6.3f ",NIRCAMcalcmag(img,x0,y0,ct0[img]));
      else
#endif
#ifdef USENIRISS
      if (hstmode[img].inst==NIRISS) fprintf(ffakeout,"%6.3f ",NIRISScalcmag(img,x0,y0,ct0[img]));
      else
#endif
#ifdef USEMIRI
      if (hstmode[img].inst==MIRI) fprintf(ffakeout,"%6.3f ",MIRIcalcmag(img,x0,y0,ct0[img]));
      else
#endif
#ifdef USEEUCLID
      if (hstmode[img].inst==NISP) fprintf(ffakeout,"%6.3f ",NISPcalcmag(img,x0,y0,ct0[img]));
      else
      if (hstmode[img].inst==VIS) fprintf(ffakeout,"%6.3f ",VIScalcmag(img,x0,y0,ct0[img]));
      else
#endif
	 fprintf(ffakeout,"%6.3f ",-2.5*log10(ct0[img]*apcor[img]/iEXP[img])+GetParamDouble(Zero));
   }
   if (i>=0) {
      imgadd(i);
      photsearch(-1,&stars[i].x,&stars[i].y,stars[i].pa,stars[i].pb,stars[i].pc,&stars[i].s,&stars[i].ss,&stars[i].chi,&stars[i].sh,&stars[i].sky,stars[i].is,stars[i].iss,stars[i].icm,&stars[i].type,-1,i);
      out1star(ext,z,i,ffakeout,1);
   }
   else out1star(ext,z,i,ffakeout,1);
   SetGlobal_NOT_THREADSAFE(XMIN,XMIN0);
   SetGlobal_NOT_THREADSAFE(XMAX,XMAX0);
   SetGlobal_NOT_THREADSAFE(YMIN,YMIN0);
   SetGlobal_NOT_THREADSAFE(YMAX,YMAX0);
   SetParam_NOT_THREADSAFE(XMIN2,XMIN20);
   SetParam_NOT_THREADSAFE(XMAX2,XMAX20);
   SetParam_NOT_THREADSAFE(YMIN2,YMIN20);
   SetParam_NOT_THREADSAFE(YMAX2,YMAX20);
   for (img=0;img<Nimg;img++) {
      shift(img,x0,y0,&x,&y,1);
      ix=(int)(x+RPSF[img])-RPSF[img];
      iy=(int)(y+RPSF[img])-RPSF[img];
      for (y1=-RPSF[img];y1<=RPSF[img] && iy+y1<dataim[img].Y;y1++) if (iy+y1>=0) for (x1=-RPSF[img];x1<=RPSF[img] && ix+x1<dataim[img].X;x1++) if (ix+x1>=0) data[img][iy+y1][ix+x1]=data0[img][y1+RPSF[img]][x1+RPSF[img]];
   }
   fflush(ffakeout);
   return;
}

#define MAX_LINE_LENGTH 40000
void fakestars(int ext,int fld) {
   FILE *f;
   int e,z,img,tw=1,i,nfake=0;
   double x,y,xw=0,yw=0,*ctin;
   char str[MAX_LINE_LENGTH];

   Nrealstar=i=0;
   if ((f=fopen(GetParamString(outfn),"r"))==NULL) {
      printf("****Error opening original photometry file %s\n",GetParamString(outfn));
      exit(-1);
   }
   while (fscanf(f,"%d %d",&e,&z)==2) {
      if (e==ext && z==fld+1) Nrealstar++;
      fgets(str,MAX_LINE_LENGTH,f);
   }
   printf("Read %d real stars\n",Nrealstar);
   fflush(stdout);
   realstar=(realstartype*)calloc(Nrealstar,sizeof(realstartype)); if (!realstar) merr();
   fseek(f,0,SEEK_SET);
   while (fscanf(f,"%d %d",&e,&z)==2) {
      if (e==ext && z==fld+1) {
	 fscanf(f,"%lf %lf %lf %lf %lf %lf %lf %lf %d",&(realstar[i].x),&(realstar[i].y),&x,&(realstar[i].sn),&x,&x,&x,&x,&(realstar[i].t));
	 i++;
      }
      fgets(str,MAX_LINE_LENGTH,f);
   }
   fclose(f);
   if (i!=Nrealstar) {
      printf("****Programming error; i!=Nrealstar\n");
      exit(-1);
   }
   if ((f=fopen(GetParamString(FakeStars),"r"))==NULL) {
      printf("****Error opening FakeStars file %s\n",GetParamString(FakeStars));
      exit(-1);
   }
   ctin=(double*)calloc(Nimg,DOUBLESIZE); if (!ctin) merr();
   data0=(chiptype*)calloc(Nimg,sizeof(chiptype)); if (!data0) merr();
   for (img=0;img<Nimg;img++) {
      //cleansat(img);
      data0[img]=allocchip(2*RPSF[img]+1,2*RPSF[img]+1);
   }
   while (fscanf(f,"%d %d",&e,&z)==2) {
      if (e==ext && z==fld+1) {
	 fscanf(f,"%lf %lf",&x,&y);
	 if (GetGlobalInt(WARMSTART)) fscanf(f,"%lf %lf %d",&xw,&yw,&tw);
#ifdef USEWFPC2
	 WFPC2readfakemag(f);
#endif
#ifdef USEACS
	 ACSreadfakemag(f);
#endif
#ifdef USEWFC3
	 WFC3readfakemag(f);
#endif
#ifdef USEROMAN
	 ROMANreadfakemag(f);
#endif
#ifdef USENIRCAM
	 NIRCAMreadfakemag(f);
#endif
#ifdef USENIRISS
	 NIRISSreadfakemag(f);
#endif
#ifdef USEMIRI
	 MIRIreadfakemag(f);
#endif
#ifdef USEEUCLID
	 NISPreadfakemag(f);
	 VISreadfakemag(f);
#endif
	 for (img=0;img<Nimg;img++) if (hstmode[img].inst==NONE) fscanf(f,"%lf",ctin+img);
	 fgets(str,MAX_LINE_LENGTH,f);
	 fakestar(ext,fld,x,y,xw,yw,tw,ctin);
	 nfake++;
      }
      else fgets(str,MAX_LINE_LENGTH,f);
   }
   fclose(f);
   printf("Completed %d artificial stars\n",nfake);
   fflush(stdout);
   for (img=0;img<Nimg;img++) freechip(data0[img],2*RPSF[img]+1,2*RPSF[img]+1);
   free(ctin);
   free(data0);
   free(realstar);
   return;
}
#undef MAX_LINE_LENGTH

void outpt(FILE*of,int ext,int fld) {
   int i,Nelim[2]={0,0};

   sortstars();
   for (i=0;i<Nstars;i++) if (stars[i].x>=GetParamInt(XMIN2) && stars[i].x<GetParamInt(XMAX2) && stars[i].y>=GetParamInt(YMIN2) && stars[i].y<GetParamInt(YMAX2) && (GetGlobalInt(WARMSTART) || (stars[i].s>0 && stars[i].s>=GetParamDouble(SigFinal)*stars[i].ss))) {
      imgadd(i);
      photsearch(-1,&stars[i].x,&stars[i].y,stars[i].pa,stars[i].pb,stars[i].pc,&stars[i].s,&stars[i].ss,&stars[i].chi,&stars[i].sh,&stars[i].sky,stars[i].is,stars[i].iss,stars[i].icm,&stars[i].type,-1,i);
      imgsub(i);
   }
   for (i=0;i<Nstars;i++) {
      imgadd(i);
      Nelim[out1star(ext,fld,i,of,0)]++;
      imgsub(i);
   }
   if (GetGlobalInt(WARMSTART)==2) for (i=0;i<Nstars;i++) imgadd(i);
   printf("%d stars written; %d stars deleted\n",Nelim[1],Nelim[0]);
   fflush(stdout);
   fflush(of);
   return;
}

void outptinfo(void) {
   int j;

   fprintf(finfo,"%d sets of output data\n",Nimg);
   for (j=0;j<Nimg;j++) {
      fprintf(finfo,"%s\n",base[j]);
      fprintf(finfo,"%16.8f\n",iEPOCH[j]);
   }
   fprintf(finfo,"\n");
   fflush(finfo);
   return;
}

void outptalign(void) {
   int i,j;
   fprintf(finfo,"Alignment\n");
   for (i=0;i<Nimg;i++) {
      if (GetParamInt(Align)<4) {for (j=0;j<5;j++) fprintf(finfo," %f",dpos[i][j]);}
      else {for (j=0;j<40;j++) fprintf(finfo," %12.6e",dpos[i][j]);}
      if (GetParamInt(UseWCS)==2) {
	 for (j=0;j<4;j++) fprintf(finfo," %12.6e",wcsref[i][j]);
	 for (j=0;j<80;j++) fprintf(finfo," %12.6e",wcs[i][j]);
      }
      fprintf(finfo,"\n");
   }
   fflush(finfo);
   return;
}

void outptapcor(void) {
   int i;
   fprintf(finfo,"Aperture corrections\n");
   for (i=0;i<Nimg;i++) fprintf(finfo,"  %f\n",-2.5*log10(apcor[i]));
   fprintf(finfo,"\n");
   fflush(finfo);
   return;
}

void allocXY(void) {
   void *ptr;
   int y;
   ran=(char**)calloc((size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef),PTRSIZE);
   snmapflag=(char**)calloc((size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef),PTRSIZE);
   snmap=(float**)calloc((size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef),PTRSIZE);
   snmmap=(float**)calloc((size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef),PTRSIZE);
   indx=(int**)calloc((size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef),PTRSIZE);
   tindx=(int*)calloc((size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef),INTSIZE);
   if (!ran || !snmapflag || !snmap || !snmmap || !indx || !tindx) merr();
   ran -= GetGlobalInt(YMIN)*GetParamInt(SubResRef);
   snmapflag -= GetGlobalInt(YMIN)*GetParamInt(SubResRef);
   snmap -= GetGlobalInt(YMIN)*GetParamInt(SubResRef);
   snmmap -= GetGlobalInt(YMIN)*GetParamInt(SubResRef);
   indx -= GetGlobalInt(YMIN)*GetParamInt(SubResRef);
   tindx -= GetGlobalInt(XMIN)*GetParamInt(SubResRef);
   ptr=malloc((size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*(size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef)*GetParamInt(SubResRef)); if (!ptr) merr();
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) ran[y] = ptr + (size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef)*(y-GetGlobalInt(YMIN)*GetParamInt(SubResRef)) - GetGlobalInt(XMIN)*GetParamInt(SubResRef);
   ptr=malloc((size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*(size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef)*GetParamInt(SubResRef)); if (!ptr) merr();
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) snmapflag[y] = ptr + (size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef)*(y-GetGlobalInt(YMIN)*GetParamInt(SubResRef)) - GetGlobalInt(XMIN)*GetParamInt(SubResRef);
   ptr=malloc((size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*(size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef)*GetParamInt(SubResRef)*FLOATSIZE); if (!ptr) merr();
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) snmap[y] = ptr + (size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef)*(y-GetGlobalInt(YMIN)*GetParamInt(SubResRef))*FLOATSIZE - GetGlobalInt(XMIN)*GetParamInt(SubResRef)*FLOATSIZE;
   ptr=malloc((size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*(size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef)*GetParamInt(SubResRef)*FLOATSIZE); if (!ptr) merr();
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) snmmap[y] = ptr + (size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef)*(y-GetGlobalInt(YMIN)*GetParamInt(SubResRef))*FLOATSIZE - GetGlobalInt(XMIN)*GetParamInt(SubResRef)*FLOATSIZE;
   ptr=malloc((size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*(size_t)(GetGlobalInt(YMAX)-GetGlobalInt(YMIN))*GetParamInt(SubResRef)*GetParamInt(SubResRef)*INTSIZE); if (!ptr) merr();
   for (y=GetGlobalInt(YMIN)*GetParamInt(SubResRef);y<GetGlobalInt(YMAX)*GetParamInt(SubResRef);y++) indx[y] = ptr + (size_t)(GetGlobalInt(XMAX)-GetGlobalInt(XMIN))*GetParamInt(SubResRef)*(y-GetGlobalInt(YMIN)*GetParamInt(SubResRef))*INTSIZE - GetGlobalInt(XMIN)*GetParamInt(SubResRef)*INTSIZE;
}

void freeXY(void) {
   free(indx[GetGlobalInt(YMIN)*GetParamInt(SubResRef)]+GetGlobalInt(XMIN)*GetParamInt(SubResRef));
   free(snmmap[GetGlobalInt(YMIN)*GetParamInt(SubResRef)]+GetGlobalInt(XMIN)*GetParamInt(SubResRef));
   free(snmap[GetGlobalInt(YMIN)*GetParamInt(SubResRef)]+GetGlobalInt(XMIN)*GetParamInt(SubResRef));
   free(snmapflag[GetGlobalInt(YMIN)*GetParamInt(SubResRef)]+GetGlobalInt(XMIN)*GetParamInt(SubResRef));
   free(ran[GetGlobalInt(YMIN)*GetParamInt(SubResRef)]+GetGlobalInt(XMIN)*GetParamInt(SubResRef));
   free(tindx+GetGlobalInt(XMIN)*GetParamInt(SubResRef));
   free(indx+GetGlobalInt(YMIN)*GetParamInt(SubResRef));
   free(snmmap+GetGlobalInt(YMIN)*GetParamInt(SubResRef));
   free(snmap+GetGlobalInt(YMIN)*GetParamInt(SubResRef));
   free(snmapflag+GetGlobalInt(YMIN)*GetParamInt(SubResRef));
   free(ran+GetGlobalInt(YMIN)*GetParamInt(SubResRef));
}

void procframe(int ext) {
   int img,i,x,y,z,it,cont,skip;
   char ch[2880];
   imtype timg;

   SetGlobal_NOT_THREADSAFE(IMDIFF,0);
   if (!isimage(dataim)) {
      for (img=0;img<Timg;img++) {
	 fopenagain(fdata+img);
	 readimage(fdata[img].f,dataim+img);
	 freclose(fdata+img,1,fwarn);
	 if (GetParamChar(FakeStars)==0 && img<Nimg) {
	    fopenagain(fres+img);
	    writeimage(fres[img].f,dataim+img);
	    freclose(fres+img,0,NULL);
	    if (GetParamChar(UsePhot)==0) {
	       fopenagain(fpsf+img);
	       writeimage(fpsf[img].f,dataim+img);
	       freclose(fpsf+img,0,NULL);
	    }
	 }
	 freeimg(dataim[img].data,dataim[img].X,dataim[img].Y,dataim[img].Z);
      }
      return;
   }
   if (Timg>Nimg) {
      SetGlobal_NOT_THREADSAFE(X,dataim[Nimg].X);
      SetGlobal_NOT_THREADSAFE(Y,dataim[Nimg].Y);
   }
   else {
      SetGlobal_NOT_THREADSAFE(X,dataim[0].X);
      SetGlobal_NOT_THREADSAFE(Y,dataim[0].Y);
   }
#ifdef PGPLOT
   allocDiagData2(ext);
#endif
   //allocXY();
   for (img=0;img<Timg;img++) {
      data[img]=allocchip(dataim[img].X,dataim[img].Y);
      if (fsky[img].lastoffset>=0) sky[img]=allocchip(dataim[img].X,dataim[img].Y);
      res[img]=allocchip(dataim[img].X,dataim[img].Y);
   }
   if (GetParamInt(GUSE)<0 && GetParamInt(CUSE)<0) {
      SetParam_NOT_THREADSAFE(XMIN2,0);
      SetParam_NOT_THREADSAFE(XMAX2,GetGlobalInt(X));
      SetParam_NOT_THREADSAFE(YMIN2,0);
      SetParam_NOT_THREADSAFE(YMAX2,GetGlobalInt(Y));
   }
   for (z=0;z<dataim[0].Z;z++) {
#ifdef PGPLOT
      setDiagFrame(ext,z);
#endif
      initimgpars();
      for (img=0;img<Timg;img++) {
	 if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	    for (i=0;i<5;i++) dpos[img][i]=dpos0[img][i];
	 }
      }
      // update params for camera
#ifdef USEWFPC2
      wfpc2initparam();
#endif
#ifdef USEACS
      acsinitparam();
#endif
#ifdef USEWFC3
      wfc3initparam();
#endif
#ifdef USEROMAN
      romaninitparam();
#endif
#ifdef USENIRCAM
      nircaminitparam();
#endif
#ifdef USENIRISS
      nirissinitparam();
#endif
#ifdef USEMIRI
      miriinitparam();
#endif
#ifdef USEEUCLID
      nispinitparam();
      visinitparam();
#endif
#ifdef USEWFPC2
      wfpc2radii(); // accounts for plate scale vs. chip # and shifts
#endif
      initcirc();
#ifdef USEACS
      acsinitpsf();
#endif
#ifdef USEWFC3
      wfc3initpsf();
#endif
#ifdef USEWFPC2
      if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0 && !GetGlobalInt(DRIZZLE_BASE)) wfpc2tweakshift();
      wfpc2initpsf();
#endif
#ifdef USEROMAN
      romaninitpsf();
#endif
#ifdef USENIRCAM
      nircaminitpsf();
#endif
#ifdef USENIRISS
      nirissinitpsf();
#endif
#ifdef USEMIRI
      miriinitpsf();
#endif
#ifdef USEEUCLID
      nispinitpsf();
      visinitpsf();
#endif
      if (GetGlobalInt(WARMSTART)==2 && refpsf.Next>=0) {
	 imtype *timg;
	 if (ext==0) timg=&(refpsf.img);
	 else timg=refpsf.ext+ext-1;
	 if (timg->X!=2*RPSF[Timg-1]+1 || timg->Y!=2*RPSF[Timg-1]+1 || timg->Z!=dataim[0].Z) {
	    printf("**Provided PSF for subtraction is the wrong size; using analytic solution only\n");
	    refpsfimg=NULL;
	 }
	 else refpsfimg=timg->data[z];
      }
      if (Nimg==Timg) SetGlobal_NOT_THREADSAFE(DRIZZLE_BASE,0);
      else if (hstmode[Nimg].inst==NONE) SetGlobal_NOT_THREADSAFE(DRIZZLE_BASE,1);
      out1starinfo();
      for (img=0;img<Timg;img++) {
	 fopenagain(fdata+img);
	 readchip(fdata[img].f,data[img],dataim+img);
	 freclose(fdata+img,1,fwarn);
	 if (fsky[img].lastoffset>=0) {
	    fopenagain(fsky+img);
	    readchip(fsky[img].f,sky[img],dataim+img);
	    freclose(fsky+img,1,fwarn);
	 }
      }
      if ((GetParamInt(GUSE)<0 || ext==GetParamInt(GUSE)) && (GetParamInt(CUSE)<0 || z==GetParamInt(CUSE))) {
	 if (ext>0) printf("** Extension %d, Chip %d **\n",ext,z+1);
	 else printf("** Chip %d **\n",z+1);
	 fflush(stdout);
	 if (GetParamChar(FakeStars)==0 && GetParamInt(VerboseData)>0)
#ifdef DOLPHOT_THREADED
#pragma omp critical
#endif
	 {
	    fprintf(fverb,"EXTENSION %d CHIP %d\n",ext,z+1);
	    fflush(fverb);
	 }
	 if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	    fprintf(finfo,"EXTENSION %d CHIP %d\n",ext,z+1);
	    fprintf(finfo,"Limits\n %d %d %d %d\n",GetParamInt(XMIN2),GetParamInt(XMAX2),GetParamInt(YMIN2),GetParamInt(YMAX2));
#ifdef USEWFPC2
	    writewfpc2info();
#endif
#ifdef USEACS
	    writeacsinfo();
#endif
#ifdef USEWFC3
	    writewfc3info();
#endif
#ifdef USEROMAN
	    writeromaninfo();
#endif
#ifdef USENIRCAM
	    writenircaminfo();
#endif
#ifdef USENIRISS
	    writenirissinfo();
#endif
#ifdef USEMIRI
	    writemiriinfo();
#endif
#ifdef USEEUCLID
	    writenispinfo();
	    writevisinfo();
#endif
	    fflush(finfo);
	 }
	 else readinfo(finfo,ext,z);
	 for (img=0;img<Timg;img++) {
	    if (iDMIN[img]<iDMIN0[img]) {
	       float *pd,*pr,*plast;
	       plast=data[img][0]+dataim[img].Y*dataim[img].X;
	       for (pd=data[img][0],pr=res[img][0];pd<plast;pd++,pr++) {
		  if (*pd<=iDMIN0[img]) *pd=*pr=safedown(iDMIN[img]);
		  else *pr=*pd;
	       }
	    }
	    else memcpy(res[img][0],data[img][0],dataim[img].X*dataim[img].Y*FLOATSIZE);
	    if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	       apcor[img]=1.;
	       for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) poff[img][y][x]=0;
	    }
	    else if (img<Nimg) {
	       float* pad;
	       int nwrite = RPSF[img];
	       if (GetParamInt(imgdata[img].RPSF)<nwrite) nwrite = GetParamInt(imgdata[img].RPSF);
	       int npad = GetParamInt(imgdata[img].RPSF) - nwrite;
	       pad = (float*)calloc(GetParamInt(imgdata[img].RPSF)*2+1,sizeof(float));
	       fopenagain(fpsf+img);
	       for (y=0;y<npad;y++) ffread(pad,4,GetParamInt(imgdata[img].RPSF)*2+1,fpsf[img].f);
	       for (y=-nwrite;y<=nwrite;y++) {
		  if (npad>0) ffread(pad,4,npad,fpsf[img].f);
		  ffread(poff[img][y]-nwrite,4,nwrite*2+1,fpsf[img].f);
		  if (npad>0) ffread(pad,4,npad,fpsf[img].f);
	       }
	       for (y=0;y<npad;y++) ffread(pad,4,GetParamInt(imgdata[img].RPSF)*2+1,fpsf[img].f);
	       freclose(fpsf+img,1,fwarn);
	       free(pad);
	       if (!GetParamInt(FakeStarPSF)) for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) poff[img][y][x]=0;
	    }
	    else {
	       apcor[img]=1.;
	       for (y=-RPSF[img];y<=RPSF[img];y++) for (x=-RPSF[img];x<=RPSF[img];x++) poff[img][y][x]=0;
	    }
	 }
#if 0
	 {
	    // Dump Sample PSFs
	    extern void ACSdumpPSFs();
	    extern void WFC3dumpPSFs();
	    ACSdumpPSFs();
	    WFC3dumpPSFs();
	    fflush(stdout);
	    exit(0);
	 }
#endif
#if 0
	 {
	    // Dump Sample PSFs
	    extern void WFPC2dumpPSFs();
	    WFPC2dumpPSFs();
	    fflush(stdout);
	    exit(0);
	 }
#endif
	 if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	    SetGlobal_NOT_THREADSAFE(XMIN,GetParamInt(XMIN2));
	    if (GetGlobalInt(XMIN)<0) SetGlobal_NOT_THREADSAFE(XMIN,0);
	    SetGlobal_NOT_THREADSAFE(XMAX,GetParamInt(XMAX2));
	    if (GetGlobalInt(XMAX)>GetGlobalInt(X)) SetGlobal_NOT_THREADSAFE(XMAX,GetGlobalInt(X));
	    SetGlobal_NOT_THREADSAFE(YMIN,GetParamInt(YMIN2));
	    if (GetGlobalInt(YMIN)<0) SetGlobal_NOT_THREADSAFE(YMIN,0);
	    SetGlobal_NOT_THREADSAFE(YMAX,GetParamInt(YMAX2));
	    if (GetGlobalInt(YMAX)>GetGlobalInt(Y)) SetGlobal_NOT_THREADSAFE(YMAX,GetGlobalInt(Y));
	    allocXY();
	    SetGlobal_NOT_THREADSAFE(fitpsf,1);
	    if (GetParamInt(UseWCS)==2) convertWCS2();
	    else if (GetParamInt(UseWCS)==1) convertWCS1();
	    setRef2Img();
	    if (GetParamInt(Align)==4) convertAlign4();
	    if (GetParamInt(Align) && Timg>1) align(ext,z);
	    outptalign();
	    freeXY();
	    if (GetParamInt(UseWCS)==2) setwcsminmax();
	 }
	 if (GetParamInt(AlignOnly)==0) {
	    setRMark();
	    SetGlobal_NOT_THREADSAFE(XMIN,GetParamInt(XMIN2)-rMark);
	    if (GetGlobalInt(XMIN)<0) SetGlobal_NOT_THREADSAFE(XMIN,0);
	    SetGlobal_NOT_THREADSAFE(XMAX,GetParamInt(XMAX2)+rMark);
	    if (GetGlobalInt(XMAX)>GetGlobalInt(X)) SetGlobal_NOT_THREADSAFE(XMAX,GetGlobalInt(X));
	    SetGlobal_NOT_THREADSAFE(YMIN,GetParamInt(YMIN2)-rMark);
	    if (GetGlobalInt(YMIN)<0) SetGlobal_NOT_THREADSAFE(YMIN,0);
	    SetGlobal_NOT_THREADSAFE(YMAX,GetParamInt(YMAX2)+rMark);
	    if (GetGlobalInt(YMAX)>GetGlobalInt(Y)) SetGlobal_NOT_THREADSAFE(YMAX,GetGlobalInt(Y));
	    allocXY();
	    if (GetParamChar(FakeStars)!=0) fakestars(ext,z);
	    else {
	       if (GetParamChar(UsePhot)!=0) SetGlobal_NOT_THREADSAFE(fitpsf,0);
	       Nstars=0;
	       if (GetGlobalInt(WARMSTART)) readwarm(ext,z);
	       else find(1);
	       sortstars();
	       if (GetParamInt(FitSky)==1) for (i=0;i<Nstars;i++) stars[i].flag=1;
	       it=0;
	       cont=1;
	       while (cont && it<GetParamInt(MaxIT)) {
		  cont=solve(ext,z,++it,1);
		  if (cont>1 && it>GetParamInt(MaxIT)/2) it=GetParamInt(MaxIT)/2;
	       }
	       
	       if (GetParamChar(UsePhot)==0) {
		  if (GetParamInt(ApCor)) {
		     char outstr[MAXNIMG][1024];
		     startapcor();
#ifdef DOLPHOT_THREADED
		     if (Nimg==1 || omp_get_max_threads()<2) {
			for (img=0;img<Nimg;img++) {
			   getapcor(ext,z,img,fapcor,outstr[img]);
			}
		     }
		     else
#pragma omp parallel for schedule(guided)
#endif
			for (img=0;img<Nimg;img++) {
			   getapcor(ext,z,img,fapcor,outstr[img]);
			}
		     for (img=0;img<Nimg;img++) printf("%s",outstr[img]);
		     fflush(stdout);
		  }
		  outptapcor();
	       }
	       outpt(of,ext,z);
	    }
	    freeXY();
	 }
      }
      // still need to advance through files
      else if (GetParamChar(FakeStars)!=0 || GetParamChar(UsePhot)!=0) {
	 //readinfo(finfo,ext,z); // doesn't write finfo for skipped frames
	 for (img=0;img<Nimg;img++) {
	    float* pad;
	    int nwrite = RPSF[img];
	    if (GetParamInt(imgdata[img].RPSF)<nwrite) nwrite = GetParamInt(imgdata[img].RPSF);
	    int npad = GetParamInt(imgdata[img].RPSF) - nwrite;
	    pad = (float*)calloc(GetParamInt(imgdata[img].RPSF)*2+1,sizeof(float));
	    fopenagain(fpsf+img);
	    for (y=0;y<npad;y++) ffread(pad,4,GetParamInt(imgdata[img].RPSF)*2+1,fpsf[img].f);
	    for (y=-nwrite;y<=nwrite;y++) {
	       if (npad>0) ffread(pad,4,npad,fpsf[img].f);
	       ffread(poff[img][y]-nwrite,4,nwrite*2+1,fpsf[img].f);
	       if (npad>0) ffread(pad,4,npad,fpsf[img].f);
	    }
	    for (y=0;y<npad;y++) ffread(pad,4,GetParamInt(imgdata[img].RPSF)*2+1,fpsf[img].f);
	    free(pad);
	    freclose(fpsf+img,1,fwarn);
	 }
      }
      if (GetParamChar(FakeStars)==0) for (img=0;img<Nimg;img++) {
	 if (GetParamChar(UsePhot)==0) {
	    float* pad;
	    int nwrite = RPSF[img];
	    if (GetParamInt(imgdata[img].RPSF)<nwrite) nwrite = GetParamInt(imgdata[img].RPSF);
	    int npad = GetParamInt(imgdata[img].RPSF) - nwrite;
	    pad = (float*)calloc(GetParamInt(imgdata[img].RPSF)*2+1,FLOATSIZE); if (!pad) merr();
	    memset(pad,0,FLOATSIZE*(GetParamInt(imgdata[img].RPSF)*2+1));
	    fopenagain(fpsf+img);
	    for (y=0;y<npad;y++) ffwrite(pad,FLOATSIZE,GetParamInt(imgdata[img].RPSF)*2+1,fpsf[img].f);
	    for (y=-nwrite;y<=nwrite;y++) {
	       if (npad>0) ffwrite(pad,FLOATSIZE,npad,fpsf[img].f);
	       ffwrite(poff[img][y]-nwrite,FLOATSIZE,nwrite*2+1,fpsf[img].f);
	       if (npad>0) ffwrite(pad,FLOATSIZE,npad,fpsf[img].f);
	    }
	    for (y=0;y<npad;y++) ffwrite(pad,FLOATSIZE,GetParamInt(imgdata[img].RPSF)*2+1,fpsf[img].f);
	    free(pad);
	    freclose(fpsf+img,0,NULL);
	 }
	 memcpy(&timg,dataim+img,sizeof(imtype));
	 timg.bscale=1.;
	 timg.bzero=0.;
	 timg.bits=-32;
	 fopenagain(fres+img);
	 writechip(fres[img].f,res[img],&timg);
	 freclose(fres+img,0,NULL);
      }
      freecirc();
   }
   memset(ch,0,sizeof(ch));
   for (img=0;img<Timg;img++) {
      skip=abs(dataim[img].bits)/8*dataim[img].X*dataim[img].Y*dataim[img].Z+dataim[img].pcount;
      skip=((skip+2879)/2880)*2880-skip;
      if (fsky[img].lastoffset>=0) fsky[img].lastoffset+=skip;
      fdata[img].lastoffset+=skip;
      if (img<Nimg) {
	 if (GetParamChar(FakeStars)==0) {
	    skip=FLOATSIZE*dataim[img].X*dataim[img].Y*dataim[img].Z+dataim[img].pcount;
	    skip=((skip+2879)/2880)*2880-skip;
	    fopenagain(fres+img);
	    fwrite(ch,1,skip,fres[img].f);
	    freclose(fres+img,0,NULL);
	 }
	 skip=FLOATSIZE*(2*GetParamInt(imgdata[img].RPSF)+1)*(2*GetParamInt(imgdata[img].RPSF)+1)*dataim[img].Z+dataim[img].pcount;
	 skip=((skip+2879)/2880)*2880-skip;
	 fopenagain(fpsf+img);
	 if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	    fwrite(ch,1,skip,fpsf[img].f);
	    freclose(fpsf+img,0,NULL);
	 }
	 else {
	    fread(ch,1,skip,fpsf[img].f);
	    freclose(fpsf+img,1,fwarn);
	 }
	 freechip(data[img],dataim[img].X,dataim[img].Y);
	 if (fsky[img].lastoffset>=0) freechip(sky[img],dataim[img].X,dataim[img].Y);
	 freechip(res[img],dataim[img].X,dataim[img].Y);
      }
   }
#ifdef USEWFPC2
   wfpc2freepsf();
#endif
#ifdef USEACS
   acsfreepsf();
#endif
#ifdef USEWFC3
   wfc3freepsf();
#endif
#ifdef USEROMAN
   romanfreepsf();
#endif
#ifdef USENIRCAM
   nircamfreepsf();
#endif
#ifdef USENIRISS
   nirissfreepsf();
#endif
#ifdef USEMIRI
   mirifreepsf();
#endif
#ifdef USEEUCLID
   nispfreepsf();
   visfreepsf();
#endif
   return;
}

int main(int argc,char**argv) {
   char str[82];
   int img,Next=0,ext,i;

   if (argc<2) {
      printf("****Usage: %s <output> <<options>>\n",*argv);
      printf("  -p<name>  for parameter file\n");
      printf("  x=y       to set flag x to value y\n");
      return 1;
   }
   INTSIZE=sizeof(int);
   LONGSIZE=sizeof(long);
   FLOATSIZE=sizeof(float);
   DOUBLESIZE=sizeof(double);
   PTRSIZE=sizeof(void*);
   if (sizeof(char)!=1) {printf("char size != 1\n"); return -1;}
   if (INTSIZE<4) {printf("int size < 4\n"); return -1;}
   if (LONGSIZE<4) {printf("long size < 4\n"); return -1;}
   if (FLOATSIZE!=4) {printf("float size != 4\n"); return -1;}
   if (DOUBLESIZE<8) {printf("double size < 8\n"); return -1;}
   if (sizeof(char*)!=PTRSIZE) {printf("char* size != void* size\n"); return -1;}
   if (sizeof(int*)!=PTRSIZE) {printf("int* size != void* size\n"); return -1;}
   if (sizeof(float*)!=PTRSIZE) {printf("float* size != void* size\n"); return -1;}
   if (sizeof(double*)!=PTRSIZE) {printf("double* size != void* size\n"); return -1;}
   if (sizeof(FILE*)!=PTRSIZE) {printf("FILE* size != void* size\n"); return -1;}

   // read parameters
   initimgdata();
   paramfile("dolphot.param",&dolphotparam);
   for (i=2;i<argc;i++) {
      if (!strncmp(argv[i],"-p",2) || !strncmp(argv[i],"-P",2)) paramfile1(argv[i]+2,&dolphotparam);
      else parseparam(argv[i],&dolphotparam);
   }
   // sanity check
   if (GetParamInt(RPSF0)<GetParamDouble(RAper0)) perr("RPSF must be at least as large as RAper");

   // allocate memory
   alloc_img_data();

   fdata=(reopenableFile*)calloc(Timg,sizeof(reopenableFile));
   fsky=(reopenableFile*)calloc(Timg,sizeof(reopenableFile));
   fpsf=(reopenableFile*)calloc(Timg,sizeof(reopenableFile));
   fres=(reopenableFile*)calloc(Timg,sizeof(reopenableFile));
   if (!fdata || !fsky || !fpsf || !fres) merr();
   SetParamString_NOT_THREADSAFE(outfn,argv[1]);
   if (GetParamChar(FakeStars)!=0) {
      if (GetParamChar(FakeOut)!=0) strcpy(str,GetParamString(FakeOut));
      else sprintf(str,"%s.fake",GetParamString(outfn));
      ffakeout=fopen(str,"w");
      if (!ffakeout) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      if (GetParamChar(UsePhot)!=0) {
	 sprintf(str,"%s.info",GetParamString(UsePhot));
	 finfo=fopen(str,"r");
	 if (!finfo) {
	    printf("****Error opening %s\n",str);
	    exit(-1);
	 }
      }
      else {
	 sprintf(str,"%s.info",GetParamString(outfn));
	 finfo=fopen(str,"r");
	 if (!finfo) {
	    printf("****Error opening %s\n",str);
	    exit(-1);
	 }
      }
   }
   else if (GetParamChar(UsePhot)!=0) {
      of=fopen(GetParamString(outfn),"w");
      if (!of) {
	 printf("****Error opening %s\n",GetParamString(outfn));
	 exit(-1);
      }
      sprintf(str,"%s.info",GetParamString(UsePhot));
      finfo=fopen(str,"r");
      if (!finfo) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      sprintf(str,"%s.warnings",GetParamString(outfn));
      fwarn=fopen(str,"w");
      if (!fwarn) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
   }
   else {
      of=fopen(GetParamString(outfn),"w");
      if (!of) {
	 printf("****Error opening %s\n",GetParamString(outfn));
	 exit(-1);
      }
      sprintf(str,"%s.info",GetParamString(outfn));
      finfo=fopen(str,"w");
      if (!finfo) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      sprintf(str,"%s.align",GetParamString(outfn));
      falign=fopen(str,"w");
      if (!falign) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      sprintf(str,"%s.apcor",GetParamString(outfn));
      fapcor=fopen(str,"w");
      if (!fapcor) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      sprintf(str,"%s.psfs",GetParamString(outfn));
      fpsfs=fopen(str,"w");
      if (!fpsfs) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      sprintf(str,"%s.warnings",GetParamString(outfn));
      fwarn=fopen(str,"w");
      if (!fwarn) {
	 printf("****Error opening %s\n",str);
	 exit(-1);
      }
      if (GetParamInt(VerboseData)>0) {
	 sprintf(str,"%s.data",GetParamString(outfn));
	 fverb=fopen(str,"w");
	 if (!fverb) {
	    printf("****Error opening %s\n",str);
	    exit(-1);
	 }
      }
   }
   if (GetGlobalInt(WARMSTART)==2) {
      if (GetParamChar(xytpsf)!=0 && !access(GetParamString(xytpsf),F_OK)) readfits(GetParamString(xytpsf),&refpsf,1);
      else {
	 refpsf.Next=-1;
	 printf("Cannot open template PSF file; ignoring\n");
      }
   }

   for (img=0;img<Timg;img++) {
      ftype tfits;
      imtype skyim;

      sprintf(str,"%s.sky.fits",base[img]);
      if (!access(str,F_OK)) {
	 fopenfirst(fsky+img,str,"rb",0);
	 fsky[img].f=readfitsh(str,&tfits,0);
	 freclose(fsky+img,1,fwarn);
	 memcpy(&skyim,&(tfits.img),sizeof(imtype));
	 if (tfits.img.Nmax) free(tfits.img.cards);
      }
      else {
	 fsky[img].lastoffset=-1;
	 skyim.Z=0;
      }
      sprintf(str,"%s.fits",base[img]);
      fopenfirst(fdata+img,str,"rb",0);
      if (img!=Nimg) printf("Image %d: ",img+1);
      else printf("Image 0: ");
      fdata[img].f=readfitsh(str,&tfits,1);
      freclose(fdata+img,1,fwarn);
      memcpy(dataim+img,&(tfits.img),sizeof(imtype));
      datahd[img].Nmax=datahd[img].Ncards=dataim[img].Ncards;
      datahd[img].cards=(cardtype*)calloc(datahd[img].Nmax,sizeof(cardtype));
      memcpy(datahd[img].cards,dataim[img].cards,sizeof(cardtype)*datahd[img].Ncards);
      read_cardvals(img);
      fflush(stdout);
      if (fsky[img].lastoffset>=0 && (skyim.X!=dataim[img].X || skyim.Y!=dataim[img].Y || skyim.Z!=dataim[img].Z)) {
	 printf("****Sky does not match the image\n");
	 exit(-1);
      }
      if (img==0) Next=tfits.Next;
      else if (Next!=tfits.Next) {
	 printf("****Number of extensions are not the same\n");
	 exit(-1);
      }
      if (GetParamInt(imgdata[img].RPSF)<=0) SetParam_NOT_THREADSAFE(imgdata[img].RPSF,GetParamInt(RPSF0));
      if (img<Nimg) {
	 if (GetParamChar(FakeStars)==0) {
	    tfits.img.Ncards=dataim[img].Ncards;
	    tfits.img.Nmax=dataim[img].Ncards+1;
	    tfits.img.cards=(cardtype*)calloc(tfits.img.Nmax,sizeof(cardtype));
	    memcpy(tfits.img.cards,dataim[img].cards,sizeof(cardtype)*tfits.img.Ncards);
	    sprintf(str,"%s.%d.res.fits",GetParamString(outfn),img+1);
	    if (isimage(&(tfits.img))) {
	       tfits.img.bscale=1.;
	       tfits.img.bzero=0.;
	       tfits.img.bits=-32;
	       if (iDMIN[img]<iDMIN0[img]) insertcards(&(tfits.img),-1.e30,-1.e30,-1.e30,iDMIN[img],-1.e30,-1.e30,-1.e30,-1.e30);
	    }
	    fopenfirst(fres+img,str,"ab",0);
	    fres[img].f=writefitsh(str,&tfits,0);
	    freclose(fres+img,0,NULL);
	 }
	 if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	    if (isimage(&(tfits.img))) {
	       tfits.img.X=2*GetParamInt(imgdata[img].RPSF)+1;
	       tfits.img.Y=2*GetParamInt(imgdata[img].RPSF)+1;
	    }
	    sprintf(str,"%s.%d.psf.fits",GetParamString(outfn),img+1);
	    fopenfirst(fpsf+img,str,"ab",0);
	    fpsf[img].f=writefitsh(str,&tfits,0);
	    freclose(fpsf+img,0,NULL);
	    free(tfits.img.cards);
	 }
	 else {
	    if (GetParamChar(UsePhot)!=0) sprintf(str,"%s.%d.psf.fits",GetParamString(UsePhot),img+1);
	    else sprintf(str,"%s.%d.psf.fits",GetParamString(outfn),img+1);
	    fopenfirst(fpsf+img,str,"rb",0);
	    fpsf[img].f=readfitsh(str,&tfits,0);
	    if (isimage(&(tfits.img))) {
	       if (tfits.img.X!=2*GetParamInt(imgdata[img].RPSF)+1 || tfits.img.Y!=2*GetParamInt(imgdata[img].RPSF)+1) {
		  printf("Mismatch in RPSF\n");
		  exit(-1);
	       }
	       if (tfits.img.Nmax) free(tfits.img.cards);
	       tfits.img.Nmax=0;
	    }
	    else {
	       readimage(fpsf[img].f,&(tfits.img));
	       freeimg(tfits.img.data,tfits.img.X,tfits.img.Y,tfits.img.Z);
	    }
	    freclose(fpsf+img,1,fwarn);
	 }
      }
   }
   if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) outptinfo();
#ifdef PGPLOT
   allocDiagData1(Next);
#endif

   // set up circles and photometry weights, etc.
   procframe(0);
   for (img=0;img<Timg;img++) if (dataim[img].Nmax) free(dataim[img].cards);
   for (ext=0;ext<Next;ext++) {
      for (img=0;img<Timg;img++) {
	 ftype tfits;
	 imtype skyim;

	 if (fsky[img].lastoffset>=0) {
	    fopenagain(fsky+img);
	    readexth(fsky[img].f,&skyim,0);
	    freclose(fsky+img,1,fwarn);
	    if (skyim.Nmax) free(skyim.cards);
	 }
	 fopenagain(fdata+img);
	 readexth(fdata[img].f,&(tfits.img),1);
	 freclose(fdata+img,1,fwarn);
	 memcpy(dataim+img,&(tfits.img),sizeof(imtype));
	 read_cardvals(img);
	 if (fsky[img].lastoffset>=0 && (skyim.X!=dataim[img].X || skyim.Y!=dataim[img].Y || skyim.Z!=dataim[img].Z)) {
	    printf("****Sky does not match the image\n");
	    exit(-1);
	 }
	 if (img!=0 && dataim[0].Z!=dataim[img].Z) {
	    printf("****Number of chips are not the same size\n");
	    exit(-1);
	 }
	 if (img<Nimg) {
	    if (GetParamChar(FakeStars)==0) {
	       tfits.img.Ncards=dataim[img].Ncards;
	       tfits.img.Nmax=dataim[img].Ncards+1;
	       tfits.img.cards=(cardtype*)calloc(tfits.img.Nmax,sizeof(cardtype));
	       memcpy(tfits.img.cards,dataim[img].cards,sizeof(cardtype)*tfits.img.Ncards);
	       if (isimage(&(tfits.img))) {
		  tfits.img.bscale=1.;
		  tfits.img.bzero=0.;
		  tfits.img.bits=-32;
		  if (iDMIN[img]<iDMIN0[img]) insertcards(&(tfits.img),-1.e30,-1.e30,-1.e30,iDMIN[img],-1.e30,-1.e30,-1.e30,-1.e30);
	       }
	       fopenagain(fres+img);
	       writeexth(fres[img].f,&(tfits.img),0);
	       freclose(fres+img,0,NULL);
	    }
	    if (GetParamChar(FakeStars)==0 && GetParamChar(UsePhot)==0) {
	       if (isimage(&(tfits.img))) {
		  tfits.img.X=2*GetParamInt(imgdata[img].RPSF)+1;
		  tfits.img.Y=2*GetParamInt(imgdata[img].RPSF)+1;
	       }
	       fopenagain(fpsf+img);
	       writeexth(fpsf[img].f,&(tfits.img),0);
	       freclose(fpsf+img,0,NULL);
	       free(tfits.img.cards);
	    }
	    else {
	       fopenagain(fpsf+img);
	       readexth(fpsf[img].f,&(tfits.img),0);
	       if (isimage(&(tfits.img))) {
		  if (tfits.img.X!=2*GetParamInt(imgdata[img].RPSF)+1 || tfits.img.Y!=2*GetParamInt(imgdata[img].RPSF)+1) {
		     printf("Mismatch in RPSF\n");
		     exit(-1);
		  }
	       }
	       else {
		  readimage(fpsf[img].f,&(tfits.img));
		  freeimg(tfits.img.data,tfits.img.X,tfits.img.Y,tfits.img.Z);
	       }
	       freclose(fpsf+img,1,fwarn);
	    }
	 }
      }
      procframe(ext+1);
      for (img=0;img<Timg;img++) if (dataim[img].Nmax) free(dataim[img].cards);
   }
   fclose(finfo);
   if (GetParamChar(FakeStars)!=0) fclose(ffakeout);
   else {
      fclose(of);
      fclose(fwarn);
      if (GetParamChar(UsePhot)==0) {
	 if (GetParamInt(VerboseData)>0) fclose(fverb);
	 fclose(falign);
	 fclose(fapcor);
	 fclose(fpsfs);
      }
   }
#ifdef PGPLOT
   plotDiagData();
#endif
   //endtictoc();
   return 0;
}
