/*
  To-do
  - DQ data seems to mask out all bright stars (DQ = 1-do not use + 16-outlier)
  - Data quality needs to be read/stored as int32 rather than float so that DQ flags of 2^24 and higher can be processed
  - Confirm DQ interpretations
  - Does DMAX calculation include saturated pixels, and does it matter?
 */


#include <fits.h>
#include "miripsfdata.h"

float RDNOISE_CAL = 24.; // (6 DN) * (4 e-/ADU) to convert to electrons
float GAIN_CAL = 4.;
double RN_SCALE = 1.0;

ftype fits;
double GAIN,RN,EXP,EXP0,EPOCH,CPS_TO_JY;
float DMIN,DMAX;
int SCI=-1,DQ=-1,WHT=-1,AREA=-1,VAR_RNOISE=-1,VAR_POISS=-1;

// optional parameters
int NCOMBINE=1;
int MASK_LYOT=0;
int ETCTIME=1;
int ESTNOISE=0;

int MIRItype(ftype *f) {
   int i;
   char extname[21];

   if (fits.Next==0) {printf("**No extension\n"); exit(-1);}
   if (strcmp(getcardval(&fits.img,"TELESCOP",1),"JWST") || strcmp(getcardval(&fits.img,"INSTRUME",1),"MIRI") || strcmp(getcardval(&fits.img,"DETECTOR",1),"MIRIMAGE")) {printf("**Not a JWST MIRI image\n"); exit(-1);}
   // identify extensions
   SCI=-1;
   DQ=-1;
   WHT=-1;
   AREA=-1;
   VAR_RNOISE=-1;
   VAR_POISS=-1;
   for (i=0;i<fits.Next;i++) {
      strcpy(extname,getcardval(fits.ext+i,"EXTNAME",1));
      //printf("%d/%d; %s\n",i,fits.Next,extname);
      if (!strcmp(extname,"SCI")) SCI=i;
      else if (!strcmp(extname,"DQ")) DQ=i;
      else if (!strcmp(extname,"WHT")) WHT=i;
      else if (!strcmp(extname,"AREA")) AREA=i;
      else if (!strcmp(extname,"VAR_RNOISE")) VAR_RNOISE=i;
      else if (!strcmp(extname,"VAR_POISSON")) VAR_POISS=i;
   }
   if (SCI==-1) {printf("**No SCI extension found\n"); exit(-1);}
   if (SCI!=0) {printf("**SCI not first extension\n"); exit(-1);}
   if (WHT==-1 && DQ==-1) {printf("**Neither DQ nor WHT extension found\n"); exit(-1);}
   if (WHT!=-1 && DQ!=-1) {printf("**Both DQ and WHT extension found\n"); exit(-1);}
   if (fits.ext[SCI].Z!=1) {printf("**SCI image is not single image\n"); exit(-1);}

   // Process drizzle
   if (WHT!=-1) {
      if (fits.ext[WHT].X!=fits.ext[SCI].X || fits.ext[WHT].Y!=fits.ext[SCI].Y || fits.ext[WHT].Z!=1) {printf("**WHT image does not match SCI size\n"); exit(-1);}
      return -1;
   }

   // Process non-drizzle
   if (DQ==-1) {printf("**No DQ extension found\n"); exit(-1);}
   if (AREA==-1) {printf("**No AREA extension found\n"); exit(-1);}
   if (fits.ext[SCI].X!=1032 || fits.ext[SCI].Y!=1024) {printf("**SCI image is not 1032x1024x1\n"); exit(-1);}
   if (fits.ext[DQ].X!=fits.ext[SCI].X || fits.ext[DQ].Y!=fits.ext[SCI].Y || fits.ext[DQ].Z!=1) {printf("**DQ image does not match SCI size\n"); exit(-1);}
   if (fits.ext[AREA].X!=fits.ext[SCI].X || fits.ext[AREA].Y!=fits.ext[SCI].Y || fits.ext[AREA].Z!=1) {printf("**AREA image does not match SCI size\n"); exit(-1);}
   return 0;
}

double getEitherValue(char*str) {
   char *ptr = getcardval(&(fits.img),str,0);
   //if (*ptr==0) ptr = getcardval(fits.ext,str,1);
   if (*ptr==0) exit(-1);
   return atof(ptr);
}

void MIRIexptime(void) {
   { // DURATION calculation
      double TFRAME = getEitherValue("TFRAME");
      double NGROUPS = getEitherValue("NGROUPS");
      double NFRAMES = getEitherValue("NFRAMES");
      double GROUPGAP = getEitherValue("GROUPGAP");
      double DRPFRMS1 = getEitherValue("DRPFRMS1");
      double NINTS = getEitherValue("NINTS");
      double DRPFRMS3 = getEitherValue("DRPFRMS3");
      double NRSTSTRT = getEitherValue("NRSTSTRT");
      double NRESETS = getEitherValue("NRESETS");
      double DURATION = TFRAME*((NGROUPS*NFRAMES+(NGROUPS-1)*GROUPGAP+DRPFRMS1)*NINTS) + TFRAME*(DRPFRMS3*NINTS+NRSTSTRT+NRESETS*(NINTS-1));
      double DUR_HDR = getEitherValue("DURATION");
      if (fabs(DUR_HDR-DURATION)>=0.01) printf("**Warning: DURATION = %f in header, calculated value = %f\n",DUR_HDR,DURATION);
   }
   { // EFFEXPTM calculation
      double TFRAME = getEitherValue("TFRAME");
      double NGROUPS = getEitherValue("NGROUPS");
      double NFRAMES = getEitherValue("NFRAMES");
      double GROUPGAP = getEitherValue("GROUPGAP");
      double DRPFRMS1 = getEitherValue("DRPFRMS1");
      double NINTS = getEitherValue("NINTS");
      double NRESETS = getEitherValue("NRESETS");
      double EFFEXPTM = TFRAME*(NGROUPS*NFRAMES+(NGROUPS-1)*GROUPGAP+DRPFRMS1)*NINTS;
      EFFEXPTM += TFRAME*NRESETS*(NINTS-1); // fudge factor to match data from GO-02128
      if (fabs(EXP-EFFEXPTM)>=0.01) printf("**Warning: EFFEXPTM = %f in header, calculated value = %f\n",EXP,EFFEXPTM);
   }
   { // time from start to end calculation
      double TGROUP = getEitherValue("TGROUP");
      double NGROUPS = getEitherValue("NGROUPS");
      double NINTS = getEitherValue("NINTS");
      double NGROUPSadj = NGROUPS-1;
      if (NGROUPS>3) NGROUPSadj = NGROUPS-3;
      else if (NGROUPS==3) NGROUPSadj = NGROUPS-2;
      double START2END = TGROUP*NGROUPSadj*NINTS;
      if (!ETCTIME) printf("Note: calculated Start2End exposure time is %f\n",START2END);
      else {
	 printf("Setting DOLPHOT exposure time to %f sec\n",START2END);
	 EXP = START2END;
      }
      double NSAMPLES = getEitherValue("NSAMPLES");
      RN_SCALE = sqrt( (12.*NGROUPSadj*NINTS)/(NSAMPLES*(NGROUPSadj+1)*(NGROUPSadj+2)) );
      printf("Scaling readout noise by %f\n",RN_SCALE);
   }
}

// Divide RDNOISE by sqrt(2) - appears it was determined by two-readout comparison like other JWST cameras
void MIRIgetcards(int cm) {
   EXP = getEitherValue("EFFEXPTM");
   RN_SCALE = 1.0;
   if (cm==0) MIRIexptime();
   RN = RDNOISE_CAL/sqrt(2.)*RN_SCALE;
   GAIN = GAIN_CAL;
   EPOCH = getEitherValue("EXPMID");
   EXP0=EXP/NCOMBINE;
   return;
}

void MIRIgetMaxMin(void) {
   int x,y,allnan=1;
   DMAX = 0;
   DMIN = 0;
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) if (!isnan(fits.ext[SCI].data[0][y][x])) {
      if (allnan==1) {
	 DMIN = fits.ext[SCI].data[0][y][x];
	 DMAX = fits.ext[SCI].data[0][y][x];
	 allnan = 0;
      }
      else {
	 if (fits.ext[SCI].data[0][y][x]<DMIN) DMIN=fits.ext[SCI].data[0][y][x];
	 if (fits.ext[SCI].data[0][y][x]>DMAX) DMAX=fits.ext[SCI].data[0][y][x];
      }
   }
   if (allnan==0) {
      if (DMAX>10) DMAX*=1.1;
      else DMAX++;
      if (DMIN<-10) DMIN*=1.1;
      else DMIN--;
   }
   /*
   int x,y;
   DMIN = fits.ext[SCI].data[0][0][0];
   DMAX = fits.ext[SCI].data[0][0][0];
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) {
      if (fits.ext[SCI].data[0][y][x]<DMIN) DMIN=fits.ext[SCI].data[0][y][x];
      if (fits.ext[SCI].data[0][y][x]>DMAX) DMAX=fits.ext[SCI].data[0][y][x];
   }
   if (DMAX>10) DMAX*=1.1;
   else DMAX++;
   if (DMIN<-10) DMIN*=1.1;
   else DMIN--;
   */
}

/*
  Not in use
     0 1: DO_NOT_USE - Bad pixel. Do not use
     1 2: SATURATED - Pixel saturated during exposure
     2 4: JUMP_DET - Jump detected during exposure
     3 8: DROPOUT - Data lost in transmission
     4 16: OUTLIER - Flagged by outlier detection
     5 32: PERSISTENCE - High persistence
     6 64:  AD_FLOOR - Below A/D floor
     7 128: RESERVED
     8 256: UNRELIABLE_ERROR - Uncertainty exceeds quoted error
     9 512: NON_SCIENCE - Pixel not on science portion of detector
    10 1024: DEAD - Dead pixel
    11 2048: HOT - Hot pixel
    12 4096: WARM - Warm pixel
    13 8192: LOW_QE - Low quantum efficiency
    14 16384: RC - RC pixel
    15 32768: TELEGRAPH - Telegraph pixel
    16 65536: NONLINEAR - Pixel highly nonlinear
    17 131072: BAD_REF_PIXEL - Reference pixel cannot be used
    18 262144: NO_FLAT_FIELD - Flat field cannot be measured
    19 524288: NO_GAIN_VALUE - Gain cannot be measured
    20 1048576: NO_LIN_CORR - Linearity correction not available
    21 2097152: NO_SAT_CHECK - Saturation check not available
    22 4194304: UNRELIABLE_BIAS - Bias variance large
    23 8388608: UNRELIABLE_DARK - Dark variance large
    24 16777216: UNRELIABLE_SLOPE - Slope variance large (i.e., noisy pixel)
    25 33554432: UNRELIABLE_FLAT - Flat variance large
    26 67108864: OPEN - Open pixel (counts move to adjacent pixels)
    27 134217728: ADJ_OPEN - Adjancent to open pixel
    28 268435456: UNRELIABLE_RESET - Sensitive to reset anomaly
    29 536870912: MSA_FAILED_OPEN - Pixel sees light from failed-open shutter
    30 1073741824: OTHER_BAD_PIXEL - A catch-all flag
    31 2147483648: REFERENCE_PIXEL - Pixel is a reference pixel
 */
void MIRI_DQ_mask(void) {
   int x,y,dq;
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) {
      dq=(int)(fits.ext[DQ].data[0][y][x]+0.5);
      // AEDDEBUG - delete this line once fits.h can read int instead of just float - note that 2^31 is the reference pixels around the edge of the frame so that will need to be masked regardless
      if (dq>=8388608 || dq<=-8388608) fits.ext[SCI].data[0][y][x]=safedown(DMIN);
      else if (dq&262656) fits.ext[SCI].data[0][y][x]=safedown(DMIN); // used for masked-out pixels (262144) and coronagraphs (512)
      //else if (dq&1 && MASK_BAD==1) fits.ext[SCI].data[0][y][x]=safedown(DMIN); // used for "bad do-not-use" pixels
      else if (MASK_LYOT==1 && x<=320 && y>=700) fits.ext[SCI].data[0][y][x]=safedown(DMIN); // mask out coronagraph area
      else if (dq&2) fits.ext[SCI].data[0][y][x]=safeup(DMAX); // 2 = saturated
      //else if (fits.ext[SCI].data[0][y][x]==0.0) fits.ext[SCI].data[0][y][x]=safedown(DMIN); // AEDDEBUG gets remaining bad pixels (odds of exactly 0.0 is very small)
      else if (fits.ext[SCI].data[0][y][x]==0.0 || isnan(fits.ext[SCI].data[0][y][x])) fits.ext[SCI].data[0][y][x]=safedown(DMIN); // AEDDEBUG gets remaining bad pixels (odds of exactly 0.0 is very small, newer pipeline uses NaN instead)
	 /*
      // echo "" | awk '{printf "%d\n",2^1}'
      if (dq&2) fits.ext[SCI].data[0][y][x]=safeup(DMAX); // 2
      //else if ((dq&512) && USE_FLAT_FLAG) fits.ext[SCI].data[0][y][x]=safedown(DMIN);
      // echo "" | awk '{printf "%d\n",2^0+2^3+2^6+2^9+2^10+2^11+2^16+2^17+2^18+2^26+2^27}'
      else if (dq&201789001) fits.ext[SCI].data[0][y][x]=safedown(DMIN); // 1+8+64+512+1024+2048+65536+131072+262144+67108864+134217728
      // echo "" | awk '{printf "%d\n",2^4}'
      else if ((dq&16) && MASKCR) fits.ext[SCI].data[0][y][x]=safedown(DMIN); //
	 */
   }
   return;
}

void MIRI_WHT_mask(void) {
   int x,y;
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) {
      if (fits.ext[WHT].data[0][y][x]==0) fits.ext[SCI].data[0][y][x]=safedown(DMIN);
   }
   return;
}

void MIRI_PAM_mult(void) {
   int x,y;
   float DMIN1,DMAX1;

   if (DMAX>0.) DMAX1=1.5*DMAX;
   else DMAX1=0.;
   if (DMIN<0.) DMIN1=1.5*DMIN;
   else DMIN1=0.;
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) if (fits.ext[SCI].data[0][y][x]>DMIN && fits.ext[SCI].data[0][y][x]<DMAX) {
      fits.ext[SCI].data[0][y][x] *= fits.ext[AREA].data[0][y][x];
      if (VAR_RNOISE>=0) fits.ext[VAR_RNOISE].data[0][y][x] *= fits.ext[AREA].data[0][y][x]*fits.ext[AREA].data[0][y][x];
      if (VAR_POISS>=0) fits.ext[VAR_POISS].data[0][y][x] *= fits.ext[AREA].data[0][y][x]*fits.ext[AREA].data[0][y][x];
   }
   else if (fits.ext[SCI].data[0][y][x]>=DMAX) fits.ext[SCI].data[0][y][x]=safeup(DMAX1);
   else fits.ext[SCI].data[0][y][x]=safedown(DMIN1);
   DMIN=DMIN1;
   DMAX=DMAX1;
   return;
}

void MIRI_exptime_mult(void) {
   int x,y;
   float scale;

   // initialize scale to convert to counts
   if (!strcmp(getcardval(fits.ext+SCI,"BUNIT",1),"MJy/sr")) {
      scale = 1 / atof(getcardval(fits.ext+SCI,"PHOTMJSR",1));
      CPS_TO_JY = atof(getcardval(fits.ext+SCI,"PHOTMJSR",1)) // CPS->MJy/sr
	 * atof(getcardval(fits.ext+SCI,"PIXAR_SR",1)) // sr/pix
	 * (1.0e6); // Jy/MJy
   }
   /*
   else if (!strcmp(getcardval(fits.ext[SCI],"BUNIT",1),"----")) {
      scale = 1 / atof(getcardval(fits.ext[SCI],"PHOTUJA2",1));
      CPS_TO_JY = atof(getcardval(fits.ext+SCI,"PHOTUJA2",1)) // CPS->uJy/asec^2
	 * atof(getcardval(fits.ext+SCI,"PIXAR_A2",1)) // asec^2/pix
	 / (1.0e6); // Jy/uJy
   }
   */
   else {
      printf("Image units (%s) unknown - should be MJy/sr\n",getcardval(fits.ext+SCI,"BUNIT",1));
      exit(-1);
   }
   scale *= EXP;

   // convert DMIN, DMAX, and data
   DMIN *= scale;
   DMAX *= scale;
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) {
      fits.ext[SCI].data[0][y][x] *= scale;
      if (VAR_RNOISE>=0) fits.ext[VAR_RNOISE].data[0][y][x] *= scale*scale;
      if (VAR_POISS>=0) fits.ext[VAR_POISS].data[0][y][x] *= scale*scale;
   }
   return;
}

void MIRI_estRN(void) {
   int I,x,y;
   double X,XX,RNest=0,RNsig=1.e99;
   if (VAR_RNOISE==-1) {
      printf("Warning: readout noise extension not found\n");
      return;
   }
   for (int iter=0;iter<5;iter++) {
      I=0; X=XX=0.;
      for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) if (fits.ext[SCI].data[0][y][x]>DMIN && fits.ext[SCI].data[0][y][x]<DMAX) {
	 if ((WHT>=0 && fits.ext[WHT].data[0][y][x]!=0) || (DQ>=0 && (int)(fits.ext[DQ].data[0][y][x]+0.5)==0)) {
	    if (fabs(fits.ext[VAR_RNOISE].data[0][y][x]-RNest)<=3.0*RNsig) {
	       I++;
	       X+=fits.ext[VAR_RNOISE].data[0][y][x];
	       XX+=fits.ext[VAR_RNOISE].data[0][y][x]*fits.ext[VAR_RNOISE].data[0][y][x];
	    }
	 }
      }
      RNest = X/I;
      RNsig = sqrt((XX-X*X/I)/(I-1.));
   }
   RNest = sqrt(RNest)*GAIN;
   if (ESTNOISE) {
      printf("Updating readout noise from %f to %f based on VAR_RNOISE\n",RN,RNest);
      RN = RNest;
   }
   else printf("Note: estimated readout noise is %f from VAR_RNOISE vs. %f\n",RNest,RN);
}

void MIRI_estBG(float BG) {
   int x,y;
   double I,X,XX,BGest=0,BGsig=1.e99;
   if (VAR_POISS==-1) {
      printf("Warning: readout noise extension not found\n");
      return;
   }
   for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) fits.ext[VAR_POISS].data[0][y][x] -= fits.ext[SCI].data[0][y][x]/GAIN;
   for (int iter=0;iter<5;iter++) {
      I=X=XX=0.;
      for (y=0;y<fits.ext[SCI].Y;y++) for (x=0;x<fits.ext[SCI].X;x++) if (fits.ext[SCI].data[0][y][x]>DMIN && fits.ext[SCI].data[0][y][x]<DMAX) {
	 if ((WHT>=0 && fits.ext[WHT].data[0][y][x]!=0) || (DQ>=0 && (int)(fits.ext[DQ].data[0][y][x]+0.5)==0)) {
	    double ct = fits.ext[SCI].data[0][y][x];
	    if (ct<0.) ct=0.;
	    if (fabs(fits.ext[VAR_POISS].data[0][y][x]-BGest)<=3.0*BGsig) {
	       // model uncertainty in SCI as RSS of RDNOISE and sqrt(SCI/GAIN)
	       double isigy2 = 1./( 1. + ct/GAIN/(RN*RN) );
	       I += isigy2;
	       X += fits.ext[VAR_POISS].data[0][y][x]*isigy2;
	       XX += fits.ext[VAR_POISS].data[0][y][x]*fits.ext[VAR_POISS].data[0][y][x]*isigy2;
	    }
	 }
      }
      if (X<0) {
	 BGest = 0.;
	 BGsig = sqrt(XX/I);
      }
      else {
	 BGest = X/I;
	 BGsig = sqrt((XX-X*X/I)/(I-1.));
      }
      //printf("   %f %f\n",BGest,BGsig);
   }
   BGest = sqrt(BGest);
   if (BG<0) {
      BG = BGest;
      printf("Setting background noise to %f DN based on VAR_POISSON\n",BG);
   }
   else if (BG>0) printf("Note: estimated background noise is %f from VAR_POISSON vs. %f\n",BGest,BG);
   else if (fabs(BGest)>10.) printf("**Warning: estimated background noise of %f ignored\n",BGest);
   else printf("Note: estimated background is %f from VAR_POISSON\n",BGest);
   if (BG>0.) {
      printf("Updating readout noise from %f to %f to account for background\n",RN,sqrt(RN*RN+BG*BG));
      RN = sqrt(RN*RN+BG*BG);
   }
}

void MIRIsetcards(void) {
   insertcards(fits.ext+SCI,GAIN,RN*sqrt(NCOMBINE),EXP,DMIN,DMAX,EPOCH,0.0,EXP0);
   return;
}

int main(int argc,char**argv) {
   int i,j,tp;
   char str[101];
   float BG=0.;

   if (argc<2) {
      printf("Usage: %s <<flags>> <fits files>\n",*argv);
      printf("  -estnoise   to estimate readout noise from FITS VAR_RNOISE extension\n");
      printf("  -bg=#       to increase noise to account for # counts of backgrond.\n");
      printf("              Note: negative BG value will use estimated background\n");
      printf("  -mask_lyot  will mask out the Lyot coronagraph region\n");
      printf("  -noetctime  to use FITS EFFEXPTM keyword for exposure time\n");
      return 1;
   }
   for (i=1;i<argc;i++) {
      if (!strcasecmp(argv[i],"-mask_lyot")) MASK_LYOT=1;
      else if (!strcasecmp(argv[i],"-noetctime")) ETCTIME=0;
      else if (!strcasecmp(argv[i],"-estnoise")) ESTNOISE=1;
      else if (!strncasecmp(argv[i],"-bg=",4)) BG = atof(argv[i]+4);
      else {
	 readfits(argv[i],&fits,1);
	 tp=MIRItype(&fits);
	 if (fits.Next>0 && strcmp(getcardval(fits.ext,"DOL_MIRI",0),"")) printf("%s already run through mirimask\n",argv[i]);
	 else if (tp<-1 || tp>0) printf("%s is not a MIRI fits file\n",argv[i]);
	 else {
	    MIRIgetcards(tp);
	    MIRIgetMaxMin();
	    if (tp<0) {
	       MIRI_WHT_mask();
	    }
	    else {
	       MIRI_DQ_mask();
	       MIRI_PAM_mult();
	    }
	    MIRI_exptime_mult();
	    MIRI_estRN();
	    MIRI_estBG(BG);
	    MIRIsetcards();
	    sprintf(str,"DOL_MIRI=                   %2d / DOLPHOT MIRI tag                             ",tp);
	    addcard(fits.ext+SCI,str);
	    sprintf(str,"DOL_C2JY= %20f / DOLPHOT MIRI DN/s to Jy conversion             ",-2.5*log10(CPS_TO_JY));
	    addcard(fits.ext+SCI,str);
	    for (j=1;j<fits.Next;j++) freeim(fits.ext+j);
	    fits.Next=1;
	    writefits(argv[i],&fits,1);
	 }
	 freefits(&fits);
      }
   }
   return 0;
}
